package ru.yandex.solomon.dumper;

import java.util.HashMap;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonShardMetricsSupplier implements MetricSupplier {
    private final DumperLocalShards localShards;
    private volatile int metricsCount = 0;

    public SolomonShardMetricsSupplier(DumperLocalShards localShards) {
        this.localShards = localShards;
    }

    @Override
    public int estimateCount() {
        return metricsCount;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        // By host metrics will be dropped on monitoring side.
        // Use constant host to reduce amount on mem-only metrics on monitoring side
        var labels = commonLabels.toBuilder().add("host", "cluster");
        var it = localShards.iterator();
        var totalByProjectId = new HashMap<String, SolomonShardMetrics>();
        int metricsCount = 0;
        while (it.hasNext()) {
            var dumperShard = it.next();
            var processByNumId = dumperShard.processes();
            for (var solomonShardProcess : processByNumId.values()) {
                var opts = solomonShardProcess.getOpts();
                var metrics = solomonShardProcess.getMetrics();

                // projectId={{projectId}}, shardId={{shardId}}
                metrics.append(tsMillis,
                        labels.add("projectId", opts.projectId)
                                .add("shardId", opts.id)
                                .build(),
                        consumer);
                metricsCount += metrics.estimateCount();

                // projectId={{projectId}}, shardId=total aggregation
                var projectMetrics = totalByProjectId.computeIfAbsent(opts.projectId, ignore -> new SolomonShardMetrics());
                projectMetrics.combine(metrics);
            }
        }

        var total = new SolomonShardMetrics();

        // projectId={{projectId}}, shardId=total
        labels.add("shardId", "total");
        for (var entry : totalByProjectId.entrySet()) {
            var projectId = entry.getKey();
            var metrics = entry.getValue();

            metrics.append(tsMillis, labels.add("projectId", projectId).build(), consumer);
            metricsCount += metrics.estimateCount();
            // projectId=total, shardId=total aggregation
            total.combine(metrics);
        }

        // projectId=total, shardId=total
        total.append(tsMillis, labels.add("projectId", "total").build(), consumer);
        metricsCount += total.estimateCount();
        this.metricsCount = metricsCount;
    }
}
