package ru.yandex.solomon.dumper;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nullable;

import com.google.common.hash.Hashing;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonShardOptsProviderImpl implements SolomonShardOptsProvider {
    private volatile Snapshot snapshot = new Snapshot(Int2ObjectMaps.emptyMap(), 0);

    private final ShardsDao shardsDao;
    private final PingActorRunner actor;

    public SolomonShardOptsProviderImpl(ShardsDao shardsDao, ScheduledExecutorService timer, ExecutorService executor) {
        this.shardsDao = shardsDao;
        this.actor = PingActorRunner.newBuilder()
                .operation("reload shards")
                .pingInterval(Duration.ofMinutes(1))
                .backoffDelay(Duration.ofSeconds(10))
                .timer(timer)
                .executor(executor)
                .onPing(this::reload)
                .build();
        this.actor.forcePing();
    }

    private CompletableFuture<Void> reload(int attempt) {
        return shardsDao.findAll()
                .thenAccept(this::updateSnapshot);
    }

    void updateSnapshot(List<Shard> shards) {
        Int2ObjectMap<SolomonShardOpts> options = new Int2ObjectOpenHashMap<>(shards.size());
        var hasher = Hashing.murmur3_128().newHasher();
        for (var shard : shards) {
            var opts = SolomonShardOpts.newBuilder()
                    .withNumId(shard.getNumId())
                    .withId(shard.getId())
                    .withProjectId(shard.getProjectId())
                    .withDecimPolicy(shard.getShardSettings().getRetentionPolicy() == DecimPolicy.UNDEFINED
                            ? DecimPolicy.DEFAULT.toProto()
                            : shard.getShardSettings().getRetentionPolicy().toProto())
                    .build();
            hasher.putInt(opts.hashCode());
            options.put(shard.getNumId(), opts);
        }
        snapshot = new Snapshot(options, hasher.hash().asLong());
    }

    @Override
    public boolean isInitialized() {
        return snapshot.isInitialized();
    }

    @Nullable
    @Override
    public SolomonShardOpts resolve(int numId) {
        return snapshot.resolve(numId);
    }

    @Override
    public SolomonShardOptsProvider snapshot() {
        return snapshot;
    }

    @Override
    public long optionsHash() {
        return snapshot.optionsHash();
    }

    private record Snapshot(Int2ObjectMap<SolomonShardOpts> options, long hash) implements SolomonShardOptsProvider {

        @Override
        public boolean isInitialized() {
            return !options.isEmpty();
        }

        @Nullable
        @Override
        public SolomonShardOpts resolve(int numId) {
            return options.get(numId);
        }

        @Override
        public SolomonShardOptsProvider snapshot() {
            return this;
        }

        @Override
        public long optionsHash() {
            return hash;
        }
    }
}
