package ru.yandex.solomon.dumper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import javax.annotation.WillClose;
import javax.annotation.WillNotClose;

import com.google.common.base.Throwables;
import io.netty.buffer.ByteBufAllocator;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.dumper.storage.shortterm.DumperTx;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.memory.layout.MemMeasurableSubsystem;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.selfmon.executors.CpuMeasureExecutor;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.Logs;
import ru.yandex.solomon.slog.ResolvedLogMetricsIteratorImpl;
import ru.yandex.solomon.slog.ResolvedLogMetricsRecord;
import ru.yandex.solomon.slog.UnresolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaRecord;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.solomon.util.time.DurationUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.util.time.DurationUtils.backoff;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonShardProcess implements MemMeasurableSubsystem, AutoCloseable {
    private static final long RESOLVE_SLEEP_MIN_MILLIS = 500;
    private static final long RESOLVE_SLEEP_MAX_MILLIS = 30_000;
    private static final long MAX_MESSAGE_SIZE = 50 << 20; // 50 MiB per resolve request

    private static final Logger logger = LoggerFactory.getLogger(SolomonShardProcess.class);
    private final int numId;
    private final LongTermStorage storage;
    private final SolomonShardProcessMetrics aggrMetrics;
    private final SolomonShardMetrics metrics;
    private final long maxSizeToResolve;
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private final InterningLabelAllocator labelAllocator;
    private final ByteBufAllocator bufferAllocator;
    private final ConcurrentMetricsCache cache;
    private final Deque<ResolveRequest> queue;
    private final AtomicLong bytesInFlight = new AtomicLong();
    private final ActorWithFutureRunner actor;
    // Field always updates and read inside actor
    private SolomonShardOpts opts;
    private volatile boolean closed;

    public SolomonShardProcess(
            SolomonShardOpts opts,
            Executor executor,
            ScheduledExecutorService timer,
            LongTermStorage storage,
            SolomonShardProcessMetrics metrics)
    {
        this(opts, executor, timer, storage, metrics, MAX_MESSAGE_SIZE);
    }

    public SolomonShardProcess(
            SolomonShardOpts opts,
            Executor executor,
            ScheduledExecutorService timer,
            LongTermStorage storage,
            SolomonShardProcessMetrics metrics,
            long maxSizeToResolve)
    {
        this.numId = opts.numId;
        this.opts = opts;
        this.storage = storage;
        this.aggrMetrics = metrics;
        this.metrics = new SolomonShardMetrics();
        this.maxSizeToResolve = maxSizeToResolve;
        this.executor = new CpuMeasureExecutor(this.metrics.cpuTimeNanos, executor);
        this.timer = timer;
        this.labelAllocator = new InterningLabelAllocator();
        this.bufferAllocator = ByteBufAllocator.DEFAULT;
        this.cache = new ConcurrentMetricsCache(this.executor, timer, metrics.cacheMetrics);
        this.queue = new ConcurrentLinkedDeque<>();
        this.actor = new ActorWithFutureRunner(this::act, this.executor);
    }

    public boolean updateOpts(SolomonShardOpts opts) {
        if (this.opts.equals(opts)) {
            return false;
        }

        this.opts = opts;
        return true;
    }

    public SolomonShardOpts getOpts() {
        return this.opts;
    }

    public SolomonShardMetrics getMetrics() {
        return metrics;
    }

    public CompletableFuture<Int2ObjectMap<Log>> enqueue(DumperTx tx, Log log) {
        if (storage.isAbleToCreateNewMetrics(numId)) {
            cache.resetSettlers();
        }

        bytesInFlight.addAndGet(log.memorySizeIncludingSelf());
        var parsing = new ParsingTask(tx, opts.decimPolicy, log, cache, Labels.allocator, bufferAllocator);
        return asyncParse(parsing)
            .thenCompose(result -> {
                if (result.unresolved == null) {
                    return completedFuture(result.resolved);
                }

                var resolved = result.resolved;
                var unresolved = result.unresolved;
                bytesInFlight.addAndGet(unresolved.memorySizeIncludingSelf());
                return resolveAndParse(tx, unresolved)
                    .handle((resolvedTwo, e) -> {
                        if (e != null) {
                            resolved.values().forEach(Log::close);
                            Throwables.throwIfUnchecked(e);
                            throw new RuntimeException(e);
                        }

                        return combine(resolved, resolvedTwo);
                    });
            });
    }

    private CompletableFuture<ParsingResult> asyncParse(ParsingTask task) {
        return CompletableFuture.supplyAsync(() -> parse(task), executor);
    }

    private ParsingResult parse(ParsingTask task) {
        var bytes = task.getLog().memorySizeIncludingSelf();
        try {
            var result = task.run();
            aggrMetrics.successParse(result.stats);
            metrics.successParse(result.stats);
            return result;
        } catch (Throwable e) {
            aggrMetrics.failedParse();
            metrics.failedParse();
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        } finally {
            bytesInFlight.addAndGet(-bytes);
        }
    }

    private CompletableFuture<?> act() {
        var items = dequeRequests();
        if (items.isEmpty()) {
            return completedFuture(null);
        }

        var req = TResolveLogsRequest.newBuilder()
            .setNumId(numId)
            .addAllUnresolvedLogMeta(items.stream()
                .map(r -> ByteStrings.fromByteBuf(r.log.meta))
                .collect(Collectors.toList()))
            .build();

        var resolveFuture = new CompletableFuture<TResolveLogsResponse>();
        resolveUntilSuccess(req, resolveFuture, 0);
        var future = resolveFuture.handle((response, e) -> {
            if (e != null) {
                for (var item : items) {
                    item.doneFuture.completeExceptionally(e);
                }
            } else {
                processResolve(response, items);
            }

            return null;
        });
        aggrMetrics.logResolve.forFuture(future);
        return future;
    }

    private List<ResolveRequest> dequeRequests() {
        List<ResolveRequest> items = new ArrayList<>();
        ResolveRequest item;
        int messageSize = 0;
        while ((item = queue.poll()) != null) {
            messageSize += item.log.meta.readableBytes();
            if (!items.isEmpty() && messageSize >= maxSizeToResolve) {
                queue.addFirst(item);
                actor.schedule();
                break;
            }
            items.add(item);
        }
        Collections.sort(items);
        return items;
    }

    private void processResolve(TResolveLogsResponse response, List<ResolveRequest> items) {
        if (response.getStatus() != EMetabaseStatusCode.OK && response.getStatus() != EMetabaseStatusCode.QUOTA_ERROR) {
            items.forEach(r -> r.doneFuture.complete(Map.of()));
            return;
        }

        var record = new ResolvedLogMetricsRecord();
        for (int index = 0; index < items.size(); index++) {
            var doneFuture = items.get(index).doneFuture;
            try {
                var resolved = response.getResolvedLogMetrics(index);
                var result = new HashMap<Labels, Metric>();
                try (var it = new ResolvedLogMetricsIteratorImpl(ByteStrings.toByteBuf(resolved), labelAllocator)) {
                    while (it.next(record)) {
                        result.put(record.labels, new MetricImpl(record.shardId, record.localId, record.type));
                    }
                }
                executor.execute(() -> doneFuture.complete(result));
            } catch (Throwable e) {
                doneFuture.completeExceptionally(e);
            }
        }
    }

    private CompletableFuture<Int2ObjectMap<Log>> resolveAndParse(DumperTx tx, @WillClose Log log) {
        return resolve(tx, log)
            .handle((resolved, e) -> {
                if (e != null) {
                    bytesInFlight.addAndGet(-log.memorySizeIncludingSelf());
                    log.close();
                    Throwables.throwIfUnchecked(e);
                    throw new RuntimeException(e);
                }
                return parseResolved(resolved, tx, log);
            });
    }

    private Int2ObjectMap<Log> parseResolved(Map<Labels, Metric> resolved, DumperTx tx, @WillClose Log log) {
        if (resolved.isEmpty()) {
            addSettlers(log);
            return Int2ObjectMaps.emptyMap();
        }

        this.cache.add(resolved);
        var result = parse(new ParsingTask(tx, opts.decimPolicy, log, new SnapshotMetricsCache(resolved), labelAllocator, bufferAllocator));
        if (result.unresolved != null) {
            addSettlers(result.unresolved);
        }
        return result.resolved;
    }

    private void addSettlers(@WillClose Log log) {
        try (log) {
            var record = new UnresolvedLogMetaRecord();
            try (var it = new UnresolvedLogMetaIteratorImpl(log.meta, labelAllocator)) {
                while (it.next(record)) {
                    cache.addSettler(record.labels);
                }
            }
        }
    }

    private CompletableFuture<Map<Labels, Metric>> resolve(DumperTx tx, @WillNotClose Log log) {
        var future = new CompletableFuture<Map<Labels, Metric>>();
        queue.add(new ResolveRequest(tx, log, future));
        actor.schedule();
        return future;
    }

    private void resolveUntilSuccess(
            TResolveLogsRequest request,
            CompletableFuture<TResolveLogsResponse> doneFuture,
            int attempt)
    {
        if (closed) {
            doneFuture.completeExceptionally(new IllegalStateException("NumId:" + Integer.toUnsignedLong(numId) + " shard already stopped"));
            return;
        }

        CompletableFutures.safeCall(() -> storage.resolve(request)).whenComplete((r, e) -> {
            if (e != null) {
                logger.warn("NumId:{} failed resolve logs {}", Integer.toUnsignedString(numId), e);
                metrics.resolveError.inc();
                scheduleResolve(request, doneFuture, attempt + 1);
                return;
            }

            switch (r.getStatus()) {
                case OK:
                case QUOTA_ERROR:
                    metrics.resolveSuccess.inc();
                    doneFuture.complete(r);
                    return;
                default:
                    logger.warn("NumId:{} failed resolve logs caused by {} {}", Integer.toUnsignedString(numId), r.getStatus(), r.getStatusMessage());
                    metrics.resolveError.inc();
                    scheduleResolve(request, doneFuture, attempt + 1);
            }
        });
    }

    private void scheduleResolve(TResolveLogsRequest request, CompletableFuture<TResolveLogsResponse> doneFuture, int attempt) {
        try {
            long delay = DurationUtils.randomize(backoff(RESOLVE_SLEEP_MIN_MILLIS, RESOLVE_SLEEP_MAX_MILLIS, attempt));
            timer.schedule(() -> {
                try {
                    executor.execute(() -> resolveUntilSuccess(request, doneFuture, attempt));
                } catch (Throwable e) {
                    doneFuture.completeExceptionally(e);
                }
            }, delay, TimeUnit.MILLISECONDS);
        } catch (Throwable e) {
            doneFuture.completeExceptionally(e);
        }
    }

    private Int2ObjectMap<Log> combine(Int2ObjectMap<Log> left, Int2ObjectMap<Log> right) {
        try {
            if (left.isEmpty()) {
                return right;
            }

            if (right.isEmpty()) {
                return left;
            }

            var result = new Int2ObjectOpenHashMap<Log>(left.size());
            for (var entry : left.int2ObjectEntrySet()) {
                int shardId = entry.getIntKey();
                var log = Logs.combineResolvedLog(entry.getValue(), right.remove(shardId));
                if (log == null) {
                    continue;
                }

                result.put(entry.getIntKey(), log);
            }
            if (!right.isEmpty()) {
                result.putAll(right);
            }
            return result;
        } catch (Throwable e) {
            left.values().forEach(Log::close);
            right.values().forEach(Log::close);
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
        memory.addMemory("dumper.shard.process.cache.resolved", cache.resolvedMemorySize());
        memory.addMemory("dumper.shard.process.cache.settlers", cache.settlersMemorySize());
        memory.addMemory("dumper.shard.process.parsing", bytesInFlight.get());
        memory.addMemory("dumper.shard.process.labelsPool", labelAllocator.memorySizeIncludingSelf());
    }

    @Override
    public void close() {
        closed = true;
        cache.close();
    }

    @Override
    public String toString() {
        return "SolomonShardProcess{" +
                "projectId=" + opts.projectId +
                ", id=" + opts.id +
                ", numId=" + Integer.toUnsignedLong(numId) +
                '}';
    }

    private static class ResolveRequest implements Comparable<ResolveRequest> {
        private final DumperTx tx;
        @WillNotClose
        private final Log log;
        private final CompletableFuture<Map<Labels, Metric>> doneFuture;

        public ResolveRequest(DumperTx tx, Log log, CompletableFuture<Map<Labels, Metric>> doneFuture) {
            this.tx = tx;
            this.log = log;
            this.doneFuture = doneFuture;
        }

        @Override
        public int compareTo(ResolveRequest o) {
            return tx.compareTo(o.tx);
        }
    }
}
