package ru.yandex.solomon.dumper;

import java.util.Arrays;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.WillClose;

import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.util.internal.RecyclableArrayList;
import it.unimi.dsi.fastutil.ints.Int2LongMap;
import it.unimi.dsi.fastutil.ints.Int2LongOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.memory.layout.MemInfoProvider;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.LogsIndex;
import ru.yandex.solomon.slog.LogsIndexSerializer;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TWriteLogRequest;
import ru.yandex.stockpile.api.TWriteLogResponse;
import ru.yandex.stockpile.client.StockpileClient;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileWriterImpl implements StockpileWriter, MemInfoProvider, AutoCloseable {
    private static final int MAX_BATCH_SIZE_IN_BYTES = 128 << 20; // 128 MiB;
    private static final long WRITE_SLEEP_MIN_MILLIS = 500;
    private static final long WRITE_SLEEP_MAX_MILLIS = TimeUnit.MINUTES.toMillis(5L);

    private static final Logger logger = LoggerFactory.getLogger(StockpileWriterImpl.class);

    private volatile ShardQueue[] shardQueues;
    private volatile boolean closed;

    private final StockpileClient client;
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private final GaugeInt64 queueSize;
    private final GaugeInt64 queueSizeBytes;

    public StockpileWriterImpl(StockpileClient client, Executor executor, ScheduledExecutorService timer, MetricRegistry metricRegistry) {
        this.client = client;
        this.executor = executor;
        this.timer = timer;
        this.queueSize = metricRegistry.gaugeInt64("stockpile.write.queue.size");
        this.queueSizeBytes = metricRegistry.gaugeInt64("stockpile.write.queue.bytes");
        this.shardQueues = new ShardQueue[0];
    }

    public CompletableFuture<EStockpileStatusCode> write(int shardId, @WillClose List<Log> logs) {
        if (closed) {
            logs.forEach(Log::close);
            return CompletableFutures.exceptionally(new RuntimeException("shutdown invoked"));
        }
        try {
            return getQueue(shardId).enqueue(logs);
        } catch (Throwable e) {
            return failedFuture(e);
        }
    }

    private void ensureQueueInit(int shardId) {
        var copy = shardQueues;
        if (shardId >= copy.length) {
            int totalShards = Math.max(client.getTotalShardsCount(), shardId);
            if (copy.length < totalShards) {
                synchronized (this) {
                    shardQueues = changeQueueSize(shardQueues, totalShards);
                }
            }
        }
    }

    private ShardQueue[] changeQueueSize(ShardQueue[] prev, int expectedSize) {
        if (prev.length >= expectedSize) {
            return prev;
        }

        var result = Arrays.copyOf(prev, expectedSize);
        for (int index = prev.length; index < result.length; index++) {
            int shardId = index + 1;
            result[index] = new ShardQueue(shardId);
        }
        return result;
    }

    private ShardQueue getQueue(int shardId) {
        ensureQueueInit(shardId);
        return shardQueues[shardId - 1];
    }

    @Override
    public MemoryBySubsystem memoryBySystem() {
        var result = new MemoryBySubsystem();
        result.addMemory("stockpile.write.queue", queueSizeBytes.get());
        return result;
    }

    @Override
    public void close() {
        closed = true;
    }

    private class ShardQueue {
        private final int shardId;
        private final Queue<Item> queue;
        private final ActorWithFutureRunner actor;
        private final Int2LongMap latestSeenSeqNo = new Int2LongOpenHashMap();

        public ShardQueue(int shardId) {
            this.shardId = shardId;
            this.queue = new ConcurrentLinkedQueue<>();
            this.actor = new ActorWithFutureRunner(this::act, executor);
        }

        CompletableFuture<EStockpileStatusCode> enqueue(List<Log> logs) {
            var future = new CompletableFuture<EStockpileStatusCode>();
            Item item = new Item(logs, future);
            queueSize.add(1);
            queueSizeBytes.add(item.memorySizeIncludingSelf());
            queue.offer(item);
            actor.schedule();
            return future;
        }

        public CompletableFuture<?> act() {
            var items = dequeRequests();
            try {
                if (items.isEmpty()) {
                    items.recycle();
                    return completedFuture(null);
                }

                LogsIndex index = new LogsIndex(items.stream().mapToInt(value -> ((Item) value).logs.size()).sum());
                ByteString content = ByteString.EMPTY;
                for (var o : items) {
                    var item = (Item) o;
                    for (var log : item.logs) {
                        int producerId = ResolvedLogMetaHeader.producerId(log.meta);
                        long producerSeqNo = ResolvedLogMetaHeader.producerSeqNo(log.meta);
                        var prevProducerSeqNo = latestSeenSeqNo.get(producerId);
                        if (producerSeqNo > prevProducerSeqNo) {
                            latestSeenSeqNo.put(producerId, producerSeqNo);
                        } else if (producerId < prevProducerSeqNo) {
                            logger.info("ProducerId:{}, ProducerSeqNo:{} obsolete, latest visible ProducerSeqNo:{}", producerId, producerSeqNo, prevProducerSeqNo);
                        }
                        index.add(log.numId, log.meta.readableBytes(), log.data.readableBytes());
                        content = content.concat(ByteStrings.fromByteBuf(log.meta)).concat(ByteStrings.fromByteBuf(log.data));
                    }
                }

                ByteBuf serializedIndex = LogsIndexSerializer.serialize(ByteBufAllocator.DEFAULT, index);
                return write(TWriteLogRequest.newBuilder()
                    .setShardId(shardId)
                    .setIndex(ByteStrings.fromByteBuf(serializedIndex))
                    .setContent(content)
                    .build())
                    .handle((response, throwable) -> {
                        serializedIndex.release();
                        onComplete(items, response, throwable);
                        return null;
                    });
            } catch (Throwable e) {
                onComplete(items, null, e);
            }
            return completedFuture(null);
        }

        private CompletableFuture<TWriteLogResponse> write(TWriteLogRequest request) {
            if (!isRetryEnable()) {
                return client.writeLog(request);
            }

            return new RetryWrite(request).write();
        }

        private boolean isRetryEnable() {
            if (StockpileFormat.CURRENT.le(StockpileFormat.HISTOGRAM_DENOM_37)) {
                return false;
            }

            var format = defineFormat();
            return format.ge(StockpileFormat.IDEMPOTENT_WRITE_38);
        }

        private StockpileFormat defineFormat() {
            var range = client.getCompatibleCompressFormat();
            if (range.isEmpty()) {
                return StockpileFormat.CURRENT;
            }

            return StockpileFormat.byNumberOrCurrent(range.upperEndpoint());
        }

        private void onComplete(RecyclableArrayList items, TWriteLogResponse status, Throwable e) {
            if (e != null) {
                for (Object o : items) {
                    Item item = (Item) o;
                    item.releaseLog();
                    item.future.completeExceptionally(e);
                }
            } else {
                for (Object o : items) {
                    Item item = (Item) o;
                    item.releaseLog();
                    item.future.complete(status.getStatus());
                }
            }
            items.recycle();
        }

        private RecyclableArrayList dequeRequests() {
            final RecyclableArrayList items = RecyclableArrayList.newInstance();
            long batchSize = 0;
            long memoryUse = 0;
            Item item;
            while (batchSize < MAX_BATCH_SIZE_IN_BYTES && (item = queue.poll()) != null) {
                batchSize += item.getSizeBytes(); // actual batch size can be above the limit by size of the last item
                memoryUse += item.memorySizeIncludingSelf();
                items.add(item);
            }

            if (!items.isEmpty()) {
                queueSize.add(-items.size());
                queueSizeBytes.add(-memoryUse);
            }
            return items;
        }
    }

    private static class Item implements MemMeasurable {
        private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(Item.class);
        private final List<Log> logs;
        private final long memory;
        private final int bytes;
        private final CompletableFuture<EStockpileStatusCode> future;

        public Item(List<Log> logs, CompletableFuture<EStockpileStatusCode> future) {
            this.logs = logs;
            this.future = future;
            int bytes = 0;
            long memory = 0;
            for (var log : logs) {
                bytes += log.getSizeBytes();
                memory += log.memorySizeIncludingSelf();
            }
            this.bytes = bytes;
            this.memory = MemoryCounter.CompletableFuture_SELF_SIZE + SELF_SIZE + memory;
        }

        public void releaseLog() {
            for (var log : logs) {
                log.close();
            }
        }

        public int getSizeBytes() {
            return bytes;
        }

        @Override
        public long memorySizeIncludingSelf() {
            return memory;
        }
    }

    private class RetryWrite {
        private final TWriteLogRequest request;
        private long sleepMillis;
        private final CompletableFuture<TWriteLogResponse> doneFuture = new CompletableFuture<>();

        public RetryWrite(TWriteLogRequest request) {
            this.request = request;
        }

        public CompletableFuture<TWriteLogResponse> write() {
            tryWrite();
            return doneFuture;
        }

        private void tryWrite() {
            if (closed) {
                doneFuture.completeExceptionally(new IllegalStateException("Shard writer already closed"));
                return;
            }

            client.writeLog(request).whenComplete(this::onCompleteWrite);
        }

        private void onCompleteWrite(TWriteLogResponse response, @Nullable Throwable e) {
            try {
                if (e != null) {
                    logger.warn("Write to stockpile shard {} failed", request.getShardId(), e);
                    scheduleRetry();
                    return;
                }

                switch (response.getStatus()) {
                    case OK:
                    case INVALID_REQUEST:
                        doneFuture.complete(response);
                        return;
                    default:
                        logger.warn("Write to stockpile shard {} failed {}:{}", request.getShardId(), response.getStatus(), response.getStatusMessage());
                        scheduleRetry();
                }
            } catch (Throwable e2) {
                doneFuture.completeExceptionally(e2);
            }
        }

        private void scheduleRetry() {
            if (closed) {
                doneFuture.completeExceptionally(new IllegalStateException("Shard writer already closed"));
                return;
            }

            try {
                long jitter = ThreadLocalRandom.current().nextLong(WRITE_SLEEP_MIN_MILLIS * 2);
                long delay = jitter + sleepMillis;
                sleepMillis = Math.min(sleepMillis, WRITE_SLEEP_MAX_MILLIS) + jitter;
                timer.schedule(() -> {
                    try {
                        executor.execute(this::tryWrite);
                    } catch (Throwable e2) {
                        doneFuture.completeExceptionally(e2);
                    }
                }, delay, TimeUnit.MILLISECONDS);
            } catch (Throwable e) {
                doneFuture.completeExceptionally(e);
            }
        }
    }
}
