package ru.yandex.solomon.dumper.spring;

import java.util.Optional;
import java.util.concurrent.TimeUnit;

import com.yandex.ydb.table.SchemeClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientImpl;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.dumper.ThreadsConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.kikimr.KikimrTransportFactory;
import ru.yandex.solomon.dumper.DumperLocalShards;
import ru.yandex.solomon.dumper.DumperShardMetricsAggregated;
import ru.yandex.solomon.dumper.LongTermStorage;
import ru.yandex.solomon.dumper.SolomonShardOptsProvider;
import ru.yandex.solomon.dumper.mon.KvMon;
import ru.yandex.solomon.dumper.storage.shortterm.KvDumperLocalShardsSyncWithHiveProcess;
import ru.yandex.solomon.dumper.storage.shortterm.KvShortTermStorageDao;
import ru.yandex.solomon.dumper.storage.shortterm.KvTabletsMapping;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.ydb.YdbClients;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
public class KvContext {
    private static final Logger logger = LoggerFactory.getLogger(KvContext.class);
    private final ThreadPoolProvider threads;
    private final MetricRegistry registry;
    private final TKikimrClientConfig config;

    @Autowired
    public KvContext(ThreadPoolProvider threads, MetricRegistry registry, @Qualifier("KikimrClientConfig") TKikimrClientConfig config) {
        this.threads = threads;
        this.registry = registry;
        this.config = config;
    }

    @Bean
    public KikimrKvClient kikimrKvClient() {
        KikimrKvClientImpl client = createKvClient(config);
        KvMon.addMetrics(client, registry);
        return client;
    }

    @Bean
    public YdbClients ydbClients(Optional<IamTokenClient> iamTokenClient, SecretProvider secrets) {
        return new YdbClients("KikimrClientConfig", config, threads, registry, iamTokenClient, secrets);
    }

    @Bean
    public KvShortTermStorageDao kvShortTermStorageDao(KikimrKvClient kvClient) {
        return KvShortTermStorageDao.create(kvClient, registry);
    }

    @Bean
    public KvTabletsMapping kvTabletsMapping(KikimrKvClient kvClient, YdbClients ydb) {
        var timer = threads.getSchedulerExecutorService();
        var executor = threads.getExecutorService("CpuLowPriority", "");
        var root = config.getSchemaRoot() + "/ShortTermStorage";
        var volumePath = root + "/KV";
        executor.submit(() -> createSchema(ydb.getSchemeClient(), root));
        return new KvTabletsMapping(volumePath, kvClient, executor, timer);
    }

    @Bean
    public KvDumperLocalShardsSyncWithHiveProcess hiveProcess(
            @Qualifier("ClusterId") Optional<Integer> clientId,
            KvShortTermStorageDao dao,
            LongTermStorage longTermStorage,
            DumperShardMetricsAggregated metrics,
            KvTabletsMapping mapping,
            DumperLocalShards localShards,
            SolomonShardOptsProvider optsProvider,
            ThreadsConfig threadsConfig)
    {
        var timer = threads.getSchedulerExecutorService();
        var executor = threads.getExecutorService(threadsConfig.getMiscThreadPool(), "Threads.MiscThreadPool");
        var parseExecutor = threads.getExecutorService(threadsConfig.getParsingThreadPool(), "Threads.ParsingThreadPool");
        var localKikimr = createKvClient(TKikimrClientConfig.newBuilder()
                .setGrpcConfig(TGrpcClientConfig.newBuilder()
                        .setThreadPoolName("CpuLowPriority")
                        .addAddresses("localhost:2135")
                        .build())
                .build());
        return new KvDumperLocalShardsSyncWithHiveProcess(
                clientId.orElse(0),
                dao,
                longTermStorage,
                metrics,
                executor,
                parseExecutor,
                timer,
                optsProvider,
                mapping,
                localKikimr,
                localShards);
    }

    private KikimrKvClientImpl createKvClient(TKikimrClientConfig config) {
        var factory = new KikimrTransportFactory("KikimrClientConfig", threads, registry);
        var transport = factory.newTransport(config);
        return new KikimrKvClientImpl(transport);
    }


    private void createSchema(SchemeClient client, String volumePath) {
        CompletableFutures.safeCall(() -> client.makeDirectories(volumePath))
            .thenAccept(status -> status.expect("Unable to create path: " + volumePath))
            .whenComplete((ignore, e) -> {
                if (e != null) {
                    logger.warn("Unable to create schema:", e);
                    threads.getSchedulerExecutorService().schedule(() -> createSchema(client, volumePath), 5, TimeUnit.SECONDS);
                }
            });
    }
}
