package ru.yandex.solomon.dumper.spring;

import java.util.Optional;

import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TStockpileClientConfig;
import ru.yandex.solomon.config.protobuf.metabase.client.TMetabaseClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.dumper.LongTermStorage;
import ru.yandex.solomon.dumper.LongTermStorageImpl;
import ru.yandex.solomon.dumper.StockpileWriter;
import ru.yandex.solomon.dumper.StockpileWriterImpl;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
        GrpcConfigurationContext.class
})
public class LongTermStorageContext {
    private final ThreadPoolProvider threads;
    private final MetricRegistry registry;

    @Autowired
    public LongTermStorageContext(ThreadPoolProvider threads, MetricRegistry registry) {
        this.threads = threads;
        this.registry = registry;
    }

    @Bean
    public StockpileClient stockpileClient(
        TStockpileClientConfig config,
        @Qualifier("ClientId") Optional<String> clientId,
        ClientOptionsFactory clientOptionsFactory)
    {
        var opts = StockpileClientOptions.newBuilder(
                clientOptionsFactory.newBuilder(
                        "StockpileClientConfig",
                        config.getGrpcConfig())
                    .setClientId(getClientId(clientId))
                    .setMetricRegistry(registry))
            .setFromConfig(config)
            .build();

        var addresses = config.getGrpcConfig().getAddressesList();
        return StockpileClients.createDynamic(addresses, opts);
    }

    @Bean
    public StockpileWriterImpl stockpileWriter(StockpileClient client) {
        var executor = threads.getExecutorService("CpuLowPriority", "ThreadPools");
        var timer = threads.getSchedulerExecutorService();
        return new StockpileWriterImpl(client, executor, timer, registry);
    }

    @Bean
    public MetabaseClient metabaseClient(
            TMetabaseClientConfig config,
            @Qualifier("ClientId") Optional<String> clientId,
            ClientOptionsFactory clientOptionsFactory)
    {
        var opts = MetabaseClientOptions.newBuilder(
                        clientOptionsFactory.newBuilder(
                        "MetabaseClientConfig",
                        config.getGrpcConfig())
                    .setClientId(getClientId(clientId))
                    .setMetricRegistry(registry))
            .setFromConfig(config)
            .setDiscoveryService(DiscoveryService.async())
            .build();

        return MetabaseClients.createDynamic(config.getGrpcConfig().getAddressesList(), opts);
    }

    @Bean
    public LongTermStorage longTermStorage(StockpileWriter stockpile, MetabaseClient metabase) {
        return new LongTermStorageImpl(stockpile, metabase);
    }

    private String getClientId(Optional<String> clientId) {
        return clientId.map(Strings::emptyToNull).orElseGet(HostUtils::getShortName);
    }

}
