package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.WillCloseWhenClosed;

import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.dumper.DumperLocalShards;
import ru.yandex.solomon.dumper.DumperShard;
import ru.yandex.solomon.dumper.DumperShardId;
import ru.yandex.solomon.dumper.DumperShardMetrics;
import ru.yandex.solomon.dumper.DumperShardMetricsAggregated;
import ru.yandex.solomon.dumper.LongTermStorage;
import ru.yandex.solomon.dumper.SolomonShardOptsProvider;
import ru.yandex.solomon.selfmon.executors.CpuMeasureExecutor;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;

/**
 * @author Vladimir Gordiychuk
 */
public class KvDumperLocalShardsSyncWithHiveProcess implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(KvDumperLocalShardsSyncWithHiveProcess.class);

    private final int clusterId;
    private final KvShortTermStorageDao shortTermStorageDao;
    private final LongTermStorage longTermStorage;
    private final DumperShardMetricsAggregated aggrMetrics;
    private final ExecutorService executor;
    private final ExecutorService parseExecutor;
    private final ScheduledExecutorService timer;
    private final KvTabletsMapping kvTabletsMapping;
    @WillCloseWhenClosed
    private final KikimrKvClient kvClient;
    private final DumperLocalShards shards;
    private final SolomonShardOptsProvider optsProvider;

    private final ActorWithFutureRunner actor;
    private final ScheduledFuture<?> future;
    private volatile boolean closed;

    private Throwable lastError;
    @InstantMillis
    private long lastErrorInstant;

    public KvDumperLocalShardsSyncWithHiveProcess(
        int clusterId,
        KvShortTermStorageDao shortTermStorageDao,
        LongTermStorage longTermStorage,
        DumperShardMetricsAggregated metrics,
        ExecutorService executor,
        ExecutorService parseExecutor,
        ScheduledExecutorService timer,
        SolomonShardOptsProvider optsProvider,
        KvTabletsMapping kvTabletsMapping,
        @WillCloseWhenClosed KikimrKvClient localKvClient,
        DumperLocalShards shards)
    {
        ProducerKey.validateClusterId(clusterId);
        this.clusterId = clusterId;
        this.shortTermStorageDao = shortTermStorageDao;
        this.longTermStorage = longTermStorage;
        this.optsProvider = optsProvider;
        this.aggrMetrics = metrics;
        this.executor = executor;
        this.parseExecutor = parseExecutor;
        this.timer = timer;
        this.kvTabletsMapping = kvTabletsMapping;
        this.kvClient = localKvClient;
        this.shards = shards;
        this.actor = new ActorWithFutureRunner(this::act, executor);
        this.future = timer.scheduleAtFixedRate(this::schedule, 0, 5, TimeUnit.SECONDS);
    }

    @ManagerMethod
    public void schedule() {
        actor.schedule();
    }

    public CompletableFuture<?> act() {
        if (closed) {
            return CompletableFuture.completedFuture(null);
        }

        if (!optsProvider.isInitialized()) {
            logger.warn("Solomon shard options not initialized yet");
            return CompletableFuture.completedFuture(null);
        }

        return kvTabletsMapping.getReadyFuture()
            .thenCompose(ignore -> kvClient.findTabletsOnLocalhost())
            .thenAccept(this::actualizeLocalShards)
            .exceptionally(e -> {
                logger.error("sync local shards with Hive failed", e);
                this.lastError = e;
                this.lastErrorInstant = System.currentTimeMillis();
                return null;
            });
    }

    private void actualizeLocalShards(long[] tabletIds) {
        var localTabletIds = new LongOpenHashSet(tabletIds);
        for (int shardId = DumperShardId.FIRST_SHARD_ID; shardId <= DumperShardId.SHARD_COUNT; shardId++) {
            long kvTabletId = kvTabletsMapping.getTabletId(shardId);

            boolean shardIsLocal = localTabletIds.contains(kvTabletId);

            // stop moved shards
            {
                DumperShard shard = shards.getShardById(shardId);
                if (shard != null) {
                    if (shardIsLocal) {
                        continue;
                    }

                    logger.info("stop shard {}, kvTabletId {}, reason tablet moved to different host", shardId, kvTabletId);
                    if (shards.remove(shard)) {
                        shard.stop();
                    }
                }
            }

            if (shardIsLocal) {
                var metrics = new DumperShardMetrics(Integer.toString(shardId), aggrMetrics);
                var executor = new CpuMeasureExecutor(metrics.cpuTimeNanos, this.executor);
                var parseExecutor = new CpuMeasureExecutor(metrics.cpuTimeNanos, this.parseExecutor);
                var reader = new KvShortTermStorageReader(clusterId, shardId, kvTabletId, 0, metrics, shortTermStorageDao, executor, timer);
                var shard = new DumperShard(shardId, executor, parseExecutor, timer, metrics, optsProvider, reader, longTermStorage);
                if (shards.addShard(shard)) {
                    logger.info("start shard {}, kvTabletId {}", shardId, kvTabletId);
                    shard.start();
                }
            }
        }
    }

    @Override
    public void close() {
        future.cancel(false);
        closed = true;
    }
}
