package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import ru.yandex.kikimr.util.NameRange;
import ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileType;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class KvLogReader implements MemMeasurable, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(KvLogReader.class);

    private final long tableId;
    private final KvShortTermStorageDao dao;
    private final KvRetry retry;
    private final int cacheBytesCapacity;
    private final Queue<TxDumperFile> cache;
    private final AtomicLong memoryInQueue = new AtomicLong();
    private final AtomicBoolean reading = new AtomicBoolean();
    private volatile boolean closed;

    public KvLogReader(long tableId, KvShortTermStorageDao dao, KvRetry retry, int cacheBytesCapacity) {
        this.tableId = tableId;
        this.dao = dao;
        this.retry = retry;
        this.cacheBytesCapacity = cacheBytesCapacity;
        this.cache = new ConcurrentLinkedQueue<>();
    }

    public CompletableFuture<TxDumperFile> readNext(long gen, long txn, long maxTxn) {
        if (closed) {
            return failedFuture(new IllegalStateException("KvReader from tablet " + tableId + " already closed"));
        }

        var cached = cache.peek();
        if (cached != null) {
            if (cached.txn != txn) {
                return failedFuture(new IllegalStateException("Non sequential read from cache, read " + cached.txn + " requested " + txn));
            }

            if (cache.poll() != cached) {
                return failedFuture(new IllegalStateException("Concurrent call readNext"));
            }
            memoryInQueue.addAndGet(-cached.memorySizeIncludingSelf());
            return completedFuture(cached);
        }

        int capacity = Math.toIntExact(Math.min(100L, maxTxn - txn + 1));
        var list = new TxDumperFileList(capacity, txn);
        var from = "c." + DumperLogFileName.format(txn);
        var to = "c." + DumperLogFileName.format(maxTxn + 1);
        var range = NameRange.inclusiveExclusive(from, to);
        if (!reading.compareAndSet(false, true)) {
            return failedFuture(new IllegalStateException("Concurrent call readNext"));
        }

        return readDataRange(gen, range, list)
            .thenApply(ignore -> {
                memoryInQueue.addAndGet(list.memorySizeIncludingSelf());
                cache.addAll(list.files);
                if (closed) {
                    releaseMemory();
                    throw new IllegalStateException("Already closed");
                }
                var result = cache.poll();
                if (result == null || result != list.files.get(0)) {
                    throw new IllegalStateException("Concurrent call readNext");
                }
                memoryInQueue.addAndGet(-result.memorySizeIncludingSelf());
                return result;
            })
            .whenComplete((ignore, ignore2) -> reading.set(false));
    }

    private CompletableFuture<Void> readDataRange(long gen, NameRange range, TxDumperFileList list) {
        CompletableFuture<Void> doneFuture = new CompletableFuture<>();
        readDataRange(gen, range, list, doneFuture);
        return doneFuture;
    }

    private void readDataRange(long gen, NameRange range, TxDumperFileList list, CompletableFuture<Void> doneFuture) {
        retry.loopUntilSuccess("readRange(" + range + ")", (attempt) -> dao.readRange(tableId, gen, range, cacheBytesCapacity))
            .whenComplete((result, e) -> {
                if (e != null) {
                    list.close();
                    doneFuture.completeExceptionally(e);
                    return;
                }

                try {
                    for (var entry : result.getEntries()) {
                        var fileName = ensureFileNameValid(entry.getName());
                        list.append(fileName, entry.getValue(), entry.toStats().getCreatedUnixtime());
                        if (list.lastNotFinished == null && list.memorySizeIncludingSelf() >= cacheBytesCapacity) {
                            onLoadDone(list);
                            doneFuture.complete(null);
                            return;
                        }
                    }

                    if (!result.isOverrun()) {
                        onLoadDone(list);
                        doneFuture.complete(null);
                        return;
                    }

                    final String end;
                    if (list.memorySizeIncludingSelf() >= cacheBytesCapacity) {
                        if (list.lastNotFinished == null) {
                            onLoadDone(list);
                            doneFuture.complete(null);
                            return;
                        }
                        end = "c." + DumperLogFileName.format(list.lastNotFinished.txn + 1);
                    } else {
                        end = range.getEnd();
                    }

                    var nextRange = new NameRange(result.getOverrunLastEntryName(), false, end, false);
                    readDataRange(gen, nextRange, list, doneFuture);
                } catch (Throwable e2) {
                    doneFuture.completeExceptionally(e2);
                    list.close();
                }
            });
    }

    private void onLoadDone(TxDumperFileList list) {
        list.ensureLatestFinished();
        list.ensureNotEmpty();
    }

    private String ensureFileNameValid(String fileName) {
        String expectPrefix = "c." + FileType.DUMPER_LOG.getPrefix();
        if (!fileName.startsWith(expectPrefix)) {
            throw new IllegalStateException("Expected prefix " + expectPrefix + " for name " + fileName);
        }
        return fileName.substring(expectPrefix.length());
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + memoryInQueue.get();
    }

    @Override
    public void close() {
        closed = true;
        releaseMemory();
    }

    private void releaseMemory() {
        TxDumperFile file;
        while ((file = cache.poll()) != null) {
            file.close();
        }
    }
}
