package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.IntFunction;

import ru.yandex.misc.concurrent.CompletableFutures;

import static ru.yandex.solomon.util.time.DurationUtils.backoff;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;

/**
 * @author Vladimir Gordiychuk
 */
public class KvRetry implements AutoCloseable {
    private static final long SLEEP_MIN_MILLIS = 1_000;
    private static final long SLEEP_MAX_MILLIS = 60_000;

    private final Consumer<Throwable> exceptionConsumer;
    private final ScheduledExecutorService timer;
    public volatile boolean closed;

    public KvRetry(Consumer<Throwable> exceptionConsumer, ScheduledExecutorService timer) {
        this.exceptionConsumer = exceptionConsumer;
        this.timer = timer;
    }

    public  <T> CompletableFuture<T> loopUntilSuccess(String name, IntFunction<CompletableFuture<T>> supplier) {
        CompletableFuture<T> result = new CompletableFuture<>();
        loopUntilSuccess(name, supplier, result, 0);
        return result;
    }

    private <T> void loopUntilSuccess(
        String name,
        IntFunction<CompletableFuture<T>> supplier,
        CompletableFuture<T> resultFuture,
        int attempt)
    {
        if (closed) {
            resultFuture.completeExceptionally(new IllegalStateException("Retry context already closed"));
            return;
        }

        CompletableFutures.safeCall(() -> supplier.apply(attempt)).whenComplete((r, e) -> {
            try {
                if (e == null) {
                    resultFuture.complete(r);
                    return;
                }

                if (KvExceptionHandler.isGenerationChanged(e)) {
                    closed = true;
                    resultFuture.completeExceptionally(e);
                    return;
                }

                if (closed) {
                    resultFuture.completeExceptionally(new IllegalStateException("Retry context already closed"));
                    return;
                }

                onError(new Exception(name + " failed, attempt " + attempt, e));
                long executeAt = randomize(backoff(SLEEP_MIN_MILLIS, SLEEP_MAX_MILLIS, attempt + 1));
                timer.schedule(() -> loopUntilSuccess(name, supplier, resultFuture, attempt + 1), executeAt, TimeUnit.MILLISECONDS);
            } catch (Throwable e2) {
                resultFuture.completeExceptionally(e2);
                onError(e2);
            }
        });
    }

    private void onError(Throwable e) {
        exceptionConsumer.accept(e);
    }

    @Override
    public void close() {
        closed = true;
    }
}
