package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.function.ToLongFunction;

import javax.annotation.Nonnull;

import com.google.common.collect.ImmutableMap;
import io.grpc.StatusException;
import io.grpc.StatusRuntimeException;

import ru.yandex.kikimr.client.KikimrAnyResponseException;
import ru.yandex.kikimr.client.ResponseStatus;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvGenerationChangedRuntimeException;
import ru.yandex.kikimr.client.kv.KvRange;
import ru.yandex.kikimr.client.kv.KvReadRangeResult;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

/**
 * @author Vladimir Gordiychuk
 */
public class KvShortTermStorageDao {
    private final KikimrKvClient kvClient;
    private final long limitBytes;
    private final Map<String, Metrics> metricsByMethod;
    private final Metrics total;

    private KvShortTermStorageDao(KikimrKvClient kvClient, long limitBytes, MetricRegistry registry) {
        this.kvClient = kvClient;
        this.limitBytes = limitBytes;
        this.metricsByMethod = ImmutableMap.<String, Metrics>builder()
            .put("lock", new Metrics(registry.subRegistry(Labels.of("method", "lock"))))
            .put("listFiles", new Metrics(registry.subRegistry(Labels.of("method", "listFiles"))))
            .put("deleteFiles", new Metrics(registry.subRegistry(Labels.of("method", "deleteFiles"))))
            .put("readRange", new Metrics(registry.subRegistry(Labels.of("method", "readRange"))))
            .put("renameFiles", new Metrics(registry.subRegistry(Labels.of("method", "renameFiles"))))
            .put("readFile", new Metrics(registry.subRegistry(Labels.of("method", "readFile"))))
            .build();
        this.total = new Metrics(registry.subRegistry(Labels.of("method", "total")));
    }

    public static KvShortTermStorageDao create(KikimrKvClient kvClient, MetricRegistry registry, long limitBytes) {
        return new KvShortTermStorageDao(kvClient, limitBytes, registry.subRegistry("dao", "ShortTermStorageDao"));
    }

    public static KvShortTermStorageDao create(KikimrKvClient kvClient, MetricRegistry registry) {
        return create(kvClient, registry, KvRange.LEN_UNLIMITED);
    }

    public CompletableFuture<Long> lock(long tableId) {
        var metrics = metricsByMethod.get("lock");
        var future = kvClient.incrementGeneration(tableId, expiredAt());
        return measureAsyncOp(future, metrics, value -> 0);
    }

    public CompletableFuture<KvReadRangeResult> listFiles(long tableId, long gen, NameRange range) {
        var metrics = metricsByMethod.get("listFiles");
        var future = kvClient.readRange(tableId, gen, range, false, KikimrKvClient.DO_NOT_EXCEED_FILE_SIZE, expiredAt());
        return measureAsyncOp(future, metrics, value -> 0);
    }

    public CompletableFuture<Void> deleteFiles(long tabletId, long gen, List<NameRange> ranges) {
        var metrics = metricsByMethod.get("deleteFiles");
        var future = kvClient.deleteRanges(tabletId, gen, ranges, expiredAt());
        return measureAsyncOp(future, metrics, value -> 0);
    }

    public CompletableFuture<Optional<byte[]>> readFile(long tableId, long gen, String name) {
        var metrics = metricsByMethod.get("readFile");
        var future = kvClient.readData(tableId, gen, name, expiredAt(), MsgbusKv.TKeyValueRequest.EPriority.REALTIME);
        return measureAsyncOp(future, metrics, value -> value.map(bytes -> bytes.length).orElse(0));
    }

    public CompletableFuture<KvReadRangeResult> readRange(long tabletId, long gen, NameRange range) {
        return readRange(tabletId, gen, range, limitBytes);
    }

    public CompletableFuture<KvReadRangeResult> readRange(long tabletId, long gen, NameRange range, long limitBytes) {
        var metrics = metricsByMethod.get("readRange");
        var future = kvClient.readRange(tabletId, gen, range, true, limitBytes, expiredAt());
        return measureAsyncOp(future, metrics, KvReadRangeResult::bytesSize);
    }

    public CompletableFuture<Void> renameFiles(long tableId, long gen, List<KikimrKvClient.Rename> renames) {
        return renameFiles(tableId, gen, renames, List.of());
    }

    public CompletableFuture<Void> renameFiles(long tabletId, long gen, List<KikimrKvClient.Rename> renames, List<KikimrKvClient.Write> writes) {
        var metrics = metricsByMethod.get("renameFiles");
        var future = kvClient.writeAndRename(tabletId, gen, writes, renames, expiredAt());
        return measureAsyncOp(future, metrics, value -> 0);
    }

    private <T> CompletableFuture<T> measureAsyncOp(CompletableFuture<T> future, Metrics metrics, ToLongFunction<T> toBytesSize) {
        metrics.async.callStarted();
        total.async.callStarted();
        long startTimeNanos = System.nanoTime();
        future.whenComplete((r, e) -> {
            long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos);
            if (e != null) {
                String statusName = errorStatusName(e);
                metrics.callCompletedError(elapsedMillis, statusName);
                total.callCompletedError(elapsedMillis, statusName);
                return;
            }

            metrics.callCompletedOk(elapsedMillis);
            total.callCompletedOk(elapsedMillis);
            long bytes = toBytesSize.applyAsLong(r);
            metrics.inboundBytes.add(bytes);
            total.inboundBytes.add(bytes);
        });
        return future;
    }

    private static long expiredAt() {
        // TODO: configurable outside
        return System.currentTimeMillis() + 30_000;
    }

    private static class Metrics {
        final MetricRegistry registry;
        final AsyncMetrics async;
        final Rate inboundBytes;
        final ConcurrentMap<String, Rate> errors = new ConcurrentHashMap<>();

        Metrics(MetricRegistry registry) {
            this.registry = registry;
            this.async = new AsyncMetrics(registry, "ydb.ops");
            this.inboundBytes = registry.rate("ydb.ops.inBoundBytes");
        }

        public void callCompletedError(long elapsedMillis, String errorName) {
            async.callCompletedError(elapsedMillis);
            Rate rate = errors.computeIfAbsent(errorName, status -> registry.rate("ydb.ops.errors", Labels.of("code", status)));
            rate.inc();
        }

        public void callCompletedOk(long elapsedMillis) {
            async.callCompletedOk(elapsedMillis);
        }
    }

    private static String errorStatusName(@Nonnull Throwable e) {
        Throwable cause = e;
        while (cause != null) {
            if (cause instanceof KikimrAnyResponseException) {
                KikimrAnyResponseException kve = (KikimrAnyResponseException) cause;
                MsgbusKv.TKeyValueResponse response = (MsgbusKv.TKeyValueResponse) kve.getResponse();
                ResponseStatus status = ResponseStatus.valueOf(response.getStatus());
                if (status != ResponseStatus.MSTATUS_UNKNOWN || response.getStatus() == 0) {
                    return status.name();
                } else {
                    return String.valueOf(response.getStatus());
                }
            } else if (cause instanceof StatusException) {
                return ((StatusException) cause).getStatus().getCode().name();
            } else if (cause instanceof StatusRuntimeException) {
                return ((StatusRuntimeException) cause).getStatus().getCode().name();
            } else if (cause instanceof KikimrKvGenerationChangedRuntimeException) {
                return "GENERATION_MISMATCH";
            }

            cause = cause.getCause();
        }

        return "UNKNOWN";
    }
}
