package ru.yandex.solomon.dumper.storage.shortterm;

import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.primitives.Longs;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.dumper.DumperShardMetrics;
import ru.yandex.solomon.dumper.storage.shortterm.file.CompositeFileNameCollector;
import ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileName;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileNamesList;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileType;
import ru.yandex.solomon.dumper.storage.shortterm.file.MemstoreSnapshotFileName;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.selfmon.failsafe.RateLimiter;
import ru.yandex.solomon.selfmon.failsafe.TokenBucketRateLimiter;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * memstore
 * c.m.type.nodeId.txn
 * c.m.s.000.0000001
 * c.m.s.001.0000001
 *
 * dumper
 * c.d.type.txn
 * c.d.l.0000001
 *
 * @author Vladimir Gordiychuk
 */
public class KvShortTermStorageReader implements ShortTermStorageReader, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(KvShortTermStorageReader.class);
    private static final NameRange MEMSTORE_FILES_RANGE = StringMicroUtils.asciiPrefixToRange("c." + FileType.MEMSTORE_SNAPSHOT.getPrefix());
    private static final long MAX_PREPARE_TX_FILES = 100;
    private static final RateLimiter errorLogLimiter = new TokenBucketRateLimiter(Clock.systemUTC(), 100, Duration.ofSeconds(1));

    private final int clusterId;
    private final int shardId;
    private final long tableId;
    private final KvShortTermStorageDao dao;
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private final KvRetry retry;
    private final KvLogReader logReader;
    private final ActorWithFutureRunner actor;
    private final DumperShardMetrics metrics;

    private long gen;
    private boolean init;
    private ScheduledFuture<?> periodicAct;

    private final AtomicLong latestWriteTxn = new AtomicLong();
    private final AtomicLong latestReadTxn = new AtomicLong();
    private final AtomicLong latestDeleteTxn = new AtomicLong();

    private volatile boolean stop = false;
    private final AtomicReference<CompletableFuture<Long>> waitTxnFuture = new AtomicReference<>(completedFuture(null));
    private volatile CountDownLatch actSync = new CountDownLatch(0);

    Throwable lastError;
    @InstantMillis
    public long lastErrorInstant;

    public KvShortTermStorageReader(int clusterId, int shardId, long tableId, long gen, DumperShardMetrics metrics, KvShortTermStorageDao dao, Executor executor, ScheduledExecutorService timer) {
        this.clusterId = clusterId;
        this.dao = dao;
        this.metrics = metrics;
        this.shardId = shardId;
        this.tableId = tableId;
        this.gen = gen;
        this.executor = executor;
        this.timer = timer;
        this.retry = new KvRetry(this::error, timer);
        this.logReader = new KvLogReader(tableId, dao, retry, 50 << 20); // 50 MiB
        this.actor = new ActorWithFutureRunner(this::act, executor);
    }

    private CompletableFuture<Void> act() {
        if (stop) {
            if (periodicAct != null) {
                periodicAct.cancel(false);
                periodicAct = null;
            }
            var wait = waitTxnFuture.get();
            if (!wait.isDone()) {
                wait.completeExceptionally(new IllegalStateException("Shard " + shardId + " already stopped"));
            }
            return completedFuture(null);
        }

        return init()
            .thenCompose(ignore -> listFiles(MEMSTORE_FILES_RANGE))
            .thenCompose(this::prepareDumperTx)
            .thenAccept(lastWriteTxn -> {
                var wait = waitTxnFuture.get();
                this.latestWriteTxn.set(lastWriteTxn);
                if (wait.isDone()) {
                    return;
                }

                if (lastWriteTxn > this.latestReadTxn.get()) {
                    wait.complete(latestReadTxn.incrementAndGet());
                } else {
                    // No more new txn, lag is zero
                    metrics.readLagSec.set(0);
                    metrics.writeLagSec.set(0);
                }
            })
            .handle((ignore, e) -> {
                if (e != null) {
                    if (!KvExceptionHandler.isGenerationChanged(e)) {
                        error(new Exception("act failed", e));
                    } else {
                        stop();
                        actor.schedule();
                    }
                }

                actSync.countDown();
                return null;
            });
    }

    private CompletableFuture<Void> init() {
        CompletableFuture<Void> future = completedFuture(null);
        if (init) {
            return future;
        }

        if (gen == 0) {
            future = future.thenCompose(ignore -> retry.loopUntilSuccess("lock", (attempt) -> dao.lock(tableId)))
                .thenAccept(gen -> this.gen = gen);
        }

        return future.thenCompose(ignore -> listFiles(StringMicroUtils.asciiPrefixToRange("c.d.")))
            .thenCompose(fileNamesList -> {
                metrics.set(fileNamesList);
                List<DumperLogFileName> logs = fileNamesList.list(FileType.DUMPER_LOG);
                if (!logs.isEmpty()) {
                    ensureTxnSequential(logs);
                    latestWriteTxn.set(logs.get(logs.size() - 1).getTxn());
                    latestReadTxn.set(logs.get(0).getTxn() - 1);
                    latestDeleteTxn.set(logs.get(0).getTxn() - 1);
                    metrics.pendingTxn.set(latestWriteTxn.get() - latestReadTxn.get());
                    return completedFuture(null);
                }

                return readLatestTxn()
                    .thenAccept(latestTxn -> {
                        latestWriteTxn.set(latestTxn);
                        latestReadTxn.set(latestTxn);
                        latestDeleteTxn.set(latestTxn);
                        metrics.pendingTxn.set(0);
                    });
            })
            .thenAccept(ignore -> {
                init = true;
                periodicAct = timer.scheduleAtFixedRate(actor::schedule, ThreadLocalRandom.current().nextInt(15_000), 15_000, TimeUnit.MILLISECONDS);
            });
    }

    private void ensureTxnSequential(List<DumperLogFileName> logs) {
        for (int index = 1; index < logs.size(); index++) {
            var prev = logs.get(index - 1);
            var next = logs.get(index);
            if (next.getTxn() - prev.getTxn() != 1) {
                var slice = logs.subList(Math.min(0, index - 2), Math.min(index + 3, logs.size()));
                throw new IllegalStateException("Invalid logs order: " + slice);
            }
        }
    }

    private CompletableFuture<Long> prepareDumperTx(FileNamesList files) {
        metrics.set(files);
        List<MemstoreSnapshotFileName> snapshots = files.list(FileType.MEMSTORE_SNAPSHOT);
        if (snapshots.isEmpty()) {
            return completedFuture(this.latestWriteTxn.get());
        }

        List<KikimrKvClient.Rename> renames = new ArrayList<>();
        long lastWriteTxn = this.latestWriteTxn.get();
        long fileSize = 0;
        long fileCount = 0;
        for (var snapshot: snapshots) {
            if (!snapshot.isValid()) {
                logger.warn("ShardId:{}, TabletId:{} invalid memstore file {}", shardId, tableId, snapshot);
                continue;
            }

            if (renames.size() > MAX_PREPARE_TX_FILES) {
                actor.schedule();
                break;
            }

            fileSize += snapshot.getBytesSize();
            fileCount += snapshot.getFileCount();
            renames.addAll(toDumperLog(++lastWriteTxn, snapshot));
        }

        if (renames.isEmpty()) {
            return completedFuture(lastWriteTxn);
        }

        return renamePreparedTxn(lastWriteTxn, fileSize, fileCount, renames);
    }

    private CompletableFuture<Long> renamePreparedTxn(long lastWriteTxn, long bytes, long count, List<KikimrKvClient.Rename> renames) {
        String name = "renameFiles(txn:" + lastWriteTxn + ", bytes: " + bytes + ", count: " + count +")";
        return retry.loopUntilSuccess(name, (attempt) -> {
            // previous rename failed, need check success it was or not, and only after that retry it
            // instead of it we read latest written txn because after some time act will be scheduled again
            if (attempt > 0) {
                actor.schedule();
                return readLatestTxn();
            }

            return dao.renameFiles(tableId, gen, renames, List.of(prepareWriteLatestTxn(lastWriteTxn)))
                    .thenApply(ignore -> lastWriteTxn);
        }).thenApply(latestTxn -> {
            long prevLatestTxn = latestWriteTxn.get();
            if (latestTxn != prevLatestTxn) {
                metrics.change(FileType.MEMSTORE_SNAPSHOT, FileType.DUMPER_LOG, bytes, count);
                metrics.prepareTxn(latestTxn - prevLatestTxn);
            }
            return latestTxn;
        });
    }

    private KikimrKvClient.Write prepareWriteLatestTxn(long latestTxn) {
        return new KikimrKvClient.Write(
            "c." + FileType.DUMPER_LATEST_TXN.getPrefix(),
            ByteStringsStockpile.unsafeWrap(Longs.toByteArray(latestTxn)),
            MsgbusKv.TKeyValueRequest.EStorageChannel.INLINE,
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME);
    }

    private CompletableFuture<Long> readLatestTxn() {
        return retry.loopUntilSuccess("readLatestTxn", (attempt) -> {
            return dao.readFile(tableId, gen, "c." + FileType.DUMPER_LATEST_TXN.getPrefix());
        }).thenApply(bytes -> {
            if (bytes.isEmpty()) {
                return 0L;
            }

            var array = bytes.get();
            if (array.length < Longs.BYTES) {
                return 0L;
            }

            return Longs.fromByteArray(array);
        });
    }

    private List<KikimrKvClient.Rename> toDumperLog(long dumperTxn, MemstoreSnapshotFileName log) {
        var result = new ArrayList<KikimrKvClient.Rename>(log.getFileCount());
        var memstoreTxn = log.getTxn();
        for (int chunkNo = 0; chunkNo < log.getFileCount(); chunkNo++) {
            boolean last = chunkNo + 1 == log.getFileCount();
            var from = "c." + MemstoreSnapshotFileName.format(memstoreTxn, chunkNo, last);
            var to = "c." + DumperLogFileName.format(dumperTxn, chunkNo, last);
            var rename = new KikimrKvClient.Rename(from, to);
            result.add(rename);
        }
        return result;
    }

    @Override
    public CompletableFuture<DumperFile> next() {
        return pullNextTxn()
            .thenCompose(this::readTxn)
            .whenComplete((ignore, e) -> {
                if (e != null) {
                    latestReadTxn.decrementAndGet();
                }
            });
    }

    private CompletableFuture<DumperFile> readTxn(long txn) {
        return metrics.aggregated.txRead.wrapFuture(() -> {
            long maxTxn = Math.min(txn + 100_000, latestWriteTxn.get());
            return logReader.readNext(gen, txn, maxTxn)
                .thenApply(file -> {
                    metrics.readLagSec.set(unixTime() - file.getCreatedAt());
                    metrics.readFiles(file.getFiles());
                    metrics.readBytes(file.getBytes());
                    metrics.add(FileType.DUMPER_LOG, file.getBytes(), file.getFiles());
                    var tx = new DumperTx(clusterId, shardId, file.txn, file.createdAt);
                    // TODO: use same file (gordiychuk@)
                    return new DumperFile(tx, file.buffer);
                });
        });
    }

    private long unixTime() {
        return System.currentTimeMillis() / 1000L;
    }

    private CompletableFuture<Long> pullNextTxn() {
        if (latestWriteTxn.get() > latestReadTxn.get()) {
            return completedFuture(latestReadTxn.incrementAndGet());
        }

        var wait = waitTxnFuture.get();
        if (!wait.isDone()) {
            return failedFuture(Status.FAILED_PRECONDITION
                    .withDescription("Previous pulling at shard " + shardId + " not finished yet")
                    .asRuntimeException());
        }

        var future = new CompletableFuture<Long>();
        if (!waitTxnFuture.compareAndSet(wait, future)) {
            return failedFuture(Status.FAILED_PRECONDITION
                    .withDescription("Previous pulling at shard " + shardId + " not finished yet")
                    .asRuntimeException());
        }

        actor.schedule();
        return future.thenApplyAsync(txn -> txn, executor);
    }

    @Override
    public CompletableFuture<Void> commit(List<DumperTx> txList) {
        if (txList.isEmpty()) {
            return completedFuture(null);
        }

        if (!isSequentialDelete(txList)) {
            return failedFuture(new IllegalStateException(
                    "Non sequential tx commit, latest commit txn "
                            + latestDeleteTxn.get()
                            + " requested "
                            + txList));
        }

        var first = txList.get(0);
        var last = txList.get(txList.size() - 1);
        var size = txList.size();
        var range = NameRange.inclusiveExclusive(
            "c." + DumperLogFileName.format(first.txn),
            "c." + DumperLogFileName.format(last.txn + 1));
        return retry.loopUntilSuccess("commit(" + size + ")", (attempt) -> dao.deleteFiles(tableId, gen, List.of(range)))
            .whenComplete((ignore, e) -> {
                if (e == null) {
                    metrics.commitTxn(size);
                    latestDeleteTxn.addAndGet(size);
                    metrics.writeLagSec.set(unixTime() - last.createdAtSec);
                }
            });
    }

    @Nullable
    private boolean isSequentialDelete(List<DumperTx> txList) {
        long lastTxn = latestDeleteTxn.get();
        for (var tx : txList) {
            long expectTxn = lastTxn + 1;
            long txn = tx.txn;
            if (txn != expectTxn) {
                return false;
            }
            lastTxn = txn;
        }
        return true;
    }

    private CompletableFuture<FileNamesList> listFiles(NameRange range) {
        CompletableFuture<FileNamesList> doneFuture = new CompletableFuture<>();
        CompositeFileNameCollector collector = new CompositeFileNameCollector();
        listFiles(range, collector, doneFuture);
        return doneFuture;
    }

    private void listFiles(NameRange range, CompositeFileNameCollector collector, CompletableFuture<FileNamesList> doneFuture) {
        retry.loopUntilSuccess("listFiles(" + range + ")", (attempt) -> dao.listFiles(tableId, gen, range))
            .thenAccept(result -> {
                for (var entry : result.getEntries()) {
                    var stats = entry.toStats();
                    String name = stats.getName().substring("c.".length());
                    if (!collector.append(name, stats.getSize(), stats.getCreatedUnixtime())) {
                        logger.info("Unknown file {} at shard {}", stats, shardId);
                    }
                }

                if (result.isOverrun()) {
                    var nextRange = NameRange.rangeFrom(result.getOverrunLastEntryName(), false);
                    if (range.getEnd() != null) {
                        nextRange = nextRange.to(range.getEnd(), range.isEndInclusive());
                    }
                    listFiles(nextRange, collector, doneFuture);
                    return;
                }

                var list = collector.complete()
                        .stream()
                        .sorted(Comparator.comparingLong(FileName::getCreatedAt))
                        .collect(Collectors.collectingAndThen(Collectors.toList(), FileNamesList::new));
                doneFuture.complete(list);
            })
            .exceptionally(e -> {
                doneFuture.completeExceptionally(e);
                return null;
            });
    }

    private void error(Throwable e) {
        if (KvExceptionHandler.isGenerationChanged(e)) {
            stop();
        }

        if (errorLogLimiter.acquire()) {
            logger.error("ShardId:{}, TabletId:{} error ", shardId, tableId, e);
        }

        lastError = e;
        lastErrorInstant = System.currentTimeMillis();
        metrics.kvError();
        actor.schedule();
    }

    public void awaitAct() throws InterruptedException {
        var sync = new CountDownLatch(1);
        actSync = sync;
        actor.schedule();
        sync.await();
    }

    @Override
    public boolean isStop() {
        return stop || retry.closed;
    }

    @Override
    public String toString() {
        return "KvShortTermStorageReader{" +
            "shardId=" + shardId +
            ", tableId=" + tableId +
            '}';
    }

    @Override
    public void stop() {
        stop = true;
        retry.close();
        logReader.close();
        actor.schedule();
    }

    @Override
    public void close() {
        stop();
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
        memory.addMemory("dumper.shard.KvLogReader", logReader.memorySizeIncludingSelf());
    }
}
