package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.dumper.DumperShardId;

/**
 * @author Vladimir Gordiychuk
 */
public class KvTabletsMapping implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(KvTabletsMapping.class);

    private final String kvVolumePath;
    private final KikimrKvClient kvClient;
    private final ExecutorService executor;
    private final ScheduledExecutorService timer;
    @Nullable
    private volatile ScheduledFuture<?> scheduled;
    private final CompletableFuture<?> ready = new CompletableFuture<>();

    private long[] tabletIdByShardId;

    public KvTabletsMapping(
        String kvVolumePath,
        KikimrKvClient kvClient,
        ExecutorService executor,
        ScheduledExecutorService timer)
    {
        this.kvVolumePath = kvVolumePath;
        this.kvClient = kvClient;
        this.executor = executor;
        this.timer = timer;
        this.executor.execute(this::init);
    }

    private void init() {
        CompletableFutures.safeCall(() -> kvClient.createKvTablets(kvVolumePath, DumperShardId.SHARD_COUNT))
            .thenCompose(ignore -> resolveKvTablets())
            .thenAccept(this::initComplete)
            .whenComplete((tabletIds, e) -> {
                if (e != null) {
                    logger.warn("cannot resolve KV tablets: " + kvVolumePath, e);
                    scheduled = timer.schedule(() -> executor.execute(this::init), 5, TimeUnit.SECONDS);
                }
            });
    }

    private CompletableFuture<long[]> resolveKvTablets() {
        return kvClient.resolveKvTablets(kvVolumePath)
            .thenApply(tabletIds -> {
                logger.info("{} tablets resolved with SchemeShard", tabletIds.length);
                return tabletIds;
            });
    }

    private void initComplete(long[] tabletIds) {
        var uniqueTabletIds = new LongOpenHashSet(tabletIds);
        if (uniqueTabletIds.size() != tabletIds.length) {
            throw new IllegalStateException("non-unique tablet ids");
        }

        if (Arrays.stream(tabletIds).anyMatch(l -> l == 0)) {
            throw new IllegalStateException("zero tablet id is invalid");
        }

        if (tabletIds.length != DumperShardId.SHARD_COUNT) {
            throw new IllegalStateException("Resolved by path "+ kvVolumePath + " tables " + tabletIds.length + " but expected " + DumperShardId.SHARD_COUNT);
        }

        tabletIdByShardId = tabletIds;
        ready.complete(null);
    }

    public CompletableFuture<?> getReadyFuture() {
        return ready;
    }

    public long getTabletId(int shardId) {
        return tabletIdByShardId[(shardId - DumperShardId.FIRST_SHARD_ID)];
    }

    @Override
    public void close() {
        var copy = scheduled;
        if (copy != null) {
            copy.cancel(false);
        }
    }
}
