package ru.yandex.solomon.dumper.storage.shortterm;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.Unpooled;

import ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
class TxDumperFile implements MemMeasurable, AutoCloseable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(TxDumperFile.class);

    final long txn;
    final ByteBuf buffer;
    long createdAt;
    int chunksCount;

    public TxDumperFile(long txn, ByteBuf buffer, long createdAt) {
        this.txn = txn;
        this.buffer = buffer;
        this.createdAt = createdAt;
        this.chunksCount = 1;
    }

    static TxDumperFile of(String fileName, byte[] bytes, long createdAt) {
        int chunkNo = DumperLogFileName.chunkNo(fileName);
        if (chunkNo != 0) {
            throw new IllegalStateException("Corrupted file " + fileName + " expected chunkNo 0, but was " + chunkNo);
        }
        long txn = DumperLogFileName.txn(fileName);
        if (DumperLogFileName.isLastChunk(fileName)) {
            return new TxDumperFile(txn, Unpooled.wrappedBuffer(bytes), createdAt);
        }

        var buffer = ByteBufAllocator.DEFAULT.buffer(bytes.length);
        buffer.writeBytes(bytes);
        return new TxDumperFile(txn, buffer, createdAt);
    }

    public void append(byte[] bytes, long createdAt) {
        this.buffer.writeBytes(bytes);
        this.createdAt = Math.max(this.createdAt, createdAt);
        this.chunksCount++;
    }

    public ByteBuf getBuffer() {
        return buffer;
    }

    public long getCreatedAt() {
        return createdAt;
    }

    public int getFiles() {
        return chunksCount;
    }

    public int getBytes() {
        return buffer.readableBytes();
    }

    @Override
    public void close() {
        buffer.release();
    }

    @Override
    public long memorySizeIncludingSelf() {
        return SELF_SIZE + MemoryCounter.byteBufSize(buffer);
    }
}
