package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName;
import ru.yandex.solomon.memory.layout.MemMeasurable;

/**
 * @author Vladimir Gordiychuk
 */
class TxDumperFileList implements MemMeasurable, AutoCloseable {
    public final List<TxDumperFile> files;
    public long memoryUsage;
    @Nullable
    public TxDumperFile lastNotFinished;
    private long expectTxn;

    public TxDumperFileList(int capacity, long expectTxn) {
        this.files = new ArrayList<>(capacity);
        this.expectTxn = expectTxn;
    }

    public void append(String fileName, byte[] bytes, long createdAt) {
        DumperLogFileName.ensureFullNameValid(fileName);
        if (lastNotFinished == null) {
            lastNotFinished = TxDumperFile.of(fileName, bytes, createdAt);
            ensureTxnSequential(lastNotFinished.txn);
            ensureFinished(lastNotFinished, fileName);
            return;
        }

        long txn = DumperLogFileName.txn(fileName);
        if (lastNotFinished.txn != txn) {
            throw new IllegalStateException("Tx " + txn + " not completed, expected chunkNo " + lastNotFinished.chunksCount);
        }

        ensureChunkSequential(lastNotFinished, fileName);
        lastNotFinished.append(bytes, createdAt);
        ensureFinished(lastNotFinished, fileName);
    }

    public void ensureLatestFinished() {
        if (lastNotFinished != null) {
            throw new IllegalStateException("Tx " + lastNotFinished.txn + " not completed, chunks " + lastNotFinished.chunksCount);
        }
    }

    public void ensureNotEmpty() {
        if (files.isEmpty()) {
            throw new IllegalStateException("Expected at least one tx " + expectTxn + " was loaded, but not");
        }
    }

    private void ensureFinished(@Nonnull TxDumperFile file, String latestChunkFileName) {
        if (DumperLogFileName.isLastChunk(latestChunkFileName)) {
            files.add(file);
            expectTxn++;
            memoryUsage += file.memorySizeIncludingSelf();
            lastNotFinished = null;
        }
    }

    private void ensureTxnSequential(long txn) {
        if (expectTxn != txn) {
            throw new IllegalStateException("Txn not sequential, expected " + expectTxn + " but was " + txn);
        }
    }

    private void ensureChunkSequential(TxDumperFile file, String latestChunkFileName) {
        int chunkNo = DumperLogFileName.chunkNo(latestChunkFileName);
        if (file.chunksCount != chunkNo) {
            throw new IllegalStateException("Txn " + file.txn + " have not sequential chunks, expect " + file.chunksCount + " but was " + chunkNo);

        }
    }

    @Override
    public void close() {
        files.forEach(TxDumperFile::close);
        if (lastNotFinished != null) {
            lastNotFinished.close();
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = memoryUsage;
        if (lastNotFinished != null) {
            size += lastNotFinished.memorySizeIncludingSelf();
        }
        return size;
    }
}
