package ru.yandex.solomon.dumper.storage.shortterm.file;

import java.time.Instant;
import java.util.regex.Pattern;

import ru.yandex.misc.dataSize.DataSize;

/**
 * @author Vladimir Gordiychuk
 */
public class DumperLogFileName extends FileName {
    private static final int CHUNK_NO_LENGTH = 6;
    private static final int CHUNK_NO_OFFSET = CHUNK_NO_LENGTH + 1;
    private static final int TXN_LENGTH = 17;
    private static final int TXN_OFFSET = CHUNK_NO_OFFSET + TXN_LENGTH + 1;
    public static final Pattern PATTERN = Pattern.compile("^(?<txn>[0-9]{17})\\.(?<chunkNo>[0-9]{6})(?<last>[yz])$");
    public static final int SUFFIX_LENGTH = TXN_LENGTH + 1 /* dot */ + CHUNK_NO_LENGTH + 1 /* last */;
    private final long txn;

    public DumperLogFileName(long txn, int fileCount, int bytesSize, long createdAt, boolean valid) {
        super(fileCount, bytesSize, createdAt, valid);
        this.txn = txn;
    }

    public long getTxn() {
        return txn;
    }

    @Override
    public FileType getType() {
        return FileType.DUMPER_LOG;
    }

    public static String format(long txn) {
        return String.format("%s%017d", FileType.DUMPER_LOG.getPrefix(), txn);
    }

    public static String format(long txn, int chunkNo, boolean last) {
        return String.format("%s%017d.%06d%s", FileType.DUMPER_LOG.getPrefix(), txn, chunkNo, last ? 'z' : "y");
    }

    public static boolean isValidFullName(String fileName) {
        return fileName.length() < SUFFIX_LENGTH || !PATTERN.matcher(fileName).matches();
    }

    public static void ensureFullNameValid(String fileName) {
        if (isValidFullName(fileName)) {
            throw new IllegalArgumentException(fileName + " not satisfy pattern " + PATTERN);
        }
    }

    public static boolean isLastChunk(String fileName) {
        return fileName.endsWith("z");
    }

    public static String withoutChunkNo(String fileName) {
        return fileName.substring(0, fileName.length() - CHUNK_NO_OFFSET - 1);
    }

    public static int chunkNo(String fileName) {
        int from = fileName.length() - CHUNK_NO_OFFSET;
        String chunkNo = fileName.substring(from, from + CHUNK_NO_LENGTH);
        return Integer.parseInt(chunkNo);
    }

    public static long txn(String fileName) {
        int from = fileName.length() - TXN_OFFSET;
        String txn = fileName.substring(from, from + TXN_LENGTH);
        return Long.parseLong(txn);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof DumperLogFileName)) return false;
        if (!super.equals(o)) return false;

        DumperLogFileName that = (DumperLogFileName) o;
        return txn == that.txn;
    }

    @Override
    public int hashCode() {
        int result = super.hashCode();
        result = 31 * result + (int) (txn ^ (txn >>> 32));
        return result;
    }

    @Override
    public String toString() {
        return "DumperLogFileName{" +
            "txn=" + txn +
            ", fileCount=" + getFileCount() +
            ", bytesSize=" + DataSize.shortString(getBytesSize()) +
            ", createdAt=" + Instant.ofEpochSecond(getCreatedAt()) +
            ", valid=" + isValid() +
            '}';
    }
}
