package ru.yandex.solomon.dumper.storage.shortterm.file;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * @author Vladimir Gordiychuk
 */
public class MemstoreSnapshotFileNameCollector implements FileNameCollector<MemstoreSnapshotFileName> {
    private static final Pattern PATTERN = Pattern.compile("^(?<node>[0-9]{3})\\.(?<txn>[0-9]{17})\\.(?<chunkNo>[0-9]{6})(?<last>[yz])$");

    private final List<MemstoreSnapshotFileName> result = new ArrayList<>();

    private String nodeId;
    private String txn;
    private int chunkNo;
    private boolean last;
    private boolean valid;

    private long createdAt;
    private int bytesSize;

    public MemstoreSnapshotFileNameCollector() {
        clean();
    }

    @Override
    public boolean append(String name, int bytesSize, long createdAt) {
        var matcher = PATTERN.matcher(name);
        if (!matcher.matches()) {
            return false;
        }

        String nodeId = matcher.group(1);
        String txn = matcher.group(2);
        int chunkNo = Integer.parseInt(matcher.group(3));
        boolean last = "z".equals(matcher.group(4));

        if (!nodeId.equals(this.nodeId) || !txn.equals(this.txn) || this.last) {
            flush();
            this.nodeId = nodeId;
            this.txn = txn;
        }

        if (this.chunkNo + 1 != chunkNo) {
            valid = false;
        }
        this.chunkNo = chunkNo;
        this.bytesSize += bytesSize;
        this.createdAt = Math.min(this.createdAt, createdAt);
        this.last = last;
        return true;
    }

    private void flush() {
        if (chunkNo == -1) {
            return;
        }

        try {
            int nodeId = Integer.parseInt(this.nodeId);
            long txn = Long.parseLong(this.txn);
            valid &= last;
            result.add(new MemstoreSnapshotFileName(nodeId, txn, chunkNo + 1, bytesSize, createdAt, valid));
        } finally {
            clean();
        }
    }

    private void clean() {
        chunkNo = -1;
        nodeId = "";
        txn = "";
        valid = true;
        createdAt = Long.MAX_VALUE;
        bytesSize = 0;
    }

    @Override
    public List<MemstoreSnapshotFileName> complete() {
        flush();
        return result;
    }
}
