package ru.yandex.solomon.dumper.www;

import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.dumper.DumperLocalShards;
import ru.yandex.solomon.dumper.DumperShard;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
@Import({
    ManagerWriterContext.class,
})
public class DumperLocalShardsWww {
    @Autowired
    private HttpAuthenticator authenticator;
    @Autowired
    private InternalAuthorizer authorizer;
    @Autowired
    private ManagerWriterContext contect;
    @Autowired
    private DumperLocalShards shards;

    @Bean
    public RootLink dumperLocalShardLinks() {
        return new RootLink("/local-shards", "Dumper local shards");
    }

    @RequestMapping(value = "/local-shards", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> stockpileLocalShards(
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authSubject -> authorizer.authorize(authSubject))
            .thenApply(account -> localShardsImpl(sortBy));
    }

    private String localShardsImpl(int sortBy) {
        List<Record> records = this.shards.stream()
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.shardId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Dumper local shards", contect, columns, records, sortBy).genString();
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
            Column.of(
                "ShardId",
                r -> {
                    String shardId = Integer.toString(r.shardId);
                    return new AHref(namedObjectLink(new NamedObjectId(DumperShard.class, shardId)), shardId);
                },
                Comparator.comparingInt(o -> o.shardId)),
            Column.of("Uptime",
                r -> DurationUtils.formatDurationMillis(r.uptimeMillis),
                Comparator.comparingLong(o2 -> o2.uptimeMillis)
            ),
            Column.of("Read lag",
                r -> DurationUtils.formatDurationSeconds(r.readLagSec),
                Comparator.comparingLong(o2 -> o2.readLagSec)
            ),
            Column.of("Write lag",
                r -> DurationUtils.formatDurationSeconds(r.writeLagSec),
                Comparator.comparingLong(o2 -> o2.writeLagSec)
            ),
            Column.of("Pending txn",
                r -> DataSize.shortString(r.pendingTxn),
                Comparator.comparingLong(o2 -> o2.pendingTxn)
            ),
            Column.of("Process count",
                r -> DataSize.shortString(r.processCount),
                Comparator.comparingDouble(o2 -> o2.processCount)
            ),
            Column.of("Cpu",
                r -> String.format("%.5f", r.cpu),
                Comparator.comparingDouble(o2 -> o2.cpu)
            ),
            Column.of("Memory",
                r -> DataSize.shortString(r.memory),
                Comparator.comparingLong(o2 -> o2.memory)
            ),
            Column.of("Errors",
                r -> DataSize.shortString(r.errors),
                Comparator.comparingLong(o2 -> o2.errors)
            ),
            Column.of("Last Error",
                r -> {
                  if (r.lastErrorInstant == 0) {
                      return "none";
                  } else {
                      return Instant.ofEpochMilli(r.lastErrorInstant);
                  }
                },
                Comparator.comparingLong(o2 -> o2.lastErrorInstant)
            )
        );
    }

    private static class Record implements TableRecord {
        int shardId;
        long readLagSec;
        long writeLagSec;
        long pendingTxn;
        long uptimeMillis;
        double cpu;
        long memory;
        int processCount;
        long errors;
        long lastErrorInstant;

        public static Record of(DumperShard shard) {
            var r = new Record();
            r.shardId = shard.getId();
            var metrics = shard.metrics();
            r.readLagSec = metrics.readLagSec.get();
            r.writeLagSec = metrics.writeLagSec.get();
            r.pendingTxn = metrics.pendingTxn.get();
            r.uptimeMillis = System.currentTimeMillis() - metrics.createdAtMs;
            r.cpu = metrics.cpuTimeNanos.getRate(TimeUnit.SECONDS) * 1e-6;
            r.memory = shard.getMemoryUsage();
            r.processCount = shard.processes().size();
            r.errors = metrics.kvErrors.get();
            r.lastErrorInstant = metrics.lastErrorInstant;
            return r;
        }
    }
}
