package ru.yandex.solomon.dumper.www;

import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.core.exceptions.NotFoundException;
import ru.yandex.solomon.dumper.DumperLocalShards;
import ru.yandex.solomon.dumper.DumperShard;
import ru.yandex.solomon.dumper.DumperShardId;
import ru.yandex.solomon.dumper.SolomonShardProcess;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.find.annotation.NamedObjectFinderAnnotation;
import ru.yandex.solomon.util.host.HostUtils;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
public class DumperObjectFinder {
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final DumperLocalShards shards;
    private final int port;

    @Autowired
    public DumperObjectFinder(HttpAuthenticator authenticator, InternalAuthorizer authorizer, DumperLocalShards shards, HttpServerConfig config) {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.shards = shards;
        this.port = config.getPort();
    }

    @RequestMapping(path = "/dumper/shards/{shardId}", method = RequestMethod.GET)
    public CompletableFuture<ResponseEntity<String>> redirectToDumperShard(
        @PathVariable("shardId") String shardIdStr,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> redirectToDumperShardImpl(shardIdStr));
    }

    @RequestMapping(path = "/solomon/shards/{numId}", method = RequestMethod.GET)
    public CompletableFuture<ResponseEntity<String>> redirectToSolomonShard(
        @PathVariable("numId") String numId,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> redirectToSolomonShardImpl(numId));
    }

    @Nullable
    @NamedObjectFinderAnnotation
    public DumperShard getShardById(int shardId) {
        return shards.getShardById(shardId);
    }

    @Nullable
    @NamedObjectFinderAnnotation
    public SolomonShardProcess getProcessByNumId(String numIdStr) {
        int numId = Integer.parseUnsignedInt(numIdStr);
        int shardId = shardId(numId);
        var shard = shards.getShardById(shardId);
        if (shard == null) {
            throw new NotFoundException("Shard " + shardId + " not found");
        }
        return shard.processes().get(numId);
    }

    private int shardId(int numId) {
        return ((DumperShardId.SHARD_COUNT - 1) & hash(numId)) + 1;
    }

    private int hash(int numId) {
        int h = Integer.hashCode(numId);
        return h ^ (h >>> 16);
    }

    private ResponseEntity<String> redirectToDumperShardImpl(String shardIdStr) {
        String linkToShard = namedObjectLink(new NamedObjectId(DumperShard.class, shardIdStr));
        String url = "http://" + HostUtils.getFqdn() + ":" + port + "" + linkToShard;
        return ResponseEntity.status(HttpStatus.FOUND)
            .header("Location", url)
            .body(null);
    }

    private ResponseEntity<String> redirectToSolomonShardImpl(String numIdStr) {
        String linkToShard = namedObjectLink(new NamedObjectId(SolomonShardProcess.class, numIdStr));
        String url = "http://" + HostUtils.getFqdn() + ":" + port + "" + linkToShard;
        return ResponseEntity.status(HttpStatus.FOUND)
            .header("Location", url)
            .body(null);
    }
}
