package ru.yandex.solomon.dumper;

import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

/**
 * @author Vladimir Gordiychuk
 */
public class DumperLocalShardsTest {

    private DumperLocalShards shards;
    private ScheduledExecutorService timer;

    @Before
    public void setUp() {
        shards = new DumperLocalShards();
        timer = Executors.newSingleThreadScheduledExecutor();
    }

    @After
    public void tearDown() throws Exception {
        timer.shutdownNow();
    }

    @Test
    public void getNoExist() {
        for (int shardId = DumperShardId.FIRST_SHARD_ID; shardId <= DumperShardId.LAST_SHARD_ID; shardId++) {
            assertNull(DumperShardId.toString(shardId), shards.getShardById(shardId));
        }
    }

    @Test
    public void emptyStream() {
        var result = shards.stream().collect(Collectors.toList());
        assertEquals(List.of(), result);
    }

    @Test
    public void emptyIterator() {
        var it = shards.iterator();
        assertFalse(it.hasNext());

        try {
            var shard = it.next();
            fail("next: " + shard);
        } catch (NoSuchElementException e) {
            // ok
        }
    }

    @Test
    public void addShard() {
        var shard = newShard(1);
        assertTrue(shards.addShard(shard));
        assertFalse(shards.addShard(shard));

        assertSame(shard, shards.getShardById(1));

        {
            var it = shards.iterator();
            assertTrue(it.hasNext());
            assertSame(shard, it.next());
            assertFalse(it.hasNext());
        }

        {
            var list = shards.stream().collect(Collectors.toList());
            assertEquals(1, list.size());
            assertSame(shard, list.get(0));
        }
    }

    @Test
    public void addMultipleShard() {
        var one = newShard(1);
        var two = newShard(2);

        assertTrue(shards.addShard(one));
        assertTrue(shards.addShard(two));

        assertSame(one, shards.getShardById(1));
        assertSame(two, shards.getShardById(2));

        {
            var list = Lists.newArrayList(shards.iterator());
            assertEquals(List.of(one, two), list);
        }

        {
            var list = shards.stream().collect(Collectors.toList());
            assertEquals(List.of(one, two), list);
        }
    }

    @Test
    public void removeShard() {
        var one = newShard(1);
        var two = newShard(2);

        assertTrue(shards.addShard(one));
        assertTrue(shards.addShard(two));

        assertTrue(shards.remove(one));
        assertFalse(shards.remove(one));

        assertNull(shards.getShardById(1));
        assertSame(two, shards.getShardById(2));
    }

    @Test
    public void gracefulShutdown() {
        var one = newShard(1);
        var two = newShard(2);

        assertTrue(shards.addShard(one));
        assertTrue(shards.addShard(two));
        assertFalse(one.isStop());
        assertFalse(two.isStop());

        shards.gracefulShutdownSync();

        assertTrue(one.isStop());
        assertTrue(two.isStop());
    }

    public DumperShard newShard(int id) {
        ShortTermStorageStub reader = new ShortTermStorageStub();
        LongTermStorageStub lts = new LongTermStorageStub(timer);
        var metrics = new DumperShardMetrics(""+id, new DumperShardMetricsAggregated());
        var executor = ForkJoinPool.commonPool();
        var optsProvider = new SolomonShardOptsProviderStub();
        return new DumperShard(id, executor, executor, timer, metrics, optsProvider, reader, lts);
    }
}
