package ru.yandex.solomon.dumper;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.IntStream;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.dumper.LogHelper.Batch;
import ru.yandex.solomon.dumper.LogHelper.Metric;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.slog.LogsIndex;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.dumper.LogHelper.concat;
import static ru.yandex.solomon.dumper.LogHelper.prepareLogBatch;

/**
 * @author Vladimir Gordiychuk
 */
public class DumperShardTest {

    private LongTermStorageStub longStorage;
    private ShortTermStorageStub shortStorage;
    private ScheduledExecutorService timer;
    private SolomonShardOptsProviderStub optsProvider;
    private DumperShard shard;

    @Before
    public void setUp() {
        timer = Executors.newSingleThreadScheduledExecutor();
        shortStorage = new ShortTermStorageStub();
        longStorage = new LongTermStorageStub(timer);
        optsProvider = new SolomonShardOptsProviderStub();
        var metrics = new DumperShardMetrics("42", new DumperShardMetricsAggregated());
        var executer = ForkJoinPool.commonPool();
        shard = new DumperShard(
                42,
                executer,
                executer,
                timer,
                metrics,
                optsProvider,
                shortStorage,
                longStorage);
        shard.start();
    }

    @After
    public void tearDown() {
        longStorage.close();
        timer.shutdownNow();
        shard.stop();
        shortStorage.stop();
    }

    @Test
    public void sequentialWrite() {
        var alice = metric("alice");
        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            write(prepareLogBatch(alice)).join();
            assertWrite(alice);
        }
    }

    @Test
    public void sequenceOrderSave() {
        var ts0 = System.currentTimeMillis();
        var alice = metric("alice");
        IntStream.range(0, 20)
            .mapToObj(idx -> prepareLogBatch(alice.add(ts0, idx)))
            .map(this::write)
            .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid))
            .join();

        assertWrite(alice);
    }

    @Test
    public void processDifferentShard() {
        var alice1 = metric(nextNumId(), "alice").addNext(2);
        var alice2 = metric(nextNumId(), "alice").addNext(42);

        for (int idx = 0; idx < 3; idx++) {
            alice1.addNext(idx);
            alice2.addNext(idx);
            write(prepareLogBatch(alice1, alice2)).join();
            assertWrite(alice1);
            assertWrite(alice2);
        }
    }

    @Test
    public void processShardManyNewAndAllKnown() {
        var smallNumId = nextNumId();
        var hugeNumId = nextNumId();

        var alice = metric(smallNumId, "alice");
        var metrics = new ArrayList<Metric>();
        List<Batch> batches = new ArrayList<>();

        for (int index = 0; index < 100; index++) {
            alice.addNext(index);
            var metric = metric(hugeNumId, "idx" + index).addNext(index);
            metrics.add(metric);
            int finalIndex = index;
            metrics.forEach(s -> s.addNext(finalIndex));
            batches.add(prepareLogBatch(alice));
            batches.add(prepareLogBatch(metrics));
        }

        batches.stream()
            .map(this::write)
            .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid))
            .join();

        assertWrite(alice);
        for (int index = 0; index < 10; index++) {
            var metric = metrics.get(ThreadLocalRandom.current().nextInt(metrics.size()));
            assertWrite(metric);
        }
    }

    @Test
    public void processShardManyNewAndAllKnownCombine() {
        var smallNumId = nextNumId();
        var hugeNumId = nextNumId();

        var alice = metric(smallNumId, "alice");
        var metrics = new ArrayList<Metric>();
        List<Batch> batches = new ArrayList<>();

        for (int index = 0; index < 100; index++) {
            alice.addNext(index);
            var metric = metric(hugeNumId, "idx" + index).addNext(index);
            metrics.add(metric);
            int finalIndex = index;
            metrics.forEach(s -> s.addNext(finalIndex));
            batches.add(prepareLogBatch(concat(metrics, List.of(alice))));
        }

        batches.stream()
            .map(this::write)
            .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid))
            .join();

        assertWrite(alice);
        for (int index = 0; index < 10; index++) {
            var metric = metrics.get(ThreadLocalRandom.current().nextInt(metrics.size()));
            assertWrite(metric);
        }
    }

    @Test
    public void stopReader() {
        var alice = metric("alice");
        alice.addNext(42);
        assertFalse(shard.isStop());
        assertFalse(shortStorage.isStop());

        var stopFuture = shard.stop();
        write(prepareLogBatch(alice));

        stopFuture.join();

        assertTrue(shard.isStop());
        assertTrue(shortStorage.isStop());
    }

    @Test(timeout = 5_000)
    public void stopWhenShortTermStorageStopped() throws InterruptedException {
        var alice = metric("alice");
        alice.addNext(42);
        assertFalse(shard.isStop());
        shortStorage.stop();
        assertTrue(shortStorage.isStop());

        shard.scheduleAct();
        shard.stopFuture().join();
        assertTrue(shard.isStop());
    }

    @Test
    public void commitFailedBatchIndexNotValid() {
        Batch batch = new Batch(randomBuffer(), 32, new LogsIndex(1));
        write(batch).join();
        assertEquals(0, batch.content.refCnt());
    }

    @Test
    public void commitFailedBatchContentNotValid() {
        var alice = metric("alice");
        alice.addNext(42);
        var bob = metric("bob");
        for (int index = 0; index < 100; index++) {
            bob.addNext(index);
        }
        var batch = prepareLogBatch(alice, bob);
        batch.content.setBytes(batch.getMetaOffset(1), randomBytes(20));
        assertEquals(1, batch.content.refCnt());
        write(batch).join();
        assertEquals(0, batch.content.refCnt());
        assertWrite(alice);
    }

    private ByteBuf randomBuffer() {
        return ByteBufAllocator.DEFAULT.buffer().writeBytes(randomBytes());
    }

    private byte[] randomBytes() {
        var random = ThreadLocalRandom.current();
        var result = new byte[random.nextInt(1, 1024)];
        random.nextBytes(result);
        return result;
    }

    private byte[] randomBytes(int size) {
        var random = ThreadLocalRandom.current();
        var result = new byte[size];
        random.nextBytes(result);
        return result;
    }

    private void assertWrite(Metric metric) {
        var actual = AggrGraphDataArrayList.of(longStorage.read(metric.numId, metric.labels));
        assertEquals(metric.toString(), metric.expected(), actual);
    }

    private Metric metric(String name) {
        return LogHelper.metric(nextNumId(), name);
    }

    private Metric metric(int numId, String name) {
        return LogHelper.metric(nextNumId(), name);
    }

    private CompletableFuture<Void> write(Batch batch) {
        return shortStorage.enqueue(batch.content);
    }

    private int nextNumId() {
        return ThreadLocalRandom.current().nextInt();
    }
}
