package ru.yandex.solomon.dumper;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;

import com.google.common.collect.Lists;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.LogsIndex;
import ru.yandex.solomon.slog.LogsIndexSerializer;
import ru.yandex.solomon.slog.SnapshotLogDataBuilderImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaBuilderImpl;

/**
 * @author Vladimir Gordiychuk
 */
public class LogHelper {
    private static final CompressionAlg COMPRESSION = CompressionAlg.values()[ThreadLocalRandom.current().nextInt(CompressionAlg.values().length)];

    public static Log prepareLog(int numId, Metric... metrics) {
        return prepareLog(numId, Arrays.asList(metrics));
    }

    public static Log prepareLog(int numId, List<Metric> metrics) {
        var buffer = ByteBufAllocator.DEFAULT;
        try (var metaBuilder = new UnresolvedLogMetaBuilderImpl(numId, COMPRESSION, buffer);
             var dataBuilder = new SnapshotLogDataBuilderImpl(COMPRESSION, numId, buffer))
        {
            metaBuilder.onCommonLabels(Labels.of());
            for (var metric : metrics) {
                var chunk = metric.flush();
                int size = dataBuilder.onTimeSeries(chunk);
                metaBuilder.onMetric(MetricType.DGAUGE, metric.labels, chunk.getRecordCount(), size);
            }

            var meta = metaBuilder.build();
            var data = dataBuilder.build();
            return new Log(numId, meta, data);
        }
    }

    public static Log copy(Log log) {
        return new Log(log.numId, log.meta.copy(), log.data.copy());
    }

    public static Batch prepareLogBatch(Metric... metrics) {
        return prepareLogBatch(Arrays.asList(metrics));
    }

    public static List<Metric> concat(List<Metric> left, List<Metric> right) {
        var result = new ArrayList<Metric>(left.size() + right.size());
        result.addAll(left);
        result.addAll(right);
        return result;
    }

    public static Batch prepareLogBatch(List<Metric> metrics) {
        var byNumId = new LinkedHashMap<Integer, List<Metric>>();
        for (var metric : metrics) {
            byNumId.computeIfAbsent(metric.numId, ignore -> new ArrayList<>()).add(metric);
        }

        var random = ThreadLocalRandom.current();
        var logs = new ArrayList<Log>();
        for (var entry : byNumId.entrySet()) {
            int numId = entry.getKey();
            var numIdMetrics = entry.getValue();
            int partition = Math.max(1, random.nextInt(0, entry.getValue().size()));
            for (var part : Lists.partition(numIdMetrics, partition)) {
                logs.add(prepareLog(numId, part));
            }
        }

        LogsIndex index = new LogsIndex(metrics.size());
        for (var log : logs) {
            index.add(log.numId, log.meta.readableBytes(), log.data.readableBytes());
        }

        var buffer = ByteBufAllocator.DEFAULT.buffer();
        try {
            LogsIndexSerializer.serialize(buffer, index);
            int indexSize = buffer.readableBytes();
            for (var log : logs) {
                try (log) {
                    buffer.writeBytes(log.meta);
                    buffer.writeBytes(log.data);
                }
            }
            return new Batch(buffer, indexSize, index);
        } catch (Throwable e) {
            buffer.release();
            throw new RuntimeException(e);
        }
    }

    public static Metric metric(int numId, String name) {
        return new Metric(numId, Labels.of("name", name));
    }

    public static class Metric {
        public final int numId;
        public final Labels labels;
        private final AggrGraphDataArrayList timeseries = new AggrGraphDataArrayList();
        private int flushedIdx;

        public Metric(int numId, Labels labels) {
            this.numId = numId;
            this.labels = labels;
        }

        public Metric addNext(double value) {
            final long lastTs;
            if (timeseries.isEmpty()) {
                lastTs = Instant.now().truncatedTo(ChronoUnit.SECONDS).toEpochMilli();
            } else {
                lastTs = timeseries.getTsMillis(timeseries.length() - 1);
            }

            return add(lastTs + 10_000, value);
        }

        public Metric add(long tsMillis, double value) {
            var point = RecyclableAggrPoint.newInstance();
            try {
                 point.setTsMillis(tsMillis);
                 point.setValue(value);
                 point.setStepMillis(10_000);
                 timeseries.addRecord(point);
            } finally {
                point.recycle();
            }
            return this;
        }

        public MetricArchiveImmutable flush() {
            var chunk = MetricArchiveImmutable.of(timeseries.slice(flushedIdx, timeseries.length()));
            flushedIdx = timeseries.length();
            return chunk;
        }

        public AggrGraphDataArrayList expected() {
            AggrGraphDataArrayList copy = AggrGraphDataArrayList.of(timeseries);
            copy.sortAndMerge();
            return copy;
        }

        @Override
        public String toString() {
            return "Metric{" +
                "numId=" + numId +
                ", labels=" + labels +
                '}';
        }
    }

    public static class Batch {
        public final ByteBuf content;
        public final int indexSize;
        public final LogsIndex index;

        public Batch(ByteBuf content, int indexSize, LogsIndex index) {
            this.content = content;
            this.indexSize = indexSize;
            this.index = index;
        }

        public final int getMetaOffset(int i) {
            int offset = indexSize;
            for (int idx = 0; idx < i; idx++) {
                offset += index.getMetaSize(idx) + index.getDataSize(idx);
            }
            return offset;
        }
    }
}
