package ru.yandex.solomon.dumper;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.WillNotClose;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsRequest;
import ru.yandex.solomon.metabase.api.protobuf.TResolveLogsResponse;
import ru.yandex.solomon.metrics.client.MetabaseClientException;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.ResolvedLogMetricsBuilderImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaRecord;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

/**
 * @author Vladimir Gordiychuk
 */
public class LongTermStorageStub implements LongTermStorage, AutoCloseable {
    public StockpileClientStub stockpile;
    private final StockpileWriterImpl writer;
    private final ConcurrentMap<Key, MetricImpl> metrics = new ConcurrentHashMap<>();
    private volatile MetabaseStatus metabaseStatus = MetabaseStatus.OK;
    private volatile boolean disableNewMetrics = false;
    public volatile Supplier<CompletableFuture<?>> beforeSupplier;

    public LongTermStorageStub(ScheduledExecutorService timer) {
        var executor = ForkJoinPool.commonPool();
        this.stockpile = new StockpileClientStub(executor);
        this.writer = new StockpileWriterImpl(stockpile, executor, timer, new MetricRegistry());
    }

    @Override
    public boolean isAbleToCreateNewMetrics(int numId) {
        return !disableNewMetrics;
    }

    @Override
    public CompletableFuture<TResolveLogsResponse> resolve(TResolveLogsRequest request) {
        return before(() -> {
            if (!metabaseStatus.isOk()) {
                throw new MetabaseClientException(metabaseStatus);
            }

            List<ByteString> resolved = new ArrayList<>(request.getUnresolvedLogMetaCount());
            for (var meta : request.getUnresolvedLogMetaList()) {
                resolved.add(resolve(request.getNumId(), ByteStrings.toByteBuf(meta)));
            }
            return TResolveLogsResponse.newBuilder()
                .setStatus(disableNewMetrics ? EMetabaseStatusCode.QUOTA_ERROR : EMetabaseStatusCode.OK)
                .addAllResolvedLogMetrics(resolved)
                .build();
        });
    }

    private ByteString resolve(int numId, @WillNotClose ByteBuf unresolvedMeta) {
        int shardId = stockpile.randomShardId();
        try (var result = new ResolvedLogMetricsBuilderImpl(numId, CompressionAlg.LZ4, ByteBufAllocator.DEFAULT);
             var it = new UnresolvedLogMetaIteratorImpl(unresolvedMeta))
        {
            var record = new UnresolvedLogMetaRecord();
            while (it.next(record)) {
                var metric = createOrResolve(key(numId, record.labels), record.type, shardId);
                if (metric != null) {
                    result.onMetric(metric.getType(), metric.getLabels(), metric.getShardId(), metric.getLocalId());
                }
            }

            var buffer = result.build();
            try {
                return UnsafeByteOperations.unsafeWrap(ByteBufUtil.getBytes(buffer));
            } finally {
                buffer.release();
            }
        }
    }

    private static Key key(int numId, Labels labels) {
        return new Key(numId, labels);
    }

    @Nullable
    private MetricImpl createOrResolve(Key key, MetricType type, int shardId) {
        MetricImpl update;
        while (true) {
            var prev = metrics.get(key);
            if (prev == null) {
                if (disableNewMetrics) {
                    return null;
                }

                update = new MetricImpl(key, type, shardId, StockpileLocalId.random());
                prev = metrics.putIfAbsent(key, update);
                if (prev == null) {
                    return update;
                }
            }

            if (prev.getType() == type) {
                return prev;
            } else {
                update = new MetricImpl(key, type, prev.getShardId(), prev.getLocalId());
                if (metrics.replace(key, prev, update)) {
                    return update;
                }
            }
        }
    }

    public void setMetabaseStatus(MetabaseStatus status) {
        this.metabaseStatus = status;
    }

    public void addMetric(int numId, Labels labels, MetricType type) {
        createOrResolve(key(numId, labels), type, stockpile.randomShardId());
    }

    public void disableNewMetrics(boolean disable) {
        this.disableNewMetrics = disable;
    }

    @Override
    public CompletableFuture<EStockpileStatusCode> write(int shardId, List<Log> resolved) {
        return writer.write(shardId, resolved);
    }

    public MetricArchiveImmutable read(int numId, Labels labels) {
        var metric = metrics.get(key(numId, labels));
        if (metric == null) {
            return MetricArchiveImmutable.empty;
        }

        return read(metric.shardId, metric.localId);
    }

    public Metric resolve(int numId, Labels labels) {
        return metrics.get(key(numId, labels));
    }

    public MetricArchiveImmutable read(int shardId, long localId) {
        return stockpile.readCompressedOne(TReadRequest.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(shardId)
                .setLocalId(localId)
                .build())
            .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
            .build())
            .thenApply(response -> {
                if (response.getStatus() != EStockpileStatusCode.OK) {
                    throw new StockpileClientException(new StockpileStatus(response.getStatus(), response.getStatusMessage()));
                }

                return (MetricArchiveImmutable) TimeSeriesCodec.sequenceDecode(response);
            })
            .join();
    }

    @Override
    public void close() {
        writer.close();
        stockpile.close();
    }

    private <T> CompletableFuture<T> before(Supplier<T> fn) {
        return before().thenApplyAsync(ignore -> fn.get());
    }

    private CompletableFuture<?> before() {
        var copy = beforeSupplier;
        if (copy == null) {
            return CompletableFuture.completedFuture(null);
        }

        return copy.get();
    }

    private static class MetricImpl implements Metric {
        private final Key key;
        private final MetricType type;
        private final int shardId;
        private final long localId;

        public MetricImpl(Key key, MetricType type, int shardId, long localId) {
            this.key = key;
            this.type = type;
            this.shardId = shardId;
            this.localId = localId;
        }

        @Override
        public int getShardId() {
            return shardId;
        }

        @Override
        public long getLocalId() {
            return localId;
        }

        @Override
        public MetricType getType() {
            return type;
        }

        public Labels getLabels() {
            return key.labels;
        }

        @Override
        public String toString() {
            return "MetricImpl{" +
                "key=" + key +
                ", type=" + type +
                ", shardId=" + shardId +
                ", localId=" + StockpileLocalId.toString(localId) +
                '}';
        }
    }

    private static class Key {
        private final int numId;
        private final Labels labels;

        public Key(int numId, Labels labels) {
            this.numId = numId;
            this.labels = labels;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            Key key = (Key) o;

            if (numId != key.numId) return false;
            return labels.equals(key.labels);
        }

        @Override
        public int hashCode() {
            int result = numId;
            result = 31 * result + labels.hashCode();
            return result;
        }

        @Override
        public String toString() {
            return "Key{" +
                "numId=" + numId +
                ", labels=" + labels +
                '}';
        }
    }
}
