package ru.yandex.solomon.dumper;

import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import io.netty.buffer.ByteBufAllocator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.dumper.storage.shortterm.DumperTx;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.protobuf.MetricTypeConverter;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.LogDataIterator;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.slog.ResolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaRecord;
import ru.yandex.solomon.slog.UnresolvedLogMetaHeader;
import ru.yandex.solomon.slog.UnresolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaRecord;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMask;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomMetricType;
import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
public class ParsingTaskTest {
    private MetricsCacheStub metricsCache;
    private MemstoreSnapshotBuilder builder;
    private int shardId;
    private long txn;
    private EDecimPolicy decimPolicy;

    @Before
    public void setUp() {
        metricsCache = new MetricsCacheStub();
        builder = new MemstoreSnapshotBuilder(ByteBufAllocator.DEFAULT);
        var random = ThreadLocalRandom.current();
        shardId = DumperShardId.random();
        txn = random.nextLong();
        decimPolicy = EDecimPolicy.values()[random.nextInt(EDecimPolicy.values().length - 1)];
    }

    @After
    public void tearDown() {
        builder.close();
    }

    @Test
    public void parsingEmpty() {
        builder.commonLabels(Labels.of());
        ParsingResult result = parse();
        assertNull(result.unresolved);
        assertEquals(0, result.resolved.values().size());
    }

    @Test
    public void parsingAllUnresolved() {
        var alice = randomData(Labels.of("host", "alice"));
        var bob = randomData(Labels.of("host", "bob").add("tier", "test"));

        builder.commonLabels(Labels.of());
        addToLog(alice, bob);

        ParsingResult result = parse();
        assertEquals(0, result.resolved.values().size());
        assertNotNull(result.unresolved);
        assertUnresolvedEqualTo(result.unresolved, alice, bob);
    }

    @Test
    public void parsingWithCommonLabels() {
        Labels commonLabels = Labels.of("dc", "Sas");
        var alice = randomData(commonLabels.add("host", "alice"));
        var bob = randomData(commonLabels.add("host", "bob"));

        builder.commonLabels(commonLabels);
        addToLog(commonLabels, alice, bob);

        ParsingResult result = parse();
        assertNotNull(result.unresolved);
        assertUnresolvedEqualTo(result.unresolved, alice, bob);
    }

    @Test
    public void parsingWholeIntoSameShard() {
        Labels commonLabels = Labels.of("dc", "Sas");
        var alice = randomData(commonLabels.add("host", "alice"), 42);
        var bob = randomData(commonLabels.add("host", "bob"), 42);
        addToCache(alice, bob);

        builder.commonLabels(commonLabels);
        addToLog(commonLabels, alice, bob);

        ParsingResult result = parse();
        assertNull(result.unresolved);
        assertResolvedEqualTo(result.resolved.get(42), alice, bob);
    }

    @Test
    public void parseIntoDifferentShards() {
        var alice = randomData(Labels.of("host", "alice"), 42);
        var bob = randomData(Labels.of("host", "bob"), 12);
        addToCache(alice, bob);

        builder.commonLabels(Labels.of());
        addToLog(alice, bob);

        ParsingResult result = parse();
        assertNull(result.unresolved);
        assertResolvedEqualTo(result.resolved.get(42), alice);
        assertResolvedEqualTo(result.resolved.get(12), bob);
    }

    @Test
    public void partlyResolve() {
        var alice = randomData(Labels.of("host", "alice"), 42);
        var bob = randomData(Labels.of("host", "bob"), 42);
        var eva = randomData(Labels.of("host", "eva", "type", "hacker"));
        var dave = randomData(Labels.of("host", "dave"));
        addToCache(alice, bob);

        builder.commonLabels(Labels.of());
        addToLog(alice, eva, bob, dave);

        ParsingResult result = parse();
        assertNotNull(result.unresolved);
        assertUnresolvedEqualTo(result.unresolved, eva, dave);
        assertResolvedEqualTo(result.resolved.get(42), alice, bob);
    }

    @Test
    public void ignoreSettlersAtBegin() {
        var alice = randomData(Labels.of("host", "alice"));
        var bob = randomData(Labels.of("host", "bob"), 42);
        var eva = randomData(Labels.of("host", "eva"));

        addToSettlers(alice);
        addToCache(bob);

        builder.commonLabels(Labels.of());
        addToLog(alice, bob, eva);

        ParsingResult result = parse();
        assertNotNull(result.unresolved);
        assertUnresolvedEqualTo(result.unresolved, eva);
        assertResolvedEqualTo(result.resolved.get(42), bob);
    }

    @Test
    public void ignoreSettlersAtMiddle() {
        var alice = randomData(Labels.of("host", "alice"), 42);
        var bob = randomData(Labels.of("host", "bob"));
        var eva = randomData(Labels.of("host", "eva"));

        addToCache(alice);
        addToSettlers(bob);

        builder.commonLabels(Labels.of());
        addToLog(alice, bob, eva);

        ParsingResult result = parse();
        assertNotNull(result.unresolved);
        assertUnresolvedEqualTo(result.unresolved, eva);
        assertResolvedEqualTo(result.resolved.get(42), alice);
    }

    @Test
    public void ignoreSettlersAtEnd() {
        var alice = randomData(Labels.of("host", "alice"), 42);
        var bob = randomData(Labels.of("host", "bob"));
        var eva = randomData(Labels.of("host", "eva"));

        addToCache(alice);
        addToSettlers(eva);

        builder.commonLabels(Labels.of());
        addToLog(alice, bob, eva);

        ParsingResult result = parse();
        assertNotNull(result.unresolved);
        assertUnresolvedEqualTo(result.unresolved, bob);
        assertResolvedEqualTo(result.resolved.get(42), alice);
    }

    @Test
    public void random() {
        var source = IntStream.range(0, 10_000)
            .mapToObj(value -> randomData(Labels.of("name", "errors", "shardId", Integer.toString(value))))
            .collect(Collectors.toList());
        var partition = source.stream().collect(Collectors.partitioningBy(data -> ThreadLocalRandom.current().nextDouble() > 0.2));
        var resolved = partition.get(true);
        var unresolved = partition.get(false);
        var byShardId = resolved.stream().collect(Collectors.groupingBy(data -> data.shardId));

        addToCache(resolved.toArray(new Data[0]));

        builder.commonLabels(Labels.of());
        addToLog(source.toArray(new Data[0]));

        ParsingResult result = parse();
        assertNotNull(result.unresolved);
        assertUnresolvedEqualTo(result.unresolved, unresolved.toArray(new Data[0]));
        for (var entry : byShardId.entrySet()) {
            int shardId = entry.getKey();
            var expected = entry.getValue();
            var resolvedLog = result.resolved.get(shardId);
            assertResolvedEqualTo(resolvedLog, expected.toArray(new Data[0]));
        }
    }

    private void assertUnresolvedEqualTo(Log log, Data... expected) {
        int expectedPoints = Stream.of(expected).mapToInt(Data::points).sum();

        var metaHeader = new UnresolvedLogMetaHeader(log.meta);
        assertEquals(metaHeader.toString(), expected.length, metaHeader.metricsCount);
        assertEquals(metaHeader.toString(),expectedPoints, metaHeader.pointsCount);

        try (
                var metaIt = new UnresolvedLogMetaIteratorImpl(metaHeader, log.meta, Labels.allocator);
                var dataIt = LogDataIterator.create(log.data);
        ) {
            assertEquals(log.numId, dataIt.getNumId());
            assertEquals(expected.length, dataIt.getMetricsCount());
            assertEquals(expectedPoints, dataIt.getPointsCount());

            int idx = 0;
            var metaRecord = new UnresolvedLogMetaRecord();
            while (metaIt.next(metaRecord)) {
                assertTrue(metaRecord.toString(), dataIt.hasNext());
                Data expectedData = expected[idx++];
                assertEquals(expectedData.labels, metaRecord.labels);
                assertEquals(expectedData.type(), metaRecord.type);
                assertEquals(expectedData.points(), metaRecord.points);

                var actualArchive = new MetricArchiveMutable();
                dataIt.next(actualArchive);
                assertEquals(builder.getNumId(), actualArchive.getOwnerShardId());
                assertEquals(expectedData.archive, actualArchive);
            }
            assertFalse(dataIt.hasNext());
        } finally {
            log.close();
        }
    }

    private void assertResolvedEqualTo(Log log, Data... expected) {
        int expectedPoints = Stream.of(expected).mapToInt(Data::points).sum();

        var metaHeader = new ResolvedLogMetaHeader(log.meta);
        assertEquals(metaHeader.toString(), expected.length, metaHeader.metricsCount);
        assertEquals(metaHeader.toString(), expectedPoints, metaHeader.pointsCount);
        assertEquals(shardId, metaHeader.producerId);
        assertEquals(txn, metaHeader.producerSeqNo);
        assertEquals(decimPolicy, metaHeader.decimPolicy);

        try (
                var metaIt = new ResolvedLogMetaIteratorImpl(metaHeader, log.meta);
                var dataIt = LogDataIterator.create(log.data);
        ) {
            assertEquals(log.numId, dataIt.getNumId());
            assertEquals(expected.length, dataIt.getMetricsCount());
            assertEquals(expectedPoints, dataIt.getPointsCount());

            int idx = 0;
            var metaRecord = new ResolvedLogMetaRecord();
            while (metaIt.next(metaRecord)) {
                assertTrue(metaRecord.toString(), dataIt.hasNext());
                Data expectedData = expected[idx++];
                assertEquals(expectedData.localId, metaRecord.localId);
                assertEquals(expectedData.type(), metaRecord.type);
                assertEquals(expectedData.points(), metaRecord.points);

                var actualArchive = new MetricArchiveMutable();
                dataIt.next(actualArchive);
                assertEquals(builder.getNumId(), actualArchive.getOwnerShardId());
                assertEquals(expectedData.archive, actualArchive);
            }
            assertFalse(dataIt.hasNext());
        } finally {
            log.close();
        }
    }

    public void addToLog(Data... data) {
        addToLog(Labels.of(), data);
    }

    public void addToLog(Labels commonLabels, Data... data) {
        for (var d : data) {
            var labels = rmCommonLabels(d.labels, commonLabels);
            builder.add(labels, d.archive);
        }
    }

    public void addToCache(Data... data) {
        for (var d : data) {
            metricsCache.add(d.type(), d.labels, d.shardId, d.localId);
        }
    }

    public void addToSettlers(Data... data) {
        for (var d : data) {
            metricsCache.addSettler(d.labels);
        }
    }

    public Labels rmCommonLabels(Labels labels, Labels common) {
        for (int index = 0; index < common.size(); index++) {
            var commonLabel = common.at(index);
            int idx = labels.indexByKey(commonLabel.getKey());
            if (idx < 0) {
                continue;
            }
            var found = labels.at(idx);
            if (!found.getValue().equals(commonLabel.getValue())) {
                continue;
            }
            labels = labels.removeByIndex(idx);
        }
        return labels;
    }

    private Data randomData(Labels labels) {
        return randomData(labels, ThreadLocalRandom.current().nextInt(1, 5000));
    }

    private Data randomData(Labels labels, int shardId) {
        return randomData(labels, shardId, StockpileLocalId.random());
    }

    private Data randomData(Labels labels, int shardId, long localId) {
        var archive = randomArchive();
        return new Data(labels, shardId, localId, archive);
    }

    private MetricArchiveMutable randomArchive() {
        var random = ThreadLocalRandom.current();
        var archive = archive();
        var type = randomMetricType();
        archive.setType(type);
        var mask = randomMask(type);
        var ts0 = TsRandomData.randomTs(random);
        var point = RecyclableAggrPoint.newInstance();
        for (int pointIndex = 0; pointIndex < pointsCount(random); pointIndex++) {
            randomPoint(point, mask);
            point.setTsMillis(ts0 + (10_000 * pointIndex));
            point.setStepMillis(10_000);
            archive.addRecord(point);
        }
        point.recycle();
        return archive;
    }

    private int pointsCount(ThreadLocalRandom random) {
        int type = random.nextInt(3);
        switch (type) {
            case 0: return 0;
            case 1: return 1;
        }
        return random.nextInt(2, 1000);
    }

    private MetricArchiveMutable archive() {
        var archive = new MetricArchiveMutable();
        archive.setOwnerShardId(builder.getNumId());
        return archive;
    }

    private ParsingResult parse() {
        var log = builder.build();
        System.out.println("Meta: " + DataSize.shortString(log.meta.readableBytes()));
        System.out.println("Data: " + DataSize.shortString(log.data.readableBytes()));
        var tx = new DumperTx(0, shardId, txn, System.currentTimeMillis() / 1000);
        var task = new ParsingTask(tx, decimPolicy, log, metricsCache, Labels.allocator, ByteBufAllocator.DEFAULT);
        return task.run();
    }

    private static class Data {
        private final Labels labels;
        private final int shardId;
        private final long localId;
        private final MetricArchiveMutable archive;

        public Data(Labels labels, int shardId, long localId, MetricArchiveMutable archive) {
            this.labels = labels;
            this.shardId = shardId;
            this.localId = localId;
            this.archive = archive;
        }

        public MetricType type() {
            return MetricTypeConverter.fromProto(archive.getType());
        }

        public int points() {
            return archive.getRecordCount();
        }
    }
}
