package ru.yandex.solomon.dumper;


import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import io.netty.buffer.ByteBuf;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.dumper.storage.shortterm.DumperFile;
import ru.yandex.solomon.dumper.storage.shortterm.DumperTx;
import ru.yandex.solomon.dumper.storage.shortterm.ShortTermStorageReader;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;

/**
 * @author Vladimir Gordiychuk
 */
public class ShortTermStorageStub implements ShortTermStorageReader {

    private final AtomicInteger allocateTxn = new AtomicInteger();
    private final Queue<Item> files = new ConcurrentLinkedQueue<>();
    private final Queue<Item> inFlightTxn = new ConcurrentLinkedQueue<>();
    private final Executor executor = ForkJoinPool.commonPool();
    private final ActorRunner actor = new ActorRunner(this::act, executor);

    private final AtomicReference<CompletableFuture<DumperFile>> nextFuture = new AtomicReference<>();
    private volatile boolean stop;

    public CompletableFuture<Void> enqueue(ByteBuf content) {
        var future = new CompletableFuture<Void>();
        var file = new DumperFile(nextTxn(), content);
        files.add(new Item(file, future));
        actor.schedule();
        return future;
    }

    private DumperTx nextTxn() {
        var createdAt = System.currentTimeMillis() / 1000L;
        return new DumperTx(0, 42, allocateTxn.incrementAndGet(), createdAt);
    }

    @Override
    public CompletableFuture<DumperFile> next() {
        var future = new CompletableFuture<DumperFile>();
        if (!nextFuture.compareAndSet(null, future)) {
            future.completeExceptionally(new IllegalStateException("Previous not finished yet"));
        }
        actor.schedule();
        return future;
    }

    @Override
    public CompletableFuture<Void> commit(List<DumperTx> txList) {
        return CompletableFuture.runAsync(() -> {
            if (stop) {
                throw new IllegalStateException("Already stopped");
            }

            for (DumperTx tx : txList) {
                var item = inFlightTxn.peek();
                if (item == null) {
                    throw new IllegalStateException("Unable to commit txn that not started " + tx);
                }

                if (item.file.tx.compareTo(tx) != 0) {
                    throw new IllegalStateException("Not seq commit txn: " + item.file.tx + " != " + tx);
                }

                if (inFlightTxn.poll() != item) {
                    throw new IllegalStateException("Concurrent txn commit");
                }
                System.out.println("Commit: " + tx);
                executor.execute(() -> item.doneFuture.complete(null));
            }
        });
    }

    @Override
    public boolean isStop() {
        return stop;
    }

    @Override
    public void stop() {
        stop = true;
        actor.schedule();
    }

    private void act() {
        var next = nextFuture.get();
        if (next == null) {
            return;
        }

        if (stop) {
            if (!nextFuture.compareAndSet(next, null)) {
                throw new IllegalStateException("concurrent modify");
            }
            next.completeExceptionally(new IllegalStateException("Already stopped"));
            return;
        }

        if (files.isEmpty()) {
            return;
        }

        var item = files.poll();
        inFlightTxn.add(item);
        if (!nextFuture.compareAndSet(next, null)) {
            throw new IllegalStateException("concurrent modify");
        }

        System.out.println("Fetch: "+ item.file.tx);
        executor.execute(() -> next.complete(item.file));
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
    }

    private static class Item {
        private DumperFile file;
        private CompletableFuture<Void> doneFuture;

        public Item(DumperFile file, CompletableFuture<Void> doneFuture) {
            this.file = file;
            this.doneFuture = doneFuture;
        }
    }
}
