package ru.yandex.solomon.dumper;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.SolomonRawConf;
import ru.yandex.solomon.core.db.dao.memory.InMemoryShardDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.stockpile.api.EDecimPolicy;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonShardOptsProviderImplTest {

    private InMemoryShardDao dao;
    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private SolomonShardOptsProviderImpl provider;

    @Before
    public void setUp() {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        dao = new InMemoryShardDao();
        provider = new SolomonShardOptsProviderImpl(dao, timer, timer);
    }

    @After
    public void tearDown() throws Exception {
        timer.shutdownNow();
    }

    @Test
    public void init() {
        int numId = ThreadLocalRandom.current().nextInt();
        assertFalse(provider.isInitialized());
        assertNull(provider.resolve(numId));

        provider.updateSnapshot(List.of(shard(numId, "test").build()));

        assertTrue(provider.isInitialized());

        var actual = provider.resolve(numId);
        var expected = SolomonShardOpts.newBuilder()
                .withNumId(numId)
                .withId("test")
                .withProjectId("project-id-test")
                .withDecimPolicy(EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS)
                .build();
        assertEquals(expected, actual);
        assertNull(provider.resolve(42));
    }

    @Test
    public void hashChangedWithChangedListOfShards() {
        var v1 = provider.optionsHash();
        assertEquals(v1, provider.optionsHash());
        var alice = shard(42, "alice").build();

        provider.updateSnapshot(List.of(alice));
        var v2 = provider.optionsHash();
        assertNotEquals("New shard appear into config, hash should be changed", v1, v2);

        provider.updateSnapshot(List.of(alice));
        var v3 = provider.optionsHash();
        assertEquals("When configuration not changed, hash should be stable", v2, v3);

        var bob = shard(85, "bob").build();

        provider.updateSnapshot(List.of(bob));
        var v4 = provider.optionsHash();

        assertNotEquals("Different shard appear", v4, v3);
        assertEquals("Hash stable", v4, provider.optionsHash());

        provider.updateSnapshot(List.of(alice, bob));
        var v5 = provider.optionsHash();
        assertNotEquals(v4, v5);
    }

    @Test
    public void snapshotImmutable() {
        var alice = shard(1, "alice").build();
        var bob = shard(2, "bob").build();

        assertEquals(provider.snapshot().optionsHash(), 0);

        provider.updateSnapshot(List.of(alice));

        var v1 = provider.snapshot();
        var v1Hash = v1.optionsHash();
        assertEquals(provider.optionsHash(), v1Hash);
        assertNotNull(v1.resolve(1));
        assertNull(v1.resolve(2));

        provider.updateSnapshot(List.of(alice, bob));
        assertNotEquals(v1.optionsHash(), provider.optionsHash());

        assertEquals(v1Hash, v1.optionsHash());
        assertNotNull(v1.resolve(1));
        assertNull(v1.resolve(2));

        var v2 = provider.snapshot();
        assertNotNull(v2.resolve(1));
        assertNotNull(v2.resolve(2));
        assertNotEquals(v1.optionsHash(), v2.optionsHash());
    }

    private Shard.Builder shard(int numId, String id) {
        return Shard.newBuilder()
                .setId(id)
                .setNumId(numId)
                .setProjectId("project-id-" + id)
                .setServiceId("service-id-" + id)
                .setServiceName("service-name-" + id)
                .setClusterId("cluster-id-"+id)
                .setClusterName("cluster-name-"+id)
                .setShardSettings(ShardSettings.of(ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        73,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, new ServiceMetricConf.AggrRule[0], false),
                        0));
    }

    private static SolomonConfWithContext solomonConf(Shard... shards) {
        List<Project> projects = Arrays.stream(shards)
                .map(s -> Project.newBuilder()
                        .setId(s.getProjectId())
                        .setName(s.getProjectId())
                        .setOwner("jamel")
                        .build())
                .distinct()
                .collect(Collectors.toList());

        List<Service> services = Arrays.stream(shards)
                .map(s -> Service.newBuilder()
                        .setId(s.getServiceId())
                        .setName(s.getServiceName())
                        .setProjectId(s.getProjectId())
                        .build())
                .distinct()
                .collect(Collectors.toList());

        List<Cluster> clusters = Arrays.stream(shards)
                .map(s -> Cluster.newBuilder()
                        .setId(s.getClusterId())
                        .setName(s.getClusterName())
                        .setProjectId(s.getProjectId())
                        .build())
                .distinct()
                .collect(Collectors.toList());

        SolomonRawConf rawConf = new SolomonRawConf(List.of(), projects, clusters, services, Arrays.asList(shards));
        return SolomonConfWithContext.create(rawConf);
    }
}
