package ru.yandex.solomon.dumper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import io.netty.buffer.ByteBufAllocator;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import org.apache.commons.lang3.RandomStringUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.dumper.LogHelper.Metric;
import ru.yandex.solomon.dumper.storage.shortterm.DumperTx;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.CoreMatchers.instanceOf;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static ru.yandex.solomon.dumper.LogHelper.concat;
import static ru.yandex.solomon.dumper.LogHelper.copy;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonShardProcessTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(60, TimeUnit.SECONDS)
            .withLookingForStuckThread(true)
            .build();

    private int numId;
    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private LongTermStorageStub storage;
    private SolomonShardProcess process;
    private AtomicLong txn = new AtomicLong(42);

    @Before
    public void setUp() {
        numId = ThreadLocalRandom.current().nextInt();
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(2, clock);
        storage = new LongTermStorageStub(timer);
        var metrics = new SolomonShardProcessMetrics(new MetricRegistry());
        var opts = SolomonShardOpts.newBuilder()
                .withId("junk")
                .withProjectId("solomon")
                .withNumId(numId)
                .build();
        process = new SolomonShardProcess(opts, ForkJoinPool.commonPool(), timer, storage, metrics, 1 << 10); // 1kb
    }

    @After
    public void tearDown() {
        process.close();
        timer.shutdownNow();
    }

    @Test
    public void sequentialWrite() {
        var alice = metric("alice");
        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            write(prepareLog(alice)).join();
            assertWrite(alice);
        }
    }

    @Test
    public void sequenceOrderSave() {
        var ts0 = System.currentTimeMillis();
        var alice = metric("alice");
        var parsed = IntStream.range(0, 20)
            .mapToObj(idx -> prepareLog(alice.add(ts0, idx)))
            .map(this::parse)
            .collect(toList());
        write(parsed).join();

        assertWrite(alice);
    }

    @Test
    public void useCacheWhenMetabaseNotReady() {
        var alice = metric("alice").addNext(1);
        write(prepareLog(alice)).join();
        assertWrite(alice);

        // metabase not ready now, we should use cache instead
        storage.setMetabaseStatus(MetabaseStatus.fromCode(EMetabaseStatusCode.SHARD_NOT_READY, "test status"));

        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            write(prepareLog(alice)).join();
            assertWrite(alice);
        }
    }

    @Test
    public void useMetabaseOnlyForNewMetrics() {
        var alice = metric("alice").addNext(1);
        write(prepareLog(alice)).join();
        assertWrite(alice);

        // metabase not ready now, we should use cache instead
        storage.setMetabaseStatus(MetabaseStatus.fromCode(EMetabaseStatusCode.SHARD_NOT_READY, "test status"));

        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            write(prepareLog(alice)).join();
            assertWrite(alice);
        }

        storage.setMetabaseStatus(MetabaseStatus.OK);

        // new metric log
        var bob = metric("bob").addNext(1);
        for (int idx = 0; idx < 3; idx++) {
            bob.addNext(idx);
            alice.addNext(3 + idx);
            write(prepareLog(alice, bob)).join();
            assertWrite(alice);
            assertWrite(bob);
        }

        assertWrite(alice);
        assertWrite(bob);
    }

    @Test
    public void slowParseFastParse() {
        var ts0 = System.currentTimeMillis();
        var alice = metric("alice").add(ts0, 1);
        write(prepareLog(alice)).join();
        assertWrite(alice);

        var huge = IntStream.range(0, 10000)
            .mapToObj(idx -> metric("idx" + idx).addNext(idx))
            .collect(Collectors.toList());

        alice.add(ts0, 2);
        var log2 = prepareLog(concat(huge, List.of(alice)));

        alice.add(ts0, 3);
        var log3 = prepareLog(alice);

        write(List.of(parse(log2), parse(log3))).join();
        assertWrite(alice);
        for (int index = 0; index < 3; index++) {
            var metric = huge.get(ThreadLocalRandom.current().nextInt(huge.size()));
            assertWrite(metric);
        }
    }

    @Test
    public void resolveAndReplaceByNext() {
        var ts0 = System.currentTimeMillis();

        List<Metric> metrics = new ArrayList<>();
        List<CompletableFuture<Int2ObjectMap<Log>>> futures = new ArrayList<>();
        for (int index = 0; index < 100; index++) {
            var metric = metric("idx" + index).add(ts0, 1);
            futures.add(parse(prepareLog(metric)));
            metric.add(ts0, 2);
            int finalIndex = index;
            metrics.forEach(s -> s.addNext(finalIndex));
            metrics.add(metric);
            futures.add(parse(prepareLog(metrics)));
        }
        write(futures).join();
        for (Metric metric : metrics) {
            assertWrite(metric);
        }
    }

    @Test
    public void resolveAndReplaceByNextConcurrent() {
        var ts0 = System.currentTimeMillis();

        List<Metric> metrics = new ArrayList<>();
        List<Log> logs = new ArrayList<>();
        for (int index = 0; index < 100; index++) {
            var metric = metric("idx" + index).add(ts0, 1);
            logs.add(prepareLog(metric));
            metric.add(ts0, 2);
            int finalIndex = index;
            metrics.forEach(s -> s.addNext(finalIndex));
            metrics.add(metric);
            logs.add(prepareLog(metrics));
        }

        write(logs.stream()
            .map(this::parse)
            .collect(Collectors.toList()))
                .join();

        for (Metric metric : metrics) {
            assertWrite(metric);
        }
    }

    @Test
    public void failedParseNotBlockValidParse() {
        var alice = metric("alice").addNext(1);
        var bob = metric("bob").addNext(1);

        var aliceLog = prepareLog(alice);
        var evaLog = new Log(numId, ByteBufAllocator.DEFAULT.buffer(), ByteBufAllocator.DEFAULT.buffer());
        var bobLog = prepareLog(bob);

        var aliceFuture = write(aliceLog);
        var evaFuture = write(evaLog);
        var bobFuture = write(bobLog);

        aliceFuture.join();
        assertWrite(alice);

        bobFuture.join();
        assertWrite(bob);

        var r = evaFuture
            .thenApply(ignore -> null)
            .exceptionally(e -> {
                e.printStackTrace();
                return e;
            })
            .join();

        assertNotNull(r);
    }

    @Test
    public void ignoreSettlers() {
        var alice = metric("alice");
        var bob = metric("bob");
        storage.addMetric(alice.numId, alice.labels, MetricType.DGAUGE);
        storage.disableNewMetrics(true);

        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            bob.addNext(idx);
            write(prepareLog(alice, bob)).join();
            assertNotNull(storage.resolve(alice.numId, alice.labels));
            assertNull(storage.resolve(bob.numId, bob.labels));
            assertWrite(alice);
        }
    }

    @Test
    public void ignoreSettlersAllResponseEmpty() {
        // warm cache
        var alice = metric("alice");
        alice.addNext(42);
        write(prepareLog(alice)).join();
        assertNotNull(storage.resolve(alice.numId, alice.labels));

        // no more new metrics
        storage.disableNewMetrics(true);

        var bob = metric("bob");
        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            bob.addNext(idx);
            write(prepareLog(alice, bob)).join();
            assertNotNull(storage.resolve(alice.numId, alice.labels));
            assertNull(storage.resolve(bob.numId, bob.labels));
            assertWrite(alice);
        }
    }

    @Test
    public void resetSettlers() {
        var alice = metric("alice");
        var bob = metric("bob");
        storage.addMetric(alice.numId, alice.labels, MetricType.DGAUGE);
        storage.disableNewMetrics(true);

        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            bob.addNext(idx);
            write(prepareLog(alice, bob)).join();
            assertNotNull(storage.resolve(alice.numId, alice.labels));
            assertNull(storage.resolve(bob.numId, bob.labels));
            assertWrite(alice);
        }

        storage.disableNewMetrics(false);
        bob = metric("bob");
        for (int idx = 0; idx < 3; idx++) {
            alice.addNext(idx);
            bob.addNext(idx);
            write(prepareLog(alice, bob)).join();
            assertWrite(alice);
            assertWrite(bob);
        }
    }

    @Test(timeout = 5_000)
    public void retryResolveMetrics() throws InterruptedException {
        var alice = metric("alice").addNext(42);
        storage.addMetric(alice.numId, alice.labels, MetricType.DGAUGE);
        List<MetabaseStatus> errors = List.of(
                MetabaseStatus.fromCode(EMetabaseStatusCode.DEADLINE_EXCEEDED, "oops deadline"),
                MetabaseStatus.fromCode(EMetabaseStatusCode.SHARD_NOT_READY, "not ready yet"),
                MetabaseStatus.fromCode(EMetabaseStatusCode.SHARD_NOT_FOUND, "not found, try another node"),
                MetabaseStatus.fromCode(EMetabaseStatusCode.NODE_UNAVAILABLE, "service temporary unavailable"));

        storage.setMetabaseStatus(errors.get(ThreadLocalRandom.current().nextInt(errors.size())));
        var sync = new CountDownLatch(1);
        var write = write(prepareLog(alice))
                .whenComplete((ignore, e) -> sync.countDown());

        assertFalse(sync.await(100, TimeUnit.MILLISECONDS));

        storage.setMetabaseStatus(MetabaseStatus.OK);
        while (!write.isDone()) {
            clock.passedTime(30, TimeUnit.SECONDS);
            sync.await(1, TimeUnit.MILLISECONDS);
        }

        write.join();
        assertWrite(alice);
    }

    @Test(timeout = 5_000)
    public void stopRetryOnClose() throws InterruptedException {
        var alice = metric("alice").addNext(42);
        storage.addMetric(alice.numId, alice.labels, MetricType.DGAUGE);
        storage.setMetabaseStatus(MetabaseStatus.fromCode(EMetabaseStatusCode.NODE_UNAVAILABLE, "service temporary unavailable"));
        var sync = new CountDownLatch(1);
        var write = write(prepareLog(alice))
                .whenComplete((ignore, e) -> sync.countDown());

        assertFalse(sync.await(100, TimeUnit.MILLISECONDS));
        process.close();
        while (!write.isDone()) {
            clock.passedTime(30, TimeUnit.SECONDS);
            sync.await(1, TimeUnit.MILLISECONDS);
        }

        try {
            write.join();
            fail("future should be completed with error if ");
        } catch (CompletionException e) {
            assertThat(e.getCause(), instanceOf(IllegalStateException.class));
        }
    }

    @Test
    public void limitMessageSizeToResolve() throws InterruptedException {
        storage.setMetabaseStatus(MetabaseStatus.fromCode(EMetabaseStatusCode.NODE_UNAVAILABLE, "expect"));

        CountDownLatch sync = new CountDownLatch(1);
        storage.beforeSupplier = () -> {
            sync.countDown();
            return CompletableFuture.completedFuture(null);
        };

        var metrics = generateMetrics(1 << 10); // 1 Kb
        var log = prepareLog(metrics);
        var future =  IntStream.range(0, 3)
                .mapToObj(ignore -> write(copy(log)))
                .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOfVoid));
        log.close();

        while (!sync.await(100, TimeUnit.MILLISECONDS)) {
            clock.passedTime(30, TimeUnit.SECONDS);
        }
        storage.setMetabaseStatus(MetabaseStatus.OK);

        System.out.println(process.memorySizeIncludingSelf());

        var doneSync = new CountDownLatch(1);
        future.whenComplete((ignore, e) -> doneSync.countDown());
        while (!doneSync.await(10, TimeUnit.MILLISECONDS)) {
            clock.passedTime(30, TimeUnit.SECONDS);
        }

        future.join();

        assertWrite(metrics.get(0));
        assertWrite(metrics.get(metrics.size() - 1));
        assertWrite(metrics.get(ThreadLocalRandom.current().nextInt(metrics.size())));
    }

    private List<Metric> generateMetrics(int byteSize) {
        var now = System.currentTimeMillis();
        List<Metric> result = new ArrayList<>();
        int total = 0;
        LabelsBuilder labels = Labels.builder(16);
        while (total < byteSize) {
            for (int index = 0; index < 16; index++) {
                labels.add("k" + index, RandomStringUtils.randomAlphanumeric(200));
                total += 200;
            }
            result.add(new Metric(numId, labels.build()).add(now, 42));
        }
        return result;
    }

    public Metric metric(String name) {
        return new Metric(numId, Labels.of("name", name));
    }

    private void assertWrite(Metric metric) {
        var actual = AggrGraphDataArrayList.of(storage.read(numId, metric.labels));
        assertEquals(metric.labels.toString(), metric.expected(), actual);
    }

    private CompletableFuture<Int2ObjectMap<Log>> parse(Log log) {
        DumperTx tx = new DumperTx(0, 42, txn.incrementAndGet(), System.currentTimeMillis());
        return process.enqueue(tx, log);
    }

    private CompletableFuture<Void> write(Log log) {
        return parse(log).thenCompose(this::write);
    }

    private CompletableFuture<Void> write(List<CompletableFuture<Int2ObjectMap<Log>>> futures) {
        return CompletableFutures.allOf(futures)
            .thenCompose(resolved -> resolved.stream()
                .map(this::write)
                .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid)));
    }

    private CompletableFuture<Void> write(Int2ObjectMap<Log> resolved) {
        List<CompletableFuture<?>> futures = new ArrayList<>(resolved.size());
        for (var entry : resolved.int2ObjectEntrySet()) {
            futures.add(storage.write(entry.getIntKey(), List.of(entry.getValue())));
        }
        return CompletableFutures.allOfVoid(futures);
    }

    private Log prepareLog(Metric... metrics) {
        return prepareLog(Arrays.asList(metrics));
    }

    private Log prepareLog(List<Metric> metrics) {
        return LogHelper.prepareLog(numId, metrics);
    }
}
