package ru.yandex.solomon.dumper;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import io.netty.buffer.ByteBufAllocator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueObject;
import ru.yandex.solomon.model.point.column.ValueRandomData;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.ResolvedLogMetaBuilderImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.slog.SnapshotLogDataBuilderImpl;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assume.assumeThat;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileWriterImplTest {
    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private StockpileClientStub stockpile;
    private StockpileWriterImpl writer;
    private final AtomicLong seqNo = new AtomicLong(42);

    @Before
    public void setUp() {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
        stockpile = new StockpileClientStub(ForkJoinPool.commonPool());
        writer = new StockpileWriterImpl(stockpile, ForkJoinPool.commonPool(), timer, new MetricRegistry());
    }

    @After
    public void tearDown() {
        timer.shutdownNow();
    }

    @Test
    public void writeOne() {
        int shardId = stockpile.randomShardId();
        long localId = StockpileLocalId.random();
        var data = randomData();
        var log = log(1, localId, data);

        var status = writer.write(shardId, List.of(log)).join();
        assertEquals(EStockpileStatusCode.OK, status);

        stockpile.readCompressedOne(TReadRequest.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(shardId)
                .setLocalId(localId)
                .build())
            .build())
            .join();

        var result = read(shardId, localId);
        assertEquals(data, result);
    }

    @Test
    public void retryWrite() throws InterruptedException {
        assumeThat(StockpileFormat.CURRENT.getFormat(), greaterThanOrEqualTo(StockpileFormat.IDEMPOTENT_WRITE_38.getFormat()));
        int shardId = stockpile.randomShardId();
        long localId = StockpileLocalId.random();
        var data = randomData();
        var log = log(1, localId, data);

        stockpile.predefineStatusCode(EStockpileStatusCode.DEADLINE_EXCEEDED);

        var sync = stockpile.requestSync();
        var future = writer.write(shardId, List.of(log));

        // request failed
        sync.await();

        sync = stockpile.requestSync();
        while (!future.isDone() && !sync.await(10, TimeUnit.MILLISECONDS)) {
            clock.passedTime(10, TimeUnit.SECONDS);
        }

        assertFalse(future.isDone());
        assertTrue(sync.await(0, TimeUnit.MILLISECONDS));

        sync = stockpile.requestSync();
        stockpile.predefineStatusCode(EStockpileStatusCode.OK);
        while (!future.isDone() && sync.await(10, TimeUnit.MILLISECONDS)) {
            clock.passedTime(10, TimeUnit.SECONDS);
        }

        assertEquals(EStockpileStatusCode.OK, future.join());

        stockpile.readCompressedOne(TReadRequest.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(shardId)
                .setLocalId(localId)
                .build())
            .build())
            .join();

        var result = read(shardId, localId);
        assertEquals(data, result);
    }

    @Test
    public void requestSequenceSafe() throws InterruptedException {
        int shardId = stockpile.randomShardId();
        long localId = StockpileLocalId.random();
        var expected = new AggrGraphDataArrayList();
        long ts0 = System.currentTimeMillis();
        var logs = IntStream.range(0, 500)
            .mapToObj(idx -> {
                var data = randomData();
                data.setTsMillis(0, ts0);
                data.setValue(0, new ValueObject(idx, 0));
                expected.addAll(data);
                return log(idx, localId, data);
            })
            .collect(toList());
        expected.sortAndMerge();

        List<CompletableFuture<Void>> list = new ArrayList<>();
        for (var log : logs) {
            var future = writer.write(shardId, List.of(log))
                .thenAccept(status -> {
                    assertEquals(EStockpileStatusCode.OK, status);
                });
            list.add(future);
            TimeUnit.MICROSECONDS.sleep(ThreadLocalRandom.current().nextInt(2));
        }
        CompletableFutures.allOfVoid(list).join();

        var result = read(shardId, localId);
        assertEquals(expected, result);
    }

    @Test
    public void concurrentWrite() {
        class Entry {
            int shardId;
            long localId;
            AggrGraphDataArrayList expected;
            Log log;
        }

        var source = IntStream.range(0, 1000)
            .parallel()
            .mapToObj(value -> {
                Entry entry = new Entry();
                entry.shardId = stockpile.randomShardId();
                entry.localId = StockpileLocalId.random();
                entry.expected = randomData();
                entry.log = log(value, entry.localId, entry.expected);
                return entry;
            })
            .collect(Collectors.toList());

        source.parallelStream()
            .map(entry -> writer.write(entry.shardId, List.of(entry.log)))
            .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid))
            .join();

        for (int i = 0; i < 10; i++) {
            int index = ThreadLocalRandom.current().nextInt(source.size());
            var entry = source.get(index);
            var result = read(entry.shardId, entry.localId);
            assertEquals(entry.expected, result);
        }
    }

    private AggrGraphDataArrayList read(int shardId, long localId) {
        var response = stockpile.readCompressedOne(TReadRequest.newBuilder()
            .setMetricId(MetricId.newBuilder()
                .setShardId(shardId)
                .setLocalId(localId)
                .build())
            .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
            .build())
            .join();

        assertEquals(EStockpileStatusCode.OK, response.getStatus());
        return AggrGraphDataArrayList.of(TimeSeriesCodec.sequenceDecode(response));
    }

    private Log log(int producerId, long localId, AggrGraphDataArrayList timeseries) {
        var random = ThreadLocalRandom.current();
        var alg = CompressionAlg.values()[random.nextInt(CompressionAlg.values().length)];
        var buffer = ByteBufAllocator.DEFAULT;
        int numId = random.nextInt();
        var header = new ResolvedLogMetaHeader(numId, alg)
            .setProducerId(producerId)
            .setProducerSeqNo(seqNo.incrementAndGet());
        try (var metaBuilder = new ResolvedLogMetaBuilderImpl(header, buffer);
             var dataBuilder = new SnapshotLogDataBuilderImpl(alg, numId, buffer))
        {
            int size = dataBuilder.onTimeSeries(MetricArchiveImmutable.of(timeseries));
            metaBuilder.onMetric(MetricType.DGAUGE, localId, timeseries.length(), size);

            var meta = metaBuilder.build();
            var data = dataBuilder.build();
            return new Log(numId, meta, data);
        }
    }

    private AggrGraphDataArrayList randomData() {
        int mask = StockpileColumns.maxColumnSet(ru.yandex.solomon.model.protobuf.MetricType.DGAUGE);
        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        try {
            var timeseries = new AggrGraphDataArrayList(mask, 10);
            var random = ThreadLocalRandom.current();
            for (int index = 0; index < 10; index++) {
                point.setTsMillis(TsRandomData.randomTs(random));
                point.setValue(ValueRandomData.randomNum(random));
                point.setStepMillis(10_000);
                point.setMerge(true);
                point.setCount(1);
                timeseries.addRecord(point);
            }
            timeseries.sortAndMerge();
            return timeseries;
        } finally {
            point.recycle();
        }
    }
}
