package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.UnpooledByteBufAllocator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvGenerationChangedRuntimeException;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class KvLogReaderTest {

    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private KikimrKvClientInMem kvClient;
    private long tabletId;
    private long gen;
    private KvLogReader reader;
    private KvRetry retry;

    @Before
    public void setUp() {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(2, clock);
        kvClient = new KikimrKvClientInMem();
        tabletId = kvClient.createKvTablet();
        gen = kvClient.incrementGeneration(tabletId, System.currentTimeMillis() + 10_000).join();
        var dao = KvShortTermStorageDao.create(kvClient, new MetricRegistry(), 1 << 20);
        retry = new KvRetry(Throwable::printStackTrace, timer);
        reader = new KvLogReader(tabletId, dao, retry, 2048);
    }

    @After
    public void tearDown() {
        retry.close();
        reader.close();
        timer.shutdownNow();
    }

    @Test(expected = IllegalStateException.class)
    public void readNone() throws Throwable {
        TxDumperFile file = syncRead(42);
        file.close();
    }

    @Test
    public void readOne() throws Throwable {
        var expectedBytes = write(41);
        try (var file = syncRead(41)) {
            assertEquals(file.txn, 41);
            assertArrayEquals(expectedBytes, ByteBufUtil.getBytes(file.buffer));
            assertNotEquals(0, file.chunksCount);
        }
    }

    @Test
    public void readMany() throws Throwable {
        int count = 1000;
        int beginTxn = 42;
        List<byte[]> expected = new ArrayList<>(count);
        for (int index = 0; index < count; index++) {
            expected.add(write(beginTxn + index));
        }

        for (int index = 0; index < count; index++) {
            try (var file = syncRead(beginTxn + index)) {
                assertEquals(beginTxn + index, file.txn);
                assertArrayEquals(expected.get(index), ByteBufUtil.getBytes(file.buffer));
            }
        }
    }

    @Test
    public void readOneByOne() throws Throwable {
        long beginTxn = 42;
        for (int index = 0; index < 10; index++) {
            long txn = beginTxn + index;
            var expected = write(txn);
            try (var file = syncRead(txn)) {
                assertEquals(txn, file.txn);
                assertArrayEquals(expected, ByteBufUtil.getBytes(file.buffer));
            }
        }
    }

    @Test(expected = KikimrKvGenerationChangedRuntimeException.class)
    public void generationMismatch() throws Throwable {
        write(41);
        kvClient.incrementGeneration(tabletId, 0).join();
        try (var file = syncRead(41)) {
            assertEquals(file.txn, 41);
        }
    }

    @Test
    public void readFromCache() throws Throwable {
        List<byte[]> expected = new ArrayList<>();
        for (int index = 1; index < 100; index++) {
            expected.add(write(index));
        }

        for (int index = 1; index < 10; index++) {
            try (var file = syncRead(index)) {
                assertEquals(file.txn, index);
                assertArrayEquals(expected.get(index - 1), ByteBufUtil.getBytes(file.buffer));
                assertNotEquals(0, file.chunksCount);
            }
        }
    }

    @Test(expected = IllegalStateException.class)
    public void unableToReadWhenRetryClosed() throws Throwable {
        write(41);
        kvClient.throwOnRead();
        var future = reader.readNext(gen, 41, 1000);
        retry.close();
        clock.passedTime(10, TimeUnit.SECONDS);
        try {
            future.join().close();
        } catch (Throwable e) {
            throw CompletableFutures.unwrapCompletionException(e);
        }
    }

    private byte[] write(long txn) {
        var expectedBuffer = UnpooledByteBufAllocator.DEFAULT.heapBuffer();
        var random = ThreadLocalRandom.current();
        int chunksCount = random.nextInt(1, 3);
        var writes = new ArrayList<KikimrKvClient.Write>(chunksCount);
        for (int chunkNo = 0; chunkNo < chunksCount; chunkNo++) {
            boolean last = chunkNo + 1 == chunksCount;
            String fileName = "c." + DumperLogFileName.format(txn, chunkNo, last);
            byte[] content = new byte[random.nextInt(128, 5000)];
            writes.add(new KikimrKvClient.Write(fileName, content, MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, MsgbusKv.TKeyValueRequest.EPriority.REALTIME));
            expectedBuffer.writeBytes(content);
        }
        kvClient.writeMulti(tabletId, 0, writes, System.currentTimeMillis() + 10_000).join();
        return ByteBufUtil.getBytes(expectedBuffer);
    }

    private TxDumperFile syncRead(long txn) throws Throwable {
        try {
            return reader.readNext(gen, txn, txn + 1000).join();
        } catch (Throwable e) {
            throw CompletableFutures.unwrapCompletionException(e);
        }
    }
}
