package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName;
import ru.yandex.solomon.dumper.storage.shortterm.file.MemstoreSnapshotFileName;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class KvShortTermStorageDaoTest {

    private KikimrKvClientInMem kvClient;
    private KvShortTermStorageDao dao;
    private long tabletId;

    @Before
    public void setUp() {
        kvClient = new KikimrKvClientInMem();
        tabletId = kvClient.createKvTablet();
        dao = KvShortTermStorageDao.create(kvClient, new MetricRegistry());
    }

    @Test
    public void listEmpty() {
        var result = dao.listFiles(tabletId, 0, NameRange.all()).join();
        assertEquals(0, result.getEntriesAll().length);
    }

    @Test
    public void listFiles() {
        String logFileName = MemstoreSnapshotFileName.format(0, 42, 0, true);
        byte[] logBytes = randomBytes();
        write(logFileName, logBytes);

        var result = dao.listFiles(tabletId, 0, NameRange.all()).join().getEntriesAll();
        assertEquals(1, result.length);
        assertEquals(logFileName, result[0].getName());
        assertEquals(logBytes.length, result[0].getSize());
    }

    @Test
    public void deleteFiles() {
        String logFileName = MemstoreSnapshotFileName.format(0, 42, 0, true);
        byte[] logBytes = randomBytes();
        write(logFileName, logBytes);

        {
            var result = dao.listFiles(tabletId, 0, NameRange.all()).join().getEntriesAll();
            assertEquals(1, result.length);
        }

        dao.deleteFiles(tabletId, 0, List.of(StringMicroUtils.asciiPrefixToRange(MemstoreSnapshotFileName.format(0, 42)))).join();

        {
            var result = dao.listFiles(tabletId, 0, NameRange.all()).join();
            assertEquals(0, result.getEntriesAll().length);
        }
    }

    @Test
    public void readRange() {
        String logFileNameOne = MemstoreSnapshotFileName.format(0, 42, 0, false);
        byte[] logBytesOne = randomBytes();
        write(logFileNameOne, logBytesOne);

        String logFileNameTwo = MemstoreSnapshotFileName.format(0, 42, 1, true);
        byte[] logBytesTwo = randomBytes();
        write(logFileNameTwo, logBytesTwo);

        var range = StringMicroUtils.asciiPrefixToRange(MemstoreSnapshotFileName.format(0, 42));
        var result = dao.readRange(tabletId, 0, range).join();
        assertEquals(2, result.getEntries().length);
        assertEquals(logFileNameOne, result.getEntries()[0].getName());
        assertArrayEquals(logBytesOne, result.getEntries()[0].getValue());
        assertEquals(logFileNameTwo, result.getEntries()[1].getName());
        assertArrayEquals(logBytesTwo, result.getEntries()[1].getValue());
    }

    @Test
    public void renameFile() {
        String memstoreLogFileName = MemstoreSnapshotFileName.format(0, 42, 0, true);
        byte[] logBytes = randomBytes();
        write(memstoreLogFileName, logBytes);

        String dumperLogFileName = DumperLogFileName.format(84, 0, true);
        dao.renameFiles(tabletId, 0, List.of(new KikimrKvClient.Rename(memstoreLogFileName, dumperLogFileName))).join();

        var range = StringMicroUtils.asciiPrefixToRange(DumperLogFileName.format(84));
        var result = dao.readRange(tabletId, 0, range).join();
        assertEquals(1, result.getEntries().length);
        assertEquals(dumperLogFileName, result.getEntries()[0].getName());
        assertArrayEquals(logBytes, result.getEntries()[0].getValue());
    }

    @Test
    public void lock() {
        long gen0 = dao.lock(tabletId).join();
        long gen1 = dao.lock(tabletId).join();
        assertNotEquals(gen0, gen1);

        // list
        {
            assertNull(nullOrError(dao.listFiles(tabletId, gen1, NameRange.all())));
            assertNotNull(nullOrError(dao.listFiles(tabletId, gen0, NameRange.all())));
        }

        // rename
        {
            write("test", new byte[123]);
            var renames = List.of(new KikimrKvClient.Rename("test", "test2"));
            assertNotNull(nullOrError(dao.renameFiles(tabletId, gen0, renames)));
            assertNull(nullOrError(dao.renameFiles(tabletId, gen1, renames)));
        }

        // read
        {
            write("test", new byte[222]);
            assertNotNull(nullOrError(dao.readRange(tabletId, gen0, NameRange.all())));
            assertNull(nullOrError(dao.readRange(tabletId, gen1, NameRange.all())));
        }

        // delete
        {
            write("test", new byte[123]);
            assertNotNull(nullOrError(dao.deleteFiles(tabletId, gen0, List.of(NameRange.all()))));
            assertNull(nullOrError(dao.deleteFiles(tabletId, gen1, List.of(NameRange.all()))));
        }
    }

    private <T> Throwable nullOrError(CompletableFuture<T> future) {
        return future.thenApply(kvEntryStats -> (Throwable) null)
            .exceptionally(e -> e)
            .join();
    }

    private void write(String fileName, byte[] bytes) {
        var expiredAt = System.currentTimeMillis() + 10_000;
        kvClient.write(tabletId, 0, fileName, bytes, MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, MsgbusKv.TKeyValueRequest.EPriority.REALTIME, expiredAt).join();
    }

    private byte[] randomBytes() {
        var random = ThreadLocalRandom.current();
        byte[] bytes = new byte[random.nextInt(256 << 10)];
        random.nextBytes(bytes);
        return bytes;
    }
}
