package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.Sets;
import io.netty.buffer.ByteBufUtil;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.inMem.KikimrKvClientInMem;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EPriority;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EStorageChannel;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.dumper.DumperShardMetrics;
import ru.yandex.solomon.dumper.DumperShardMetricsAggregated;
import ru.yandex.solomon.dumper.storage.shortterm.file.MemstoreSnapshotFileName;
import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static org.hamcrest.Matchers.greaterThan;
import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.assertTrue;


/**
 * @author Vladimir Gordiychuk
 */
public class KvShortTermStorageReaderTest {
    private KikimrKvClientInMem kvClient;
    private long tabletId;
    private KvShortTermStorageDao dao;
    private ManualClock clock;
    private ManualScheduledExecutorService timer;
    private KvShortTermStorageReader reader;

    private Unit alice;
    private Unit bob;

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(30, TimeUnit.SECONDS)
            .build();

    @Before
    public void setUp() {
        kvClient = new KikimrKvClientInMem();
        tabletId = kvClient.createKvTablet();
        dao = KvShortTermStorageDao.create(kvClient, new MetricRegistry(), 1 << 20);
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(2, clock);
        alice = new Unit(0);
        bob = new Unit(1);
        restart(false);
    }

    @After
    public void tearDown() {
        timer.shutdownNow();
        if (reader != null) {
            reader.close();
        }
    }

    @Test
    public void readOneTxn() {
        var data = alice.nextData();
        write(data);
        var file = reader.next().join();
        assertFile(file, data);
    }

    @Test
    public void txnSequentiallyOrdered() {
        long prevTxn = 0;
        for (int index = 0; index < 20; index++) {
            var data = (index % 2) == 0 ? alice.nextData() : bob.nextData();
            write(data);
            var file = reader.next().join();
            assertFile(file, data);
            assertTrue(prevTxn < file.tx.txn);
            prevTxn = file.tx.txn;
        }
    }

    @Test
    public void txnPreparedBeforeRead() throws InterruptedException {
        Data[] expected = IntStream.range(0, 10)
            .mapToObj(ignore -> alice.nextData())
            .toArray(Data[]::new);
        write(expected);
        reader.awaitAct();

        long prevTxn = 0;
        for (int index = 0; index < 10; index++) {
            var data = expected[index];
            write(data);
            var file = reader.next().join();
            assertFile(file, data);
            assertTrue(prevTxn < file.tx.txn);
            prevTxn = file.tx.txn;
        }
    }

    @Test
    public void txnPrepareButNotRead() throws InterruptedException {
        Data[] expected = new Data[20];
        for (int index = 0; index < expected.length; index++) {
            var data = (index % 2) == 0 ? alice.nextData() : bob.nextData();
            expected[index] = data;
            write(data);
            reader.awaitAct();
        }

        reader.awaitAct();
        long prevTxn = 0;
        for (int index = 0; index < 10; index++) {
            var data = expected[index];
            write(data);
            var file = reader.next().join();
            assertFile(file, data);
            assertTrue(prevTxn < file.tx.txn);
            prevTxn = file.tx.txn;
        }
    }

    @Test
    public void readWaitTxn() throws InterruptedException {
        for (int index = 0; index < 3; index++) {
            reader.awaitAct();
        }

        var next = reader.next();
        assertFalse(next.isDone());

        for (int index = 0; index < 2; index++) {
            reader.awaitAct();
        }

        assertFalse(next.isDone());
        var data = alice.nextData();
        write(data);

        reader.awaitAct();
        var file = next.join();
        assertFile(file, data);
    }

    @Test
    public void restartOneTxn() {
        var data = alice.nextData();
        write(data);
        var fileOne = reader.next().join();
        assertFile(fileOne, data);

        restart(true);
        var fileTwo = reader.next().join();
        assertEquals(fileOne.tx, fileTwo.tx);
        assertFile(fileTwo, data);
    }

    @Test
    public void restartFromCommitTxn() throws InterruptedException {
        var one = alice.nextData();
        var two = alice.nextData();
        write(one);
        reader.awaitAct();
        write(two);
        reader.awaitAct();

        var fileOne = reader.next().join();
        assertFile(fileOne, one);

        reader.commit(List.of(fileOne.tx)).join();
        restart(true);

        var fileTwo = reader.next().join();
        assertThat(fileTwo.tx, greaterThan(fileOne.tx));
        assertFile(fileTwo, two);
    }

    @Test
    public void restartWhenNoMorePreparedTxn() {
        var one = alice.nextData();
        write(one);
        var fileOne = reader.next().join();
        assertFile(fileOne, one);
        reader.commit(List.of(fileOne.tx));

        restart(true);
        var two = alice.nextData();
        write(two);

        var fileTwo = reader.next().join();
        assertThat(fileTwo.tx, greaterThan(fileOne.tx));
        assertFile(fileTwo, two);
    }

    @Test
    public void restartContinueTxn() throws InterruptedException {
        var expected = new Data[10];
        {
            int index =0;
            for (; index < 5; index++) {
                var data = alice.nextData();
                write(data);
                expected[index] = data;
                reader.awaitAct();
            }
            restart(true);
            for (; index < 10; index++) {
                var data = alice.nextData();
                write(data);
                expected[index] = data;
                reader.awaitAct();
            }
        }

        long prevTxn = 0;
        for (int index = 0; index < 10; index++) {
            var data = expected[index];
            var file = reader.next().join();
            assertFile(file, data);
            assertTrue(prevTxn < file.tx.txn);
            prevTxn = file.tx.txn;
        }
    }

    @Test
    public void stopActivityWhenNewShardInit() {
        var one = alice.nextData();
        write(one);
        var fileOne = reader.next().join();
        assertFile(fileOne, one);

        var two = alice.nextData();
        write(two);
        var fileTwo = reader.next().join();
        assertFile(fileTwo, two);
        assertThat(fileTwo.tx, greaterThan(fileOne.tx));
        reader.commit(List.of(fileOne.tx)).join();

        var prev = reader;
        restart(false);

        var tree = alice.nextData();
        write(tree);

        var fileTwoAgain = reader.next().join();
        assertEquals(fileTwo.tx, fileTwoAgain.tx);
        assertFile(fileTwoAgain, two);

        var fileTree = reader.next().join();
        assertThat(fileTree.tx, greaterThan(fileTwoAgain.tx));
        assertFile(fileTree, tree);

        assertNotNull(prev.next().thenApply(file -> null).exceptionally(e -> e).join());
    }

    @Test
    public void retryInit() {
        var one = alice.nextData();
        write(one);
        var fileOne = reader.next().join();
        assertFile(fileOne, one);
        reader.commit(List.of(fileOne.tx)).join();

        kvClient.throwOnRead();
        restart(true);

        var two = alice.nextData();
        write(two);
        var nextFuture = reader.next();
        assertFalse(wait(nextFuture, 100, TimeUnit.MILLISECONDS));

        kvClient.resumeReads();
        clock.passedTime(15, TimeUnit.SECONDS);

        var fileTwo = nextFuture.join();
        assertThat(fileTwo.tx, greaterThan(fileOne.tx));
        assertFile(fileTwo, two);
    }

    @Test
    public void retryRead() throws InterruptedException {
        var one = alice.nextData();
        write(one);
        reader.awaitAct();
        kvClient.throwOnRead();
        var nextFuture = reader.next();
        assertFalse(wait(nextFuture, 100, TimeUnit.MILLISECONDS));

        kvClient.resumeReads();
        clock.passedTime(15, TimeUnit.SECONDS);

        var fileOne = nextFuture.join();
        assertFile(fileOne, one);
    }

    @Test
    public void readHugeFiles() throws InterruptedException {
        kvClient.setDoNotExceedFileSizeForTest(1 << 20);
        var one = alice.nextData(25 << 20);
        write(one);
        reader.awaitAct();

        kvClient.throwOnRead();
        var nextFuture = reader.next();

        var resumeFuture = timer.scheduleAtFixedRate(() -> ForkJoinPool.commonPool().execute(() -> kvClient.resumeReads()), 2, 5, TimeUnit.SECONDS);
        var throwFuture = timer.scheduleAtFixedRate(() -> ForkJoinPool.commonPool().execute(() -> kvClient.throwOnRead()), 0, 5, TimeUnit.SECONDS);
        kvClient.throwOnRead();

        while (!wait(nextFuture, 1, TimeUnit.MILLISECONDS)) {
            clock.passedTime(5, TimeUnit.SECONDS);
            clock.passedTime(5, TimeUnit.SECONDS);
        }
        var fileOne = nextFuture.join();
        assertFile(fileOne, one);
        resumeFuture.cancel(false);
        throwFuture.cancel(false);
    }

    @Test
    public void ignoreCorruptedTxn() throws InterruptedException {
        var one = alice.nextData();
        write(one);
        kvClient.deleteFiles(tabletId, 0, List.of(
            "c." + MemstoreSnapshotFileName.format(one.nodeId, one.txn, 0, true),
            "c." + MemstoreSnapshotFileName.format(one.nodeId, one.txn, 0, false)
            ), System.currentTimeMillis() + 10_000)
            .join();

        reader.awaitAct();

        var two = bob.nextData();
        write(two);
        reader.awaitAct();
        reader.awaitAct();

        var fileTwo = reader.next().join();
        assertFile(fileTwo, two);

        var next = reader.next();
        assertFalse(wait(next, 100, TimeUnit.MILLISECONDS));

        var tree = alice.nextData();
        write(tree);
        reader.awaitAct();

        var fileTree = next.join();
        assertThat(fileTree.tx, greaterThan(fileTwo.tx));
        assertFile(fileTree, tree);
    }

    @Test
    public void commitRemoveTxFiles() throws InterruptedException {
        var one = alice.nextData();
        write(one);
        var unpreparedTxn = listFiles(NameRange.all());
        reader.awaitAct();
        var preparedTxn = listFiles(NameRange.all());
        assertEquals("Prepare tnx lead to change name space",
            List.of(), List.copyOf(Sets.intersection(new HashSet<>(unpreparedTxn), new HashSet<>(preparedTxn))));

        var fileOne = reader.next().join();
        assertFile(fileOne, one);
        var uncommit = listFiles(NameRange.all());
        assertEquals("Previous txn stay not touch until commit", preparedTxn, uncommit);
        reader.commit(List.of(fileOne.tx)).join();
        var commit = listFiles(NameRange.all());
        assertEquals("After commit txn files should be removed", List.of("c.d.txn.last"), commit);
    }

    @Test
    public void commitMultipleTxn() throws InterruptedException {
        var one = alice.nextData();
        var two = alice.nextData();
        write(one);
        reader.awaitAct();
        write(two);
        reader.awaitAct();

        var fileOne = reader.next().join();
        assertFile(fileOne, one);

        var fileTwo = reader.next().join();
        assertFile(fileTwo, two);
        assertThat(fileTwo.tx, greaterThan(fileOne.tx));

        reader.commit(List.of(fileOne.tx, fileTwo.tx)).join();
        var commit = listFiles(NameRange.all());
        assertEquals("After commit txn files should be removed", List.of("c.d.txn.last"), commit);
    }

    @Test
    public void validateCommitSequntial() throws InterruptedException {
        var one = alice.nextData();
        var two = alice.nextData();
        write(one);
        reader.awaitAct();
        write(two);
        reader.awaitAct();

        var fileOne = reader.next().join();
        assertFile(fileOne, one);

        var fileTwo = reader.next().join();
        assertFile(fileTwo, two);
        assertThat(fileTwo.tx, greaterThan(fileOne.tx));

        var fileNames = listFiles(NameRange.all());
        Throwable error = reader.commit(List.of(fileTwo.tx))
                .thenApply(ignore -> (Throwable) null)
                .exceptionally(throwable -> throwable)
                .join();
        assertNotNull(error);
        assertEquals("All files should be the same because commit invalid order", fileNames, listFiles(NameRange.all()));

        reader.commit(List.of(fileOne.tx, fileTwo.tx)).join();
        var commit = listFiles(NameRange.all());
        assertEquals("After commit txn files should be removed", List.of("c.d.txn.last"), commit);
    }

    @Test
    public void ignoreEmptyLatestTxnFile() {
        kvClient.write(tabletId, 0, "c.d.txn.last", new byte[0], EStorageChannel.MAIN, EPriority.REALTIME, 0).join();
        restart(false);

        var one = alice.nextData();
        write(one);

        var fileOne = reader.next().join();
        assertFile(fileOne, one);
    }

    private List<String> listFiles(NameRange range) {
        return kvClient.readRangeNames(tabletId, 0, range, 0)
            .thenApply(response -> response.stream()
                .map(KikimrKvClient.KvEntryStats::getName)
                .collect(Collectors.toList()))
            .join();
    }

    private <T> boolean wait(CompletableFuture<T> future, long timeout, TimeUnit unit) {
        try {
            future.get(timeout, unit);
            return true;
        } catch (InterruptedException|TimeoutException|ExecutionException e) {
            return false;
        }
    }

    private void restart(boolean stopPrev) {
        if (reader != null && stopPrev) {
            reader.close();
        }
        var metrics = new DumperShardMetrics("42", new DumperShardMetricsAggregated());
        reader = new KvShortTermStorageReader(0, 42, tabletId, 0, metrics, dao, ForkJoinPool.commonPool(), timer);
        System.out.println("restart reader");
    }

    private void assertFile(DumperFile file, Data expected) {
        try {
            assertArrayEquals(expected.content, ByteBufUtil.getBytes(file.content));
        } finally {
            file.release();
        }
    }

    private void write(Data... sources) {
        var random = ThreadLocalRandom.current();
        long expiredAt = System.currentTimeMillis() + 30_000;
        List<KikimrKvClient.Write> writes = new ArrayList<>();
        for (var source : sources) {
            var log = ByteStringsStockpile.unsafeWrap(source.content);
            var logChunks = ByteStringsStockpile.split(log, (256 << 10) + Math.min(random.nextInt(log.size()), 1 << 20));
            for (int chunkNo = 0; chunkNo < logChunks.length; chunkNo++) {
                boolean last = chunkNo + 1 == logChunks.length;
                var file = "c." + MemstoreSnapshotFileName.format(source.nodeId, source.txn, chunkNo, last);
                writes.add(new KikimrKvClient.Write(file, logChunks[chunkNo], MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN, MsgbusKv.TKeyValueRequest.EPriority.REALTIME));
            }
        }

        kvClient.writeMulti(tabletId, 0, writes, expiredAt).join();
    }

    private class Unit {
        private int nodeId;
        private AtomicLong latestTxn = new AtomicLong(ThreadLocalRandom.current().nextInt(1000));

        public Unit(int nodeId) {
            this.nodeId = nodeId;
        }

        public Data nextData() {
            var random = ThreadLocalRandom.current();
            return nextData(1 + random.nextInt(1 << 20));
        }

        public Data nextData(int logSize) {
            var random = ThreadLocalRandom.current();
            var content = new byte[logSize];

            random.nextBytes(content);
            return new Data(nodeId, latestTxn.incrementAndGet(), content);
        }
    }

    private static class Data {
        private int nodeId;
        private long txn;
        private byte[] content;

        public Data(int nodeId, long txn, byte[] content) {
            this.nodeId = nodeId;
            this.txn = txn;
            this.content = content;
        }
    }
}
