package ru.yandex.solomon.dumper.storage.shortterm;

import java.util.concurrent.ThreadLocalRandom;

import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import org.junit.Test;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;

/**
 * @author Vladimir Gordiychuk
 */
public class TxDumperFileListTest {

    @Test(expected = IllegalStateException.class)
    public void absentFirstChunk() {
        try (var list = new TxDumperFileList(1, 42)) {
            list.append("00000000000000042.000001z", bytes(), now());
        }
    }

    @Test(expected = IllegalStateException.class)
    public void absentChunk() {
        try (var list = new TxDumperFileList(1, 42)) {
            list.append("00000000000000042.000000y", bytes(), now());
            list.append("00000000000000042.000002z", bytes(), now());
        }
    }

    @Test(expected = IllegalStateException.class)
    public void absentTxn() {
        try (var list = new TxDumperFileList(1, 42)) {
            list.append("00000000000000043.000000y", bytes(), now());
        }
    }

    @Test(expected = IllegalArgumentException.class)
    public void fileNameNotValid() {
        try (var list = new TxDumperFileList(1, 42)) {
            list.append("0000000hi00000043.000000y", bytes(), now());
        }
    }

    @Test(expected = IllegalStateException.class)
    public void absentTxInMiddle() {
        try (var list = new TxDumperFileList(1, 42)) {
            list.append("00000000000000042.000000z", bytes(), now());
            list.append("00000000000000045.000000z", bytes(), now());
        }
    }

    @Test(expected = IllegalStateException.class)
    public void notFinished() {
        try (var list = new TxDumperFileList(1, 42)) {
            list.append("00000000000000042.000000y", bytes(), now());
            list.ensureLatestFinished();
        }
    }

    @Test(expected = IllegalStateException.class)
    public void empty() {
        try (var list = new TxDumperFileList(1, 42)) {
            list.ensureNotEmpty();
        }
    }

    @Test(expected = IllegalStateException.class)
    public void txnEndChunksAbsent() {
        try (var list = new TxDumperFileList(1, 45)) {
            list.append("00000000000000045.000000y", bytes(), now());
            list.append("00000000000000045.000001y", bytes(), now());
            list.append("00000000000000046.000000z", bytes(), now());
        }
    }

    @Test
    public void one() {
        var bytes = bytes();
        var now = now();
        try (var list = new TxDumperFileList(1, 42)) {
            list.append("00000000000000042.000000z", bytes, now);
            list.ensureLatestFinished();
            list.ensureNotEmpty();
            assertNull(list.lastNotFinished);
            assertEquals(1, list.files.size());

            var file = list.files.get(0);
            assertEquals(42, file.txn);
            assertEquals(now, file.createdAt);
            assertEquals(1, file.chunksCount);
            assertArrayEquals(bytes, ByteBufUtil.getBytes(file.buffer));
        }
    }

    @Test
    public void oneMultipleChunks() {
        var one = bytes();
        var two = bytes();
        var tree = bytes();
        var now = now();
        try (var list = new TxDumperFileList(1, 45)) {
            list.append("00000000000000045.000000y", one, now);
            list.append("00000000000000045.000001y", two, now);
            list.append("00000000000000045.000002z", tree, now);
            list.ensureLatestFinished();
            list.ensureNotEmpty();
            assertNull(list.lastNotFinished);
            assertEquals(1, list.files.size());

            var file = list.files.get(0);
            assertEquals(45, file.txn);
            assertEquals(now, file.createdAt);
            assertEquals(3, file.chunksCount);
            assertArrayEquals(ByteBufUtil.getBytes(Unpooled.wrappedBuffer(one, two, tree)), ByteBufUtil.getBytes(file.buffer));
        }
    }

    @Test
    public void fewSmallFiles() {
        var one = bytes();
        var two = bytes();
        var tree = bytes();
        var now = now();
        try (var list = new TxDumperFileList(1, 45)) {
            list.append("00000000000000045.000000z", one, now + 1);
            list.append("00000000000000046.000000z", two, now + 2);
            list.append("00000000000000047.000000z", tree, now + 3);
            list.ensureLatestFinished();
            list.ensureNotEmpty();
            assertNull(list.lastNotFinished);
            assertEquals(3, list.files.size());

            {
                var file = list.files.get(0);
                assertEquals(45, file.txn);
                assertEquals(now + 1, file.createdAt);
                assertEquals(1, file.chunksCount);
                assertArrayEquals(one, ByteBufUtil.getBytes(file.buffer));
            }

            {
                var file = list.files.get(1);
                assertEquals(46, file.txn);
                assertEquals(now + 2, file.createdAt);
                assertEquals(1, file.chunksCount);
                assertArrayEquals(two, ByteBufUtil.getBytes(file.buffer));
            }

            {
                var file = list.files.get(2);
                assertEquals(47, file.txn);
                assertEquals(now + 3, file.createdAt);
                assertEquals(1, file.chunksCount);
                assertArrayEquals(tree, ByteBufUtil.getBytes(file.buffer));
            }
        }
    }

    @Test
    public void fewHuge() {
        var aliceOne = bytes();
        var aliceTwo = bytes();
        var bobOne = bytes();
        var bobTwo = bytes();
        var now = now();
        try (var list = new TxDumperFileList(1, 45)) {
            list.append("00000000000000045.000000y", aliceOne, now + 1);
            list.append("00000000000000045.000001z", aliceTwo, now + 1);
            list.append("00000000000000046.000000y", bobOne, now + 2);
            list.append("00000000000000046.000001z", bobTwo, now + 2);
            list.ensureLatestFinished();
            list.ensureNotEmpty();
            assertNull(list.lastNotFinished);
            assertEquals(2, list.files.size());

            {
                var file = list.files.get(0);
                assertEquals(45, file.txn);
                assertEquals(now + 1, file.createdAt);
                assertEquals(2, file.chunksCount);
                assertArrayEquals(ByteBufUtil.getBytes(Unpooled.wrappedBuffer(aliceOne, aliceTwo)), ByteBufUtil.getBytes(file.buffer));
            }

            {
                var file = list.files.get(1);
                assertEquals(46, file.txn);
                assertEquals(now + 2, file.createdAt);
                assertEquals(2, file.chunksCount);
                assertArrayEquals(ByteBufUtil.getBytes(Unpooled.wrappedBuffer(bobOne, bobTwo)), ByteBufUtil.getBytes(file.buffer));
            }
        }
    }

    private long now() {
        return System.currentTimeMillis() / 1000;
    }

    private byte[] bytes() {
        byte[] result = new byte[128];
        ThreadLocalRandom.current().nextBytes(result);
        return result;
    }
}
