package ru.yandex.solomon.dumper.storage.shortterm.file;

import java.util.List;

import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName.chunkNo;
import static ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName.ensureFullNameValid;
import static ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName.format;
import static ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName.isLastChunk;
import static ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName.txn;
import static ru.yandex.solomon.dumper.storage.shortterm.file.DumperLogFileName.withoutChunkNo;

/**
 * @author Vladimir Gordiychuk
 */
public class DumperLogFileNameTest {

    @Test
    public void formatNoChunkNo() {
        assertEquals("d.l.00000000000000042", format(42));
    }

    @Test
    public void formatWithChunkNo() {
        assertEquals("d.l.00000000000000042.000002y", format(42, 2, false));
        assertEquals("d.l.00000000000000042.000003z", format(42, 3, true));
    }

    @Test
    public void cutChunkNo() {
        assertEquals(format(42), withoutChunkNo(format(42, 0, true)));
    }

    @Test
    public void ensureValid() {
        ensureFullNameValid("00000000000000042.000002y");
        ensureFullNameValid("00000000000000042.000003z");
        ensureFullNameValid("00000000000011143.000001y");
        ensureFullNameValid("00000000000011143.000001y");
    }

    @Test
    public void ensureNotValid() {
        var names = List.of(
            "",
            "hi",
            "00000000000011143.00000y",
            "00000000000011143.000001",
            "0000000000011143.000001y",
            "00000000000011143000001y"
        );

        for (var name : names) {
            try {
                ensureFullNameValid(name);
                fail("Expected fail because name is not valid: " + name);
            } catch (IllegalArgumentException e) {
                // ok
            }
        }
    }

    @Test
    public void lastCheck() {
        for (String prefix : List.of("", "d.l.", "c.d.l.")) {
            assertFalse(prefix, isLastChunk(prefix + "00000000000000042.000002y"));
            assertTrue(prefix, isLastChunk(prefix + "00000000000000042.000002z"));
        }
    }

    @Test
    public void getChunkNo() {
        for (String prefix : List.of("", "d.l.", "c.d.l.")) {
            assertEquals(prefix, 2, chunkNo(prefix + "00000000000000042.000002y"));
            assertEquals(prefix, 3, chunkNo(prefix + "00000000000000042.000003z"));
            assertEquals(prefix, 123456, chunkNo(prefix + "00000000000000043.123456z"));
        }
    }

    @Test
    public void getTxn() {
        for (String prefix : List.of("", "d.l.", "c.d.l.")) {
            assertEquals(prefix, 42, txn(prefix + "00000000000000042.000002y"));
            assertEquals(prefix, 42, txn(prefix + "00000000000000042.000003z"));
            assertEquals(prefix, 43, txn(prefix + "00000000000000043.123456z"));
            assertEquals(prefix, 123456789000L, txn(prefix + "00000123456789000.000000z"));
            assertEquals(prefix, 12345678901234567L, txn(prefix + "12345678901234567.000000z"));
        }
    }
}
