#include "cloud_iam.h"

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

using namespace NActors;


namespace NSolomon::NFetcher {
namespace {
    class TExecutorActor final: public TActorBootstrapped<TExecutorActor> {
    public:
        TExecutorActor(std::function<void(void)>&& f, TDuration delay, TDuration interval)
            : Func_(std::move(f))
            , Delay_{delay}
            , Interval_{interval}
        {
            Y_VERIFY_DEBUG(Interval_ > TDuration::Zero());
        }

        void Bootstrap(const TActorContext&) {
            Become(&TThis::StateWork);
            Schedule(Delay_, new TEvents::TEvWakeup);
        }

        STFUNC(StateWork) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                cFunc(TEvents::TSystem::Wakeup, OnWakeup);
                cFunc(TEvents::TSystem::PoisonPill, PassAway);
            }
        }

        void OnWakeup() {
            Func_();
            Schedule(Interval_, new TEvents::TEvWakeup);
        }

    private:
        const std::function<void(void)> Func_;
        TDuration Delay_;
        TDuration Interval_;
    };

    class TActorExecutor final: public NCloud::IExecutor {
    public:
        TActorExecutor(TActorSystem& as)
            : ActorSystem_{as}
        {
        }

        ~TActorExecutor() {
            Stop();
        }

        void Schedule(std::function<void(void)> fn, TDuration delay, TDuration interval) override {
            Actor_ = ActorSystem_.Register(new TExecutorActor{std::move(fn), delay, interval});
        }

        void Stop() override {
            if (std::atomic_fetch_or(&Stopped_, char(1))) {
                return;
            }

            ActorSystem_.Send(Actor_, new TEvents::TEvPoisonPill());
        }

    private:
        std::atomic_char Stopped_{0};
        TActorSystem& ActorSystem_;
        TActorId Actor_;
    };

    class TActorLogger final: public NCloud::ILogger {
    public:
        TActorLogger(TActorSystem& as)
            : ActorSystem_{as}
        {
        }

        void Write(ELogPriority severity, TString msg) override {
            MON_LOG_C(ActorSystem_, IamClient, NLog::EPriority(severity), msg);
        }

    private:
        TActorSystem& ActorSystem_;
    };
}

    THolder<NCloud::IExecutor> CreateActorExecutor(TActorSystem& as) {
        return MakeHolder<TActorExecutor>(as);
    }

    NCloud::ILoggerPtr CreateActorLogger(TActorSystem& as) {
        return MakeIntrusive<TActorLogger>(as);
    }
} // namespace NSolomon::NFetcher
