#include "cluster.h"

#include <solomon/services/fetcher/lib/app_data.h>
#include <solomon/services/fetcher/lib/cluster/cluster.h>
#include <solomon/services/fetcher/lib/data_sink/data_sink.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/string_utils/url/url.h>
#include <library/cpp/threading/future/future.h>

#include <util/system/hostname.h>
#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/string/cast.h>

using namespace NActors;
using namespace NThreading;
using namespace NMonitoring;

namespace NSolomon::NFetcher {
namespace {

constexpr i64 DEFAULT_SINK_MEM_LIMIT = 10_GB;
constexpr i64 DEFAULT_NON_LOCAL_SINK_MEM_LIMIT = 1_GB;
    TString FormatEndpoint(TStringBuf host, ui16 port) {
        TStringBuilder sb;
        sb << host;

        if (port != 0) {
            sb << ":" << port;
        }

        return sb;
    }

    class TRpcSinkFactoryActor: public TActorBootstrapped<TRpcSinkFactoryActor> {
    public:
        TRpcSinkFactoryActor(
                TPromise<THashMap<TClusterNode, TActorId>> promise,
                std::shared_ptr<IMetricRegistry> metrics,
                std::shared_ptr<IMetricRegistry> shardMetrics,
                const THashSet<TClusterNode>& locations,
                IProcessingClusterClientPtr client,
                i64 localMemLimit,
                size_t executorPool)
            : Promise_{std::move(promise)}
            , Metrics_{std::move(metrics)}
            , ShardMetrics_{std::move(shardMetrics)}
            , Locations_{locations}
            , Client_{std::move(client)}
            , MemLimit_{localMemLimit}
            , ExecutorPool_{executorPool}
        {
        }

        void Bootstrap() {
            THashMap<TClusterNode, TActorId> sinks;

            auto localMemLimit = MemLimit_
                ? MemLimit_
                : DEFAULT_SINK_MEM_LIMIT;

            const TAppData& appData = GetAppData();
            for (auto& loc: Locations_) {
                const auto memLimit = IsLocal(loc)
                    ? localMemLimit
                    : DEFAULT_NON_LOCAL_SINK_MEM_LIMIT;

                TCoremonSinkConf conf{
                    .ShardMetrics = ShardMetrics_,
                    .MemLimit = memLimit,
                    .MetricVerbosity = appData.MetricVerbosity(),
                };

                auto client = Client_->GetClient(loc);
                Y_ENSURE(client, "Cannot create client for " << loc);
                auto* sink = CreateDataSink(
                    CreateShardWriterFactory(client, conf),
                    Metrics_->Counter(MakeLabels({{"sensor", "coremonSink.rejectedOnMemoryOverflow"}})),
                    ExecutorPool_
                );

                auto sinkId = this->RegisterWithSameMailbox(sink);
                sinks.emplace(loc, sinkId);
            }

            Promise_.SetValue(std::move(sinks));
            PassAway();
        }

    private:
        TPromise<THashMap<TClusterNode, TActorId>> Promise_;
        std::shared_ptr<IMetricRegistry> Metrics_;
        std::shared_ptr<IMetricRegistry> ShardMetrics_;
        const THashSet<TClusterNode>& Locations_;
        IProcessingClusterClientPtr Client_;
        i64 MemLimit_{};
        size_t ExecutorPool_{};
    };
} // namespace

    const TVector<TString>& TFetcherCluster::Endpoints() const {
        return Endpoints_;
    }

    TFetcherCluster::TFetcherCluster() = default;
    TFetcherCluster::~TFetcherCluster() = default;

    TFetcherCluster::TFetcherCluster(TClusterMapBase clusterMap)
        : TClusterMapBase{std::move(clusterMap)}
    {
    }

    TIntrusivePtr<TFetcherCluster> TFetcherCluster::FromString(TStringBuf data) {
        auto c = TClusterMapBase::Load(data);
        auto result = MakeIntrusive<TFetcherCluster>(std::move(*c));

        for (auto&& node: result->Nodes_) {
            result->Endpoints_.emplace_back(FormatEndpoint(node.Fqdn, node.Port));
        }

        return result;
    }

    THashMap<TClusterNode, TActorId> CreateRpcSinks(
        TActorRuntime& runtime,
        std::shared_ptr<IMetricRegistry> metrics,
        std::shared_ptr<IMetricRegistry> shardMetrics,
        const THashSet<TClusterNode>& locations,
        IProcessingClusterClientPtr rpcClusterClient,
        i64 localMemLimit,
        size_t executorPool)
    {
        // In order to ensure that metric recycler and all the sink actors work
        // on a single mailbox (i.e. share single thread of execution) we have to
        // create them all from within the actor system context
        auto promise = NewPromise<THashMap<TClusterNode, TActorId>>();
        runtime.Register(new TRpcSinkFactoryActor{
            promise,
            std::move(metrics),
            std::move(shardMetrics),
            locations,
            std::move(rpcClusterClient),
            localMemLimit,
            executorPool
        });

        return promise.GetFuture().ExtractValueSync();
    }

    EOperationMode OperationModeFromProto(TFetcherConfig::EOperationMode proto) {
        switch (proto) {
            case TFetcherConfig::GLOBAL:
                return EOperationMode::Global;
            case TFetcherConfig::LOCAL:
                return EOperationMode::Local;
            case TFetcherConfig::LOCAL_TEST:
                return EOperationMode::LocalTest;
            default:
                ythrow yexception() << "Unsupported operation mode " << TFetcherConfig::EOperationMode_Name(proto);
        }
    }
} // namespace NSolomon::NFetcher

template<>
void Out<NSolomon::NFetcher::TFetcherCluster>(IOutputStream& os, const NSolomon::NFetcher::TFetcherCluster& c) {
    os << static_cast<const NSolomon::IClusterMap&>(c);
}
