#include "http.h"

#include <solomon/libs/cpp/coordination/load_balancer/load_balancer.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <solomon/services/fetcher/lib/app_data.h>
#include <solomon/services/fetcher/lib/router/router.h>

#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/html/escape/escape.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/stream/format.h>
#include <util/system/hostname.h>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon;
using namespace NCoordination;
using namespace std::string_view_literals;

namespace NSolomon::NFetcher {
namespace {

using ::yandex::monitoring::selfmon::Page;

class TBalancerPageRequest: public NActors::TActor<TBalancerPageRequest> {
public:
    TBalancerPageRequest()
        : NActors::TActor<TBalancerPageRequest>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TLoadBalancerEvents::TEvDescribeResponse, OnResult);
        }
    }

    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        Send(MakeLoadBalancerId(), new TLoadBalancerEvents::TEvDescribeRequest);
    }

    void OnResult(TLoadBalancerEvents::TEvDescribeResponse::TPtr ev) {
        Page page;
        page.set_title("Load balancer state");
        auto* grid = page.mutable_grid();
        auto* tableCluster = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
        tableCluster->set_numbered(true);

        auto* leaderColumn = tableCluster->add_columns();
        leaderColumn->set_title("Leader");
        auto* leaderValues = leaderColumn->mutable_string();

        auto* hostNameColumn = tableCluster->add_columns();
        hostNameColumn->set_title("HostName");
        auto* hostNameValues = hostNameColumn->mutable_string();

        auto* nodeIDColumn = tableCluster->add_columns();
        nodeIDColumn->set_title("Node ID");
        auto* nodeIDValues = nodeIDColumn->mutable_uint32();

        auto* stateColumn = tableCluster->add_columns();
        stateColumn->set_title("State");
        auto* stateValues = stateColumn->mutable_string();

        for (auto&& node: ev->Get()->ClusterState.Nodes()) {
            if (ev->Get()->ClusterState.Leader().NodeId == node.NodeId()) {
                leaderValues->add_values("Leader");
            } else {
                leaderValues->add_values("Follower");
            }
            hostNameValues->add_values(node.Hostname());
            nodeIDValues->add_values(node.NodeId());
            TString warning;
            if (!node.IsOk()) {
                warning = " WARNING";
            }
            stateValues->add_values(TStringBuilder{} << node.State() << warning);
        }

        auto* tableAssignments = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
        tableAssignments->set_numbered(true);

        auto* fromColumn = tableAssignments->add_columns();
        fromColumn->set_title("From");
        auto* fromValues = fromColumn->mutable_uint64();

        auto* toColumn = tableAssignments->add_columns();
        toColumn->set_title("To");
        auto* toValues = toColumn->mutable_uint64();

        for (auto&& slice: ev->Get()->CurrentAssignments.Slices()) {
            fromValues->add_values(slice.first);
            toValues->add_values(slice.second);
        }
        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    NActors::TActorId ReplyTo_;
};

class TBalancerPage: public NActors::TActor<TBalancerPage> {
public:
    explicit TBalancerPage(bool balancerEnabled)
        : NActors::TActor<TBalancerPage>(&TThis::StateFunc)
        , BalancerEnabled_(balancerEnabled)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        if (!BalancerEnabled_) {
            Page page;
            page.set_title("Load balancer state");
            page.mutable_component()->mutable_code()->set_content("Balancer is not enabled");
            Send(ev->Sender, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        } else {
            TActorId reqId = Register(new TBalancerPageRequest);
            TActivationContext::Send(ev->Forward(reqId));
        }
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    const bool BalancerEnabled_;
};

class TRouterShardsPageRequest: public NActors::TActor<TRouterShardsPageRequest> {
public:
    TRouterShardsPageRequest()
        : NActors::TActor<TRouterShardsPageRequest>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TEvRouterStateShardsResponse, OnResult);
        }
    }

    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        TryFromString(ev->Get()->Param("limit"), Limit_);
        Send(MakeRouterServiceId(), new TEvRouterStateShardsRequest);
    }

    void OnResult(TEvRouterStateShardsResponse::TPtr ev) {
        Page page;
        page.set_title("Router shards");
        auto* grid = page.mutable_grid();
        auto* ref = grid->add_rows()->add_columns()->mutable_component()->mutable_value()->mutable_reference();
        ref->set_page("routerShards");
        if (Limit_ != -1) {
            ref->set_title("Show all " + ToString(ev->Get()->Shards.size()));
            ref->set_args("limit=-1");
        } else {
            ref->set_title("Show 20");
            ref->set_args("limit=20");
        }
        auto* table = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
        table->set_numbered(true);

        auto* shardNumIdColumn = table->add_columns();
        shardNumIdColumn->set_title("Shard numeric Id");
        auto* shardNumIdValues = shardNumIdColumn->mutable_uint32();

        auto* shardStrIdColumn = table->add_columns();
        shardStrIdColumn->set_title("Shard string Id");
        auto* shardStrIdValues = shardStrIdColumn->mutable_string();

        auto* nodeIdColumn = table->add_columns();
        nodeIdColumn->set_title("Node Id");
        auto* nodeIdValues = nodeIdColumn->mutable_int32();

        auto* endpointColumn = table->add_columns();
        endpointColumn->set_title("Endpoint");
        auto* endpointValues = endpointColumn->mutable_string();

        i64 row = 0;
        for (auto&& shard: ev->Get()->Shards) {
            if (Limit_ != -1 && ++row > Limit_) {
                break;
            }
            shardNumIdValues->add_values(shard.Id.NumId());
            shardStrIdValues->add_values(shard.Id.StrId());
            nodeIdValues->add_values(shard.Location.NodeId);
            endpointValues->add_values(shard.Location.Endpoint);
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    i64 Limit_{20};
    NActors::TActorId ReplyTo_;
};

class TRouterShardsPage: public NActors::TActor<TRouterShardsPage> {
public:
    TRouterShardsPage()
        : NActors::TActor<TRouterShardsPage>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        TActorId reqId = Register(new TRouterShardsPageRequest);
        TActivationContext::Send(ev->Forward(reqId));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }
};

class TBacklogsPageRequest: public NActors::TActor<TBacklogsPageRequest> {
public:
    TBacklogsPageRequest()
        : NActors::TActor<TBacklogsPageRequest>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TEvRouterStateBacklogsResponse, OnResult);
        }
    }

    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        Send(MakeRouterServiceId(), new TEvRouterStateBacklogsRequest);
    }

    void OnResult(TEvRouterStateBacklogsResponse::TPtr ev) {
        Page page;
        page.set_title("Router backlogs");
        auto* table = page.mutable_component()->mutable_table();
        table->set_numbered(true);

        auto* shardNumIdColumn = table->add_columns();
        shardNumIdColumn->set_title("Shard numeric Id");
        auto* shardNumIdValues = shardNumIdColumn->mutable_uint32();

        auto* shardStrIdColumn = table->add_columns();
        shardStrIdColumn->set_title("Shard string Id");
        auto* shardStrIdValues = shardStrIdColumn->mutable_string();

        auto* entrySizeColumn = table->add_columns();
        entrySizeColumn->set_title("Entry Size");
        auto* entrySizeValues = entrySizeColumn->mutable_uint64();

        auto* memColumn = table->add_columns();
        memColumn->set_title("Memory");
        auto* memValues = memColumn->mutable_data_size();

        auto* stateColumn = table->add_columns();
        stateColumn->set_title("State");
        auto* stateValues = stateColumn->mutable_string();

        for (auto&& backlog: ev->Get()->Backlogs) {
            shardNumIdValues->add_values(backlog.ShardId.NumId());
            shardStrIdValues->add_values(backlog.ShardId.StrId());
            entrySizeValues->add_values(backlog.EntrySize);
            memValues->add_values(static_cast<double>(backlog.MemorySize));
            stateValues->add_values((backlog.IsFull ? "full" : "not full"));
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    NActors::TActorId ReplyTo_;
};

class TBacklogsPage: public NActors::TActor<TBacklogsPage> {
public:
    TBacklogsPage()
        : NActors::TActor<TBacklogsPage>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        TActorId reqId = Register(new TBacklogsPageRequest);
        TActivationContext::Send(ev->Forward(reqId));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }
};

class TPeersPageRequest: public NActors::TActor<TPeersPageRequest> {
public:
    TPeersPageRequest()
        : NActors::TActor<TPeersPageRequest>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(TEvRouterStatePeersResponse, OnResult);
        }
    }

    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        ReplyTo_ = ev->Sender;
        Send(MakeRouterServiceId(), new TEvRouterStatePeersRequest);
    }

    void OnResult(TEvRouterStatePeersResponse::TPtr ev) {
        Page page;
        page.set_title("Router peers");
        auto* table = page.mutable_component()->mutable_table();
        table->set_numbered(true);

        auto* nodeIdColumn = table->add_columns();
        nodeIdColumn->set_title("Node Id");
        auto* nodeIdValues = nodeIdColumn->mutable_int32();

        auto* endpointColumn = table->add_columns();
        endpointColumn->set_title("Endpoint");
        auto* endpointValues = endpointColumn->mutable_string();

        for (auto&& peer: ev->Get()->Peers) {
            nodeIdValues->add_values(peer.NodeId);
            endpointValues->add_values(peer.Endpoint);
        }

        Send(ReplyTo_, new NSolomon::NSelfMon::TEvPageDataResp{std::move(page)});
        PassAway();
    }

private:
    NActors::TActorId ReplyTo_;
};

class TPeersPage: public NActors::TActor<TPeersPage> {
public:
    TPeersPage()
        : NActors::TActor<TPeersPage>(&TThis::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSolomon::NSelfMon::TEvPageDataReq, OnRequest);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(NSolomon::NSelfMon::TEvPageDataReq::TPtr ev) {
        TActorId reqId = Register(new TPeersPageRequest);
        TActivationContext::Send(ev->Forward(reqId));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }
};

} // namespace;

void AddFetcherSpecificHttpHandlers(NActors::TActorSystem& actorSystem, bool balancerEnabled) {
    NSelfMon::RegisterPage(actorSystem, "/balancer", "Balancer", std::make_unique<TBalancerPage>(balancerEnabled));
    NSelfMon::RegisterPage(actorSystem, "/routerPeers", "Router peers", std::make_unique<TPeersPage>());
    NSelfMon::RegisterPage(actorSystem, "/routerBacklogs", "Router backlogs", std::make_unique<TBacklogsPage>());
    NSelfMon::RegisterPage(actorSystem, "/routerShards", "Router shards", std::make_unique<TRouterShardsPage>());
}

} // namespace NSolomon::NFetcher
