#include "detail/cloud_iam.h"
#include "detail/cluster.h"
#include "detail/http.h"
#include "detail/agent_cleaner.h"

#include <solomon/libs/cpp/actors/metrics/actor_runtime_metrics.h>
#include <solomon/libs/cpp/auth/core/internal_authorizer.h>
#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/actors/scheduler/scheduler.h>
#include <solomon/libs/cpp/clients/coremon/coremon_client.h>
#include <solomon/libs/cpp/grpc/server/server.h>
#include <solomon/libs/cpp/http/server/core/http_server.h>
#include <solomon/libs/cpp/http/server/handlers/metrics.h>
#include <solomon/libs/cpp/http/server/handlers/version.h>
#include <solomon/libs/cpp/clients/ingestor/ingestor_client.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/selfmon/service/proto_page.h>
#include <solomon/libs/cpp/selfmon/service/service.h>
#include <solomon/libs/cpp/solomon_env/solomon_env.h>
#include <solomon/libs/cpp/cloud/envoy/endpoint_v3_rpc.h>
#include <solomon/libs/cpp/minidump/minidump.h>

#include <solomon/services/fetcher/lib/api/api.h>
#include <solomon/services/fetcher/lib/clients/access_service.h>
#include <solomon/services/fetcher/lib/config/config.h>
#include <solomon/services/fetcher/lib/config_updater/config_updater.h>
#include <solomon/services/fetcher/lib/data_sink/data_sink.h>
#include <solomon/services/fetcher/lib/dns/continuous_resolver.h>
#include <solomon/services/fetcher/lib/host_list_cache/host_list_cache.h>
#include <solomon/services/fetcher/lib/load_balancer/load_balancer.h>
#include <solomon/services/fetcher/lib/multishard_pulling/auth_gatekeeper.h>
#include <solomon/services/fetcher/lib/racktables/racktables_actor.h>
#include <solomon/services/fetcher/lib/router/router.h>
#include <solomon/services/fetcher/lib/router/unknown_shard.h>
#include <solomon/services/fetcher/lib/shard_manager/shard_manager.h>
#include <solomon/services/fetcher/lib/shard_manager/shard_resolver.h>
#include <solomon/services/fetcher/lib/sink/sink.h>
#include <solomon/services/fetcher/lib/yasm/itype_white_list.h>

#include <solomon/protos/configs/rpc/rpc_config.pb.h>

#include <solomon/libs/cpp/auth/tvm/tvm.h>
#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/config_includes/config_includes.h>
#include <solomon/libs/cpp/grpc/executor/limits.h>
#include <solomon/libs/cpp/secrets/secrets.h>
#include <solomon/libs/cpp/ydb/driver.h>

#include <ydb/public/lib/jwt/jwt.h>

#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/process_stats.h>
#include <library/cpp/getoptpb/getoptpb.h>
#include <library/cpp/grpc/server/grpc_server.h>
#include <library/cpp/protobuf/util/pb_io.h>
#include <library/cpp/sighandler/async_signals_handler.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/generic/size_literals.h>
#include <util/string/join.h>
#include <util/system/event.h>
#include <util/system/getpid.h>
#include <util/system/hostname.h>

using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NDb;
using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NAuth::NTvm;

namespace NRpc = yandex::solomon::config::rpc;

constexpr auto DEFAULT_CONNECT_TIMEOUT = TDuration::Seconds(5);

struct TDao {
    TDao() = default;
    // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
    TDao(NYdb::TDriver driver, const NDb::TSessionPoolConfig& conf, TMetricRegistry& registry, const TString& pathPrefix)
        : DbConnection{CreateYdbConnection(std::move(driver), conf, registry)}
        , Service{DbConnection->CreateServiceDao({pathPrefix + "Service"})}
        , Shard{DbConnection->CreateShardDao(TShardTables{
            .ShardTablePath = pathPrefix + "Shard",
            .NumPcsTablePath = pathPrefix + "ShardPcsKey",
            .NumIdPcsTablePath = pathPrefix + "ShardPcsNumId",
        })
        }
        , Cluster{DbConnection->CreateClusterDao(pathPrefix + "Cluster")}
        , Agent{DbConnection->CreateAgentDao({pathPrefix + "Agent"})}
        , Provider{DbConnection->CreateProviderDao({pathPrefix + "ServiceProvider"})}
    {
        Y_ENSURE(!pathPrefix.empty() && pathPrefix.back() == '/', "Wrong path prefix format");
    }

    IDbConnectionPtr DbConnection;

    IServiceConfigDaoPtr Service;
    IShardConfigDaoPtr Shard;
    IClusterConfigDaoPtr Cluster;
    IAgentConfigDaoPtr Agent;
    IProviderConfigDaoPtr Provider;
};

IHostListCachePtr CreateHostListCache(const TFetcherConfig& conf) {
    if (!conf.HasHostListCacheConfig()) {
        return nullptr;
    }

    auto&& cacheConf = conf.GetHostListCacheConfig();
    switch (cacheConf.GetType()) {
        case THostListCacheConfig::YDB:
            Y_ENSURE(cacheConf.HasYdbConfig(), "YdbConfig must be provided for the YDB cache type");
            Y_ENSURE(!cacheConf.GetPath().empty(), "Path cannot be empty for YDB cache type");
            return CreateYdbCache({
                .YdbConfig = cacheConf.GetYdbConfig(),
                .Path = cacheConf.GetPath(),
            });
        case THostListCacheConfig::MEM:
            return CreateMemCache();
        default:
            Y_FAIL();
    }
}

class TFetcherApp {
    enum class EIngestorAssignmentsMode {
        FromLeader = 1,
        FromCluster,
    };

    TClusterInfo GetClusterInfo() const {
        const auto operationMode = OperationModeFromProto(Config_.GetOperationMode());
        auto clusterId = Config_.GetClusterId();

        EDc dc = EDc::UNKNOWN;
        if (Config_.GetDatacenter().empty()) {
            auto res = LoadSolomonEnv();
            Y_ENSURE(res.Success(), "cannot load solomon env: " << res.Error().Message());
            dc = res.Value().Dc;
        } else {
            dc = FromString<EDc>(Config_.GetDatacenter());
        }

        TClusterInfo info{std::move(clusterId), operationMode, dc};
        Y_VERIFY(FetcherCluster_);
        info.SetCluster(FetcherCluster_);
        return info;
    }

    NCloud::ITokenProviderPtr CreateIamTokenProvider(
            NCloud::TUpdatingProviderConf providerConf,
            NCloud::EStartPolicy startPolicy)
    {
        auto provider = AppData_.GetIamTokenProvider(providerConf.ServiceAccountId);
        if (provider) {
            return provider;
        }

        auto logger = CreateActorLogger(ActorRuntime_->ActorSystem());
        auto counters = MakeHolder<NCloud::TDefaultCounters>(*AppData_.Metrics);

        InitializeIamClient();
        auto& ex = AppData_.IamExecutors.emplace_back(CreateActorExecutor(ActorRuntime_->ActorSystem()));

        provider = CreateUpdatingProvider(
            std::move(providerConf),
            *ex,
            AppData_.IamClient,
            std::move(counters),
            std::move(logger));
        provider->Start(startPolicy);

        return provider;
    }

    NCloud::TUpdatingProviderConf ProviderConfFromYdbConfig(const NSolomon::NDb::TYdbConfig& conf) {
        const auto& authKey = conf.private_iam_key_auth();
        auto jwtParams = NYdb::ParseJwtParams(TFileInput(authKey.path()).ReadAll());

        NCloud::TUpdatingProviderConf providerConf;
        providerConf.PrivateKey = std::move(jwtParams.PrivKey);
        providerConf.PublicKey = std::move(jwtParams.PubKey);
        providerConf.ServiceAccountId = std::move(jwtParams.AccountId);
        providerConf.KeyId = std::move(jwtParams.KeyId);

        return providerConf;
    }

    std::unique_ptr<NYdb::TDriver> CreateYdbDriver(const NSolomon::NDb::TYdbConfig& ydbConfig) {
        NCloud::ITokenProviderPtr iamTokenProvider;
        if (ydbConfig.has_private_iam_key_auth()) {
            iamTokenProvider = CreateIamTokenProvider(
                    ProviderConfFromYdbConfig(ydbConfig),
                    NCloud::EStartPolicy::Sync);
        }

        // this driver can be created only after the Actor Runtime is set up and running,
        // because if iamTokenProvider is created, it will use an actor executor
        return std::make_unique<NYdb::TDriver>(
                CreateDriver(ydbConfig, *SecretProvider_, std::move(iamTokenProvider)));
    }

public:
    TFetcherApp(TFetcherConfig config, std::shared_ptr<NSecrets::ISecretProvider> secretProvider)
        : Config_{std::move(config)}
        , ActorRuntime_{TActorRuntime::Create(Config_.GetActorSystemConfig(), AppData_.Metrics, &AppData_)}
        , SecretProvider_{std::move(secretProvider)}
    {
        ActorRuntime_->Start();

        YdbDriver_ = CreateYdbDriver(Config_.GetYdbConfig());
        Dao_ = {*YdbDriver_, Config_.GetYdbConfig().GetSessionPoolConfig(), *AppData_.Metrics, Config_.GetConfigPathPrefix()};

        AppData_.SetDataHttpClient(CreateHttpClient("curlDwnld", Config_.GetDownloadConfig().http_client()));
        AppData_.SetResolverHttpClient(CreateHttpClient("curlRslvr", Config_.GetClusterResolveConfig().http_client()));

        AppData_.DnsClient = CreateDnsClient();
        if (Config_.HasYpConfig()) {
            const auto& ypConfig = Config_.GetYpConfig();
            AppData_.YpToken = SecretProvider_->GetSecret(ypConfig.token());
            Y_ENSURE(AppData_.YpToken.has_value(), "missing YP token, key: " << ypConfig.token());
        }

        AppData_.SetMetricVerbosity(Config_.GetMetricVerbosity());
        // XXX remove later
        if (Config_.GetDisableAgentPulling()) {
            Dao_.Agent = nullptr;
        }

        AppData_.Metrics->IntGauge({{"sensor", "version"}, {"revision", GetArcadiaLastChange()}})->Set(1);

        auto quitFunction = [this] (int) {
            QuitEvent_.Signal();
        };

        auto reopenLog = [this] (int) {
            ActorRuntime_->ReopenLog();
            TLogBackend::ReopenAllBackends();
        };

        SetAsyncSignalFunction(SIGTERM, quitFunction);
        SetAsyncSignalFunction(SIGINT, quitFunction);
        SetAsyncSignalFunction(SIGUSR2, reopenLog);
    }

    template <typename TComponentConfig>
    ui64 PoolFromConfig(
        bool (TFetcherConfig::*hasConf)() const,
        const TComponentConfig& (TFetcherConfig::*getConf)() const,
        ui64 defaultValue = 0)
    {
        if (!(Config_.*hasConf)()) {
            return defaultValue;
        }

        auto&& conf = (Config_.*getConf)();
        auto&& name = conf.GetExecutorPool();

        if (name.empty()) {
            return defaultValue;
        }

        return ActorRuntime_->FindExecutorByName(name);
    }

#define GET_POOL(component) \
    PoolFromConfig(&TFetcherConfig::Has##component##Config, &TFetcherConfig::Get##component##Config)

    ~TFetcherApp() {
        if (GrpcSrv_) {
            GrpcSrv_->Stop();
        }

        AppData_.Deinit();
        ActorRuntime_->Stop();
    }

    void InitServices() {
        if (Config_.HasHttpServerConfig()) {
            HttpServer_ = std::make_unique<NSolomon::NHttp::THttpServer>(*ActorRuntime_, AppData_.Metrics, Config_.GetHttpServerConfig());
        }

        if (Config_.HasRackTablesConfig()) {
            const auto& rtConfig = Config_.GetRackTablesConfig();
            TRackTablesActorConf RackTablesConfig {
                .RefreshInterval = FromProtoTime(rtConfig.GetRefreshInterval(), TDuration::Minutes(5)),
                .ConnectTimeout = FromProtoTime(rtConfig.GetConnectTimeout(), TDuration::Seconds(5)),
                .ReadTimeout = FromProtoTime(rtConfig.GetReadTimeout(), TDuration::Seconds(10)),
                .Retries = static_cast<ui8>(rtConfig.GetRetries()),
                .Url = rtConfig.GetUrl(),
                .FileCache = rtConfig.GetFileCache(),
                .HttpClient = AppData_.ResolverHttpClient(),
                .Registry = *AppData_.Metrics,
            };
            AppData_.RackTablesActorId = ActorRuntime_->Register(CreateRackTablesActor(RackTablesConfig));
        } else {
            AppData_.RackTablesActorId = ActorRuntime_->Register(CreateRacktablesActorStub(*AppData_.Metrics));
        }

        TDnsResolverActorConf dnsResolverConfig {
            .DnsClient = AppData_.DnsClient,
            .MetricRegistry = *AppData_.Metrics,
            .RackTablesActorId = AppData_.RackTablesActorId,
        };

        const auto continuousResolverId = ActorRuntime_->Register(
            CreateDnsResolverActor(std::move(dnsResolverConfig)),
            TMailboxType::Simple,
            GET_POOL(DnsResolver)
        );

        ActorRuntime_->RegisterLocalService(MakeDnsResolverId(), continuousResolverId);

        if (Config_.HasApiConfig()) {
            const auto conf = Config_.GetApiConfig();
            Y_VERIFY(conf.HasGrpcServerConfig(), "gRPC server config is required to run the API server");
            GrpcSrv_ = NSolomon::NGrpc::MakeGRpcServerFromConfig(
                ActorRuntime_->ActorSystem(),
                *AppData_.Metrics,
                conf.GetGrpcServerConfig());
        }

        AddActorRuntimeMetrics(*ActorRuntime_, *AppData_.Metrics);
        CreateConfigUpdater();
        CreateAgentCleaner();
        CreateLoadBalancer();
    }

    void CreateAgentCleaner() {
        if (Dao_.Agent == nullptr) {
            return;
        }

        ActorRuntime_->Register(
            CreateAgentCleanerActor(Dao_.Agent, TDuration::Hours(4), TDuration::Hours(24))
        );
    }

    void CreateConfigUpdater() {
        EUpdaterMode mode = EUpdaterMode::All;
        auto enableYasm = false;

        switch (Config_.GetScope()) {
            case TFetcherConfig::ENV:
                mode = EUpdaterMode::Env;
                break;
            case TFetcherConfig::NONE:
                mode = EUpdaterMode::None;
                break;
            case TFetcherConfig::ONLY_SOLOMON:
                // TODO(ivanzhukov@): actually support this mode
                break;
            case TFetcherConfig::ONLY_YASM:
                mode = EUpdaterMode::OnlyYasm;
                enableYasm = true;
                break;
            case TFetcherConfig::SOLOMON_AND_YASM:
                enableYasm = true;
                break;
            default:
                Y_FAIL("Unsuppoted scope: %s", TFetcherConfig::EScope_Name(Config_.GetScope()).c_str());
        }

        auto defaultUpdateInterval = TDuration::Seconds(60);
        auto updateInterval = Config_.HasUpdaterConfig()
                ? FromProtoTime(Config_.GetUpdaterConfig().GetInterval(), defaultUpdateInterval)
                : defaultUpdateInterval;

        TConfigUpdaterConfig shardUpdaterConfig {
            .ServiceDao = Dao_.Service,
            .ShardDao = Dao_.Shard,
            .ClusterDao = Dao_.Cluster,
            .AgentDao = Dao_.Agent,
            .ProviderDao = Dao_.Provider,
            .Interval = updateInterval,
            .MetricRegistry = *AppData_.Metrics,
            .Mode = mode,
            .EnableYasmPulling = enableYasm,
        };

        const auto shardUpdaterId = ActorRuntime_->Register(
            CreateConfigUpdaterActor(shardUpdaterConfig),
            TMailboxType::Simple,
            GET_POOL(Updater)
        );

        ActorRuntime_->RegisterLocalService(MakeConfigUpdaterId(), shardUpdaterId);
    }

    void CreateLoadBalancer() {
        if (!Config_.HasLoadBalancerConfig()) {
            Cerr << "Load balancer is not configured, skipping" << Endl;
            return;
        }

        auto&& loadBalancerConf = Config_.GetLoadBalancerConfig();

        if (TLoadBalancerConfig::STANDALONE == loadBalancerConf.GetBalancingMode()) {
            Cerr << "running in the STANDALONE mode -- skipping load balancing configuration" << Endl;
            return;
        }

        Y_ENSURE(loadBalancerConf.HasLockConfig(), "lock config must be specified");
        auto&& lockConf = loadBalancerConf.GetLockConfig();

        // create a separate driver with a dedicated thread for locks
        auto ydbDriverConf = Config_.GetYdbConfig();
        ydbDriverConf.SetClientThreadCount(1);
        auto ydbDriver = CreateYdbDriver(ydbDriverConf);

        auto lock = CreateLock(*ydbDriver, lockConf);
        auto config = CreateLoadBalancerConfig(
            *AppData_.Metrics,
            loadBalancerConf
        );

        auto* actor = CreateLoadBalancerActor(
            NCoordination::CreateStaticBalancer(),
            std::move(lock),
            std::move(config)
        );

        const auto loadBalancerId = ActorRuntime_->Register(actor);
        const auto nodeId = ActorRuntime_->NodeId();

        Cerr << "Initializing load balancer with node ID " << nodeId << Endl;
        ActorRuntime_->RegisterLocalService(
            NCoordination::MakeLoadBalancerId(),
            loadBalancerId
        );

        ActorRuntime_->RegisterLocalService(
            NCoordination::MakeLoadBalancerId(nodeId),
            loadBalancerId
        );
    }

    IYasmItypeWhiteListPtr GetYasmWhiteList() const {
        IYasmItypeWhiteListPtr white;
        IYasmItypeWhiteListPtr black;

        if (Config_.HasItypeWhiteList()) {
            TVector<TString> itypes;
            auto& whiteListConf = Config_.GetItypeWhiteList();
            for (const auto& itype: whiteListConf.GetItypesList()) {
                itypes.emplace_back(TString(itype.data(), itype.size()));
            }

            white = MakeYasmItypeWhiteList(itypes);
        }

        if (Config_.HasItypeBlackList()) {
            TVector<TString> itypes;
            auto& blackListConf = Config_.GetItypeBlackList();
            for (const auto& itype: blackListConf.GetItypesList()) {
                itypes.emplace_back(TString(itype.data(), itype.size()));
            }

            black = MakeYasmItypeBlackList(itypes);
        }

        return CombYasmWhiteLists(std::move(white), std::move(black));
    }

    TString GetYasmPrefix() const {
        TStringBuf prefix;
        if (Config_.HasYasmPrefix()) {
            auto& proto = Config_.GetYasmPrefix();
            prefix = TStringBuf(proto.GetPrefix().data(), proto.GetPrefix().size());
        } else {
            prefix = DEFAULT_YASM_PREFIX;
        }

        Y_ENSURE(
                prefix.StartsWith(TStringBuf("yasm_")) && prefix.back() == '_',
                "invalid yasm prefix:" << prefix << ", yasm_prefix must starts with \"yasm_\" and ends with \"_\"");

        Cerr << "yasm prefix:" << prefix << '\n';

        return TString{prefix};
    }

    TString GetDownloadIamAccountId() const {
        if (Config_.HasDownloadConfig()) {
            const auto& downloadConfig = Config_.GetDownloadConfig();
            return downloadConfig.iam_service_id();
        }
        return {};
    }

    TIntrusivePtr<NAuth::NTvm::ITvmServicesFactory> CreateTvmFactory(const NSolomon::NAuth::TTvmConfig& tvmConfig) {
        MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Read TVM config");
        auto clientSecret = SecretProvider_->GetSecret(tvmConfig.secret());
        Y_ENSURE(
                clientSecret.has_value(),
                "Cannot create TVM client " << tvmConfig.GetSelfId()
                                            << ", since client secret is missing, key: " << tvmConfig.secret());

        auto tvmConfigCopy = tvmConfig;
        tvmConfigCopy.set_secret(*clientSecret);

        TTicketProviderConfig config{
                .ActorSystem = ActorRuntime_->ActorSystem(),
                .ExecutorPool = 0,
                .Config = tvmConfigCopy,
                .Registry = *AppData_.Metrics,
        };

        return NAuth::NTvm::ITvmServicesFactory::Create(std::move(config));
    }

    void Run() {
        MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Fetcher starting");
        InitServices();
        const auto sinkId = ActorRuntime_->Register(CreateDataSink());
        ActorRuntime_->RegisterLocalService(MakeRouterServiceId(), sinkId);

        const auto dataSinkType = Config_.GetDataSinkConfig().GetType();

        if (dataSinkType != TDataSinkConfig::DEBUG && dataSinkType != TDataSinkConfig::DEVNULL) {
            TClusterInfo clusterInfo = GetClusterInfo();
            Cerr << "Cluster: " << clusterInfo << Endl;
            AppData_.SetClusterInfo(clusterInfo);
        }

        if (Config_.has_cloud_auth()) {
            InitializeIamTokenProviders(Config_.cloud_auth());
        }

        if (Config_.HasInstanceGroupConfig()) {
            CreateInstanceGroupClient(Config_.GetInstanceGroupConfig());
        }

        TFetchConfig fetchConf {
                .ConnectTimeout = DEFAULT_CONNECT_TIMEOUT,
                .DoSkipAuth = Config_.GetDoSkipAuth(),
                .SkipProviderDataWithWrongService = Config_.GetSkipProviderDataWithWrongService(),
        };
        ui64 maxInflight = 0;
        if (Config_.HasDownloadConfig()) {
            auto&& dlConf = Config_.GetDownloadConfig();
            maxInflight = dlConf.GetMaxInFlight();

            if (dlConf.HasConnectTimeout()) {
                fetchConf.ConnectTimeout = FromProtoTime(dlConf.GetConnectTimeout());
            }

            if (const auto& name = dlConf.GetDownloadPool(); !name.empty()) {
                fetchConf.DownloadPoolId = ActorRuntime_->FindExecutorByName(name);
            }

            if (const auto& name = dlConf.GetParsePool(); !name.empty()) {
                fetchConf.ParsePoolId = ActorRuntime_->FindExecutorByName(name);
            }
        }

        if (Config_.HasTvmConfig()) {
            auto tvmFactory = CreateTvmFactory(Config_.GetTvmConfig());
            AppData_.TvmClient = tvmFactory->CreateTvmClient();
            AppData_.TicketProvider = tvmFactory->CreateTicketProvider();
            auto userTvmAuthenticator = NAuth::CreateUserTvmAuthenticator(AppData_.TvmClient, ::NTvmAuth::EBlackboxEnv::Test);
            Authenticators_.emplace_back(std::move(userTvmAuthenticator));
            auto serviceTvmAuthenticator = NAuth::CreateServiceTvmAuthenticator(AppData_.TvmClient);
            Authenticators_.emplace_back(std::move(serviceTvmAuthenticator));
        }

        if (!Authenticators_.empty()) {
            MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Create authentication actor");
            auto muxAuthenticator = NSolomon::NAuth::CreateAuthenticatorMultiplexer(Authenticators_);
            auto authActor = NSolomon::NAuth::CreateAuthenticationActor(muxAuthenticator);
            AppData_.AuthenticationActorId = ActorRuntime_->Register(std::move(authActor));
        }

        NAuth::IInternalAuthorizerPtr internalAuthorizer;
        if (Config_.has_internalaccess()) {
            internalAuthorizer = NSolomon::NAuth::CreateInternalAuthorizer(Config_.internalaccess());
        }

        AppData_.SetFetchConfig(fetchConf);

        if (HttpServer_) {
            MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Start HTTP handlers");
            NSolomon::NHttp::TMetricsAuthorizationConfig authConfig{
                .AuthenticationActor = AppData_.AuthenticationActorId,
                .Authorizer = internalAuthorizer,
                .AuthenticationOff = Config_.has_cloud_auth() && Config_.cloud_auth().http_authentication_off(),
            };
            if (authConfig.AuthenticationOff) {
                MON_WARN_C(ActorRuntime_->ActorSystem(), Fetcher, "Metrics HTTP handlers authorization is off");
            }

            HttpServer_->Handle("/metrics", NSolomon::NHttp::CreateMetricsHandler(AppData_.Metrics, authConfig));
            HttpServer_->Handle(
                    "/metrics/shards",
                    NSolomon::NHttp::CreateAuxMetricsHandler(AppData_.ShardMetrics, std::move(authConfig)));
            HttpServer_->Handle("/version", NSolomon::NHttp::TVersionHandler{});

            MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Start selfmon service");
            NSelfMon::InitService(
                    AppData_.Metrics,
                    ActorRuntime_->ActorSystem(),
                    HttpServer_->Proxy(),
                    HttpServer_->ExecutorId(),
                    AppData_.AuthenticationActorId,
                    internalAuthorizer);
            AddFetcherSpecificHttpHandlers(ActorRuntime_->ActorSystem(), Config_.HasLoadBalancerConfig());
            NSelfMon::RegisterPage(
                    ActorRuntime_->ActorSystem(),
                    "/config", "Config",
                    NSelfMon::ProtoPage(std::make_shared<TFetcherConfig>(Config_)));
        }

        bool isStandalone = false;
        if (Config_.HasLoadBalancerConfig()) {
            isStandalone = Config_.GetLoadBalancerConfig().GetBalancingMode() == TLoadBalancerConfig::STANDALONE;
        }

        auto assnConsumer = CreateAssignmentsConsumer();
        auto urlLocator = isStandalone
                          ? StubUrlLocator()
                          : assnConsumer;

        std::map<ECloudEnv, TStringBuf> envoyAddresses;
        NCloud::NEnvoy::IEndpointClusterRpcPtr envoyRpc;

        if (Config_.HasCloudEnvoyConfig()) {
            const auto& cloudEnvoyConfig = Config_.GetCloudEnvoyConfig();
            auto clientConfig = cloudEnvoyConfig.GetClientConfig();
            clientConfig.ClearAddresses();

            for (const auto& [env, addr]: cloudEnvoyConfig.GetAddresses()) {
                envoyAddresses[FromString<ECloudEnv>(env)] = addr;
                clientConfig.AddAddresses(addr);
            }

            envoyRpc = NCloud::NEnvoy::CreateEndpointClusterGrpc(clientConfig, *AppData_.Metrics);
        }

        AppData_.SetResolverFactory(CreateHostResolverFactory(
            THostResolverFactoryConfig{
                .Registry = *AppData_.Metrics,
                .ClusterInfo = AppData_.ClusterInfo(),
                .DnsClient = AppData_.DnsClient,
                .HttpClient = AppData_.ResolverHttpClient(),
                .ConductorConfig = Config_.GetClusterResolveConfig().conductor_config(),
                .YpToken = AppData_.YpToken.value_or(TString{}),
                .InstanceGroupClient = AppData_.InstanceGroupClient,
                .UrlLocator = std::move(urlLocator),
                .EnvoyAddresses = std::move(envoyAddresses),
                .EnvoyRpc = std::move(envoyRpc),
            }
        ));

        auto hostListCache = CreateHostListCache(Config_);

        if (hostListCache) {
            ActorRuntime_->Register(CreateHostListCacheActor(hostListCache, {}));
        }

        const auto clusterManagerId = ActorRuntime_->Register(CreateClusterManager(
            *AppData_.Metrics,
            AppData_.ResolverFactory(),
            hostListCache
        ));

        if (!isStandalone) {
            IFetcherClusterPtr cluster = Config_.GetScope() == TFetcherConfig::ONLY_YASM
                    ? CreateYasmAgentCluster()
                    : nullptr;
            ActorRuntime_->Register(
                    CreateLoadBalancerActor(
                            NCoordination::MakeLoadBalancerId(),
                            clusterManagerId,
                            cluster,
                            assnConsumer)
            );
        }

        TActorId accessServiceId{};
        TActorId authGatekeeper{};

        if (Config_.has_multishard_pulling()) {
            const auto& mpAuth = Config_.multishard_pulling().auth();
            TDuration unusedRecordsTtl = FromProtoTime(mpAuth.unused_records_ttl(), TDuration::Minutes(5));
            TDuration cacheDuration = FromProtoTime(mpAuth.cache_duration(), TDuration::Minutes(1));
            TDuration gcInterval = FromProtoTime(mpAuth.gc_interval(), TDuration::Minutes(5));
            bool dontTakeAuthorizationIntoAccount = mpAuth.dont_take_authorization_into_account();

            if (mpAuth.has_tvm_config()) {
                auto tvmFactory = CreateTvmFactory(mpAuth.tvm_config());
                auto authenticator = NAuth::CreateServiceTvmAuthenticator(tvmFactory->CreateTvmClient());

                authGatekeeper = ActorRuntime_->Register(CreateTvmAuthGatekeeper(
                    std::move(authenticator),
                    MakeConfigUpdaterId(),
                    sinkId,
                    unusedRecordsTtl,
                    cacheDuration,
                    gcInterval,
                    AppData_.Metrics).release());
            } else if (mpAuth.has_access_service()) {
                InitializeCloudAccessService(mpAuth);

                auto scheduler = ActorRuntime_->Register(
                    CreateScheduler(TInstant::Now(), TDuration::MilliSeconds(100)).release());

                TAccessServiceConfig accessServiceConfig = AppData_.AccessServiceConfig().value();
                auto accessService = CreateAccessServiceActor(
                    AppData_.AccessServiceClient,
                    scheduler,
                    std::move(accessServiceConfig),
                    AppData_.Metrics);
                accessServiceId = ActorRuntime_->Register(accessService.release());

                authGatekeeper = ActorRuntime_->Register(CreateIamAuthGatekeeper(
                    accessServiceId,
                    MakeConfigUpdaterId(),
                    sinkId,
                    unusedRecordsTtl,
                    cacheDuration,
                    gcInterval,
                    dontTakeAuthorizationIntoAccount,
                    AppData_.Metrics).release());
            }
        }

        const auto shardManagerId = ActorRuntime_->Register(
            CreateShardManagerActor(TShardManagerConfig{
                .DataSinkId = sinkId,
                .AuthGatekeeperId = authGatekeeper,
                .ClusterManagerId = clusterManagerId,
                .MaxInflight = maxInflight,
                .YasmWhiteList = GetYasmWhiteList(),
                .YasmPrefix = GetYasmPrefix(),
                .IamServiceId = GetDownloadIamAccountId()
            })
        );

        if (GrpcSrv_) {
            const auto poolName = Config_.GetApiConfig().GetExecutorPool();
            const auto executorPool = poolName.empty()
                ? 0
                : ActorRuntime_->FindExecutorByName(poolName);

            const auto proxyId = ActorRuntime_->Register(CreateFetcherApiProxy(
                *AppData_.Metrics,
                shardManagerId,
                executorPool));

            GrpcSrv_->AddService(CreateFetcherService(proxyId, ActorRuntime_->ActorSystem(), *AppData_.Metrics));
            GrpcSrv_->Start();
        }

        QuitEvent_.WaitI();
    }

    void InitializeIamTokenProviders(const ::NSolomon::NFetcher::CloudAuth& cloudAuth) {
        MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Initialize IAM token providers");

        for (const auto& credentials: cloudAuth.iam_credentials()) {
            credentials.GetAccountId();
            NCloud::TUpdatingProviderConf providerConf;
            providerConf.ServiceAccountId = credentials.GetAccountId();
            providerConf.KeyId =  credentials.GetKeyId();
            providerConf.PublicKeyFile = credentials.GetPublicKeyFile();
            providerConf.PrivateKeyFile = credentials.GetPrivateKeyFile();

            auto tokenProvider = CreateIamTokenProvider(providerConf, NCloud::EStartPolicy::Async);
            AppData_.AddIamTokenProvider(credentials.GetAccountId(), std::move(tokenProvider));
        }
    }

    void InitializeCloudAccessService(::NSolomon::NFetcher::MultishardPullingConfig::Auth mpAuth) {
        MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Read cloud auth config");
        auto& asConfig = *mpAuth.mutable_access_service();

        if (asConfig.GetRetriesCnt() == 0) {
            asConfig.SetRetriesCnt(3);
        }

        if (asConfig.GetMaxInflight() == 0) {
            asConfig.SetMaxInflight(190); // IAM says that AccessService can handle only 192 concurrent reqs
            // https://t.me/c/1116364552/81668
        }

        if (!asConfig.HasMinRetryDelay() ||
            asConfig.GetMinRetryDelay().GetValue() == 0 ||
            asConfig.GetMinRetryDelay().GetUnit() == yandex::solomon::config::NANOSECONDS)
        {
            auto& minRetryDelay = *asConfig.MutableMinRetryDelay();
            minRetryDelay.SetValue(2);
            minRetryDelay.SetUnit(yandex::solomon::config::SECONDS);
        }

        if (!asConfig.HasMaxRetryDelay() ||
            asConfig.GetMaxRetryDelay().GetValue() == 0 ||
            asConfig.GetMaxRetryDelay().GetUnit() == yandex::solomon::config::NANOSECONDS)
        {
            auto& maxRetryDelay = *asConfig.MutableMaxRetryDelay();
            maxRetryDelay.SetValue(12);
            maxRetryDelay.SetUnit(yandex::solomon::config::SECONDS);
        }

        if (!asConfig.HasTimeout() ||
            asConfig.GetTimeout().GetValue() == 0 ||
            asConfig.GetTimeout().GetUnit() == yandex::solomon::config::NANOSECONDS)
        {
            auto& timeout = *asConfig.MutableTimeout();
            timeout.SetValue(12);
            timeout.SetUnit(yandex::solomon::config::SECONDS);
        }

        AppData_.SetAccessServiceConfig(mpAuth.access_service());
        AppData_.AccessServiceClient = CreateAccessServiceGrpcClient(
                mpAuth.access_service().GetClientConfig(),
                *AppData_.Metrics,
                Config_.GetClientId());

        auto iamAuthenticator = NSolomon::NAuth::CreateIamAuthenticator(AppData_.AccessServiceClient);
        Authenticators_.emplace_back(std::move(iamAuthenticator));
    }

    IHttpClientPtr CreateHttpClient(TString name, const HttpClientConfig& config) {
        TCurlClientOptions opts;
        opts.Name = std::move(name);
        opts.WorkerThreads = config.worker_threads();
        opts.HandlerThreads = config.handler_threads();

        opts.DnsCacheLifetime = config.has_dns_cache_lifetime()
                ? FromProtoTime(config.dns_cache_lifetime(), TDuration::Zero())
                : TDuration::Max();

        if (!config.ca_cert_dir().empty()) {
            opts.CaCertDir = config.ca_cert_dir();
        }

        opts.MaxInflight = config.max_in_flight();
        opts.QueueSizeLimit = config.queue_size_limit();

        if (config.has_bind()) {
            opts.BindOptions = TCurlBindOptions{
                .Host = config.bind().host(),
                .Port = static_cast<ui16>(config.bind().port()),
                .PortRange = static_cast<ui16>(config.bind().port_range()),
            };
        }

        return CreateCurlClient(std::move(opts), *AppData_.Metrics);
    }

    IActor* CreateDataSink() {
        Y_VERIFY(Config_.HasDataSinkConfig());
        const auto type = Config_.GetDataSinkConfig().GetType();
        if (type == TDataSinkConfig::DEBUG) {
            return CreateDebugDataSink();
        } else if (type == TDataSinkConfig::COREMON) {
            Y_VERIFY(Config_.GetDataSinkConfig().HasCoremonClientConfig(), "COREMON sink type requires client configuration");
            return CreateCoremonDataSink();
        } else if (type == TDataSinkConfig::DEVNULL) {
            return CreateNullDataSink();
        } else if (type == TDataSinkConfig::INGESTOR) {
            Y_VERIFY(Config_.GetDataSinkConfig().HasIngestorClientConfig(), "INGESTOR sink type requires ingestor client configuration");
            return CreateIngestorDataSink(EIngestorAssignmentsMode::FromLeader);
        } else if (type == TDataSinkConfig::INGESTOR_NEW_API) {
            Y_VERIFY(Config_.GetDataSinkConfig().HasIngestorClientConfig(), "INGESTOR_NEW_API sink type requires ingestor client configuration");
            return CreateIngestorDataSink(EIngestorAssignmentsMode::FromCluster);
        } else {
            Y_FAIL("Unsupported sink type: %" PRIu32, ui32(type));
        }
    }

    NRpc::TGrpcClientConfig PrepareGrpcConfig(
            const NRpc::TGrpcClientConfig &rawConf,
            bool doSkipClusterAffiliationChecks)
    {
        NRpc::TGrpcClientConfig conf{rawConf};

        TStringBuilder sb;
        for (auto addr: conf.GetAddresses()) {
            sb << addr << '\n';
        }

        FetcherCluster_ = TFetcherCluster::FromString(TString{sb});

        if (!doSkipClusterAffiliationChecks) {
            Y_ENSURE(
                    !FetcherCluster_->Local().IsUnknown(),
                    "Current host does not belong to the cluster " << *FetcherCluster_);
        }

        Cerr << "Cluster resolved into: " << *FetcherCluster_ << Endl;

        conf.MutableAddresses()->Clear();

        for (auto&& addr: FetcherCluster_->Endpoints()) {
            Cerr << addr << Endl;
            *conf.MutableAddresses()->Add() = addr;
        }

        return conf;
    }

    IActor* CreateRpcDataSink(IProcessingClusterClientPtr client) {
        auto localMemLimit = Config_.GetDataSinkConfig().HasMemoryLimit()
            ? FromProtoDataSize(Config_.GetDataSinkConfig().GetMemoryLimit())
            : 0;

        const auto pool = Config_.GetDataSinkConfig().GetExecutorPool();

        THashMap<TClusterNode, TActorId> sinks = CreateRpcSinks(
            *ActorRuntime_,
            AppData_.Metrics,
            AppData_.ShardMetrics,
            FetcherCluster_->Nodes(),
            client,
            localMemLimit,
            ActorRuntime_->FindExecutorByName(pool)
        );

        TRouterActorConf routerConf;
        routerConf.Peers = std::move(sinks);
        routerConf.Cluster = FetcherCluster_.Get();
        routerConf.UnknownShardHandlerId =
            ActorRuntime_->Register(CreateShardCreatorActor(std::move(client), *FetcherCluster_));

        return CreateRouterActor(std::move(routerConf), *AppData_.Metrics);
    }

    IActor* CreateCoremonDataSink() {
        auto coremonClient = NSolomon::NCoremon::CreateGrpcClusterClient(
            PrepareGrpcConfig(
                    Config_.GetDataSinkConfig().GetCoremonClientConfig(),
                    Config_.GetDoSkipClusterAffiliationChecks()),
            *AppData_.Metrics,
            Config_.GetClientId());

        auto processingClient = WrapClusterClient(coremonClient);

        auto shardMapId = ActorRuntime_->Register(CreateShardMapBuilderRequestingLeader(processingClient, FetcherCluster_));
        ActorRuntime_->RegisterLocalService(MakeShardMapBuilderId(), shardMapId);

        return CreateRpcDataSink(std::move(processingClient));
    }

    IActor* CreateIngestorDataSink(EIngestorAssignmentsMode assignmentsMode) {
        auto ingestorClient = NSolomon::NIngestor::CreateIngestorGrpcClusterClient(
            PrepareGrpcConfig(
                    Config_.GetDataSinkConfig().GetIngestorClientConfig(),
                    Config_.GetDoSkipClusterAffiliationChecks()),
            *AppData_.Metrics,
            Config_.GetClientId());

        auto processingClient = WrapClusterClient(ingestorClient);

        switch (assignmentsMode) {
            case EIngestorAssignmentsMode::FromCluster: {
                auto shardMapId = ActorRuntime_->Register(CreateShardMapBuilderRequestingCluster(ingestorClient, FetcherCluster_));
                ActorRuntime_->RegisterLocalService(MakeShardMapBuilderId(), shardMapId);
                break;
            }
            case EIngestorAssignmentsMode::FromLeader: {
                auto shardMapId = ActorRuntime_->Register(CreateShardMapBuilderRequestingLeader(processingClient, FetcherCluster_));
                ActorRuntime_->RegisterLocalService(MakeShardMapBuilderId(), shardMapId);
                break;
            }
            default:
                ythrow yexception() << "unknown assignments mode";
        }

        return CreateRpcDataSink(std::move(processingClient));
    }

    void InitializeIamClient() {
        if (!AppData_.IamClient) {
            const auto& iamClientConf = Config_.GetIamClientConfig();
            AppData_.IamClient = NCloud::CreateGrpcIamClient(iamClientConf.GetClientConfig(), *AppData_.Metrics);
        }
    }

    void CreateInstanceGroupClient(const TInstanceGroupClientConfig& conf) {
        MON_INFO_C(ActorRuntime_->ActorSystem(), Fetcher, "Create instance group client");

        const auto& iamAccountId = conf.GetServiceAccountId();
        auto iamTokenProvider = AppData_.GetIamTokenProvider(iamAccountId);
        Y_VERIFY(
                iamTokenProvider,
                "IAM token provider with account id %s is not created. IAM credentials should be supplied in cloud_auth config.",
                iamAccountId.c_str());

        AppData_.InstanceGroupClient = NCloud::CreateInstanceGroupClient(
            conf.GetClientConfig(),
            std::move(iamTokenProvider),
            *AppData_.Metrics);
    }

private:
    TFetcherConfig Config_;
    TAppData AppData_;

    // must be destroyed last
    TActorRuntime::TPtr ActorRuntime_;
    std::shared_ptr<NSecrets::ISecretProvider> SecretProvider_;
    TIntrusivePtr<TFetcherCluster> FetcherCluster_;

    std::unique_ptr<NYdb::TDriver> YdbDriver_;
    TDao Dao_;
    std::unique_ptr<NSolomon::NHttp::THttpServer> HttpServer_;
    THolder<::NGrpc::TGRpcServer> GrpcSrv_;
    TAutoEvent QuitEvent_;
    TVector<NAuth::IAuthenticatorPtr> Authenticators_;
};

int main(int argc, const char** argv) {
    NLastGetopt::TOpts opts;
    opts.AddLongOption("config", "path to configuration file")
        .Required()
        .RequiredArgument("<path>")
        .Completer(NLastGetopt::NComp::File("*.conf"));
    opts.AddLongOption("secrets", "path to secrets file")
        .Required()
        .RequiredArgument("<path>")
        .Completer(NLastGetopt::NComp::File("*.secrets"));
    opts.AddVersionOption();
    opts.AddHelpOption();
    opts.AddCompletionOption("fetcher");
    opts.SetFreeArgsNum(0);

    try {
        NLastGetopt::TOptsParseResult r(&opts, argc, argv);
        Cerr << "Starting Fetcher (pid: " << GetPID() << ", SVN revision: " << GetArcadiaLastChange() << ')' << Endl;

        TFetcherConfig config = NSolomon::NFetcher::ParseTextProto(r.Get("config"));
        NSolomon::MergeIncludes(config);

        NSolomon::SetupMinidump(config.minidump_config());

        TFsPath secrets(r.Get("secrets"));
        auto secretProvider = secrets.Exists()
            ? NSecrets::FileSecretProvider(secrets.GetPath())
            : NSecrets::EmptySecretProvider();

        NSolomon::NGrpc::SetThreadsLimit(2);

        TFetcherApp app{config, secretProvider};
        app.Run();
        return 0;
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("fetcher");
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
    }

    return 1;
}
