#include "fetcher_client.h"

#include <solomon/services/fetcher/api/fetcher_service.grpc.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/system/hostname.h>

using yandex::monitoring::fetcher::FetcherService;

namespace NSolomon::NFetcher {
namespace {
    using namespace grpc;
    using namespace NThreading;
    using namespace NMonitoring;
    using yandex::solomon::config::rpc::TGrpcClientConfig;

    using NGrpc::TGRpcClientConfig;
    using NGrpc::TGrpcStatus;

    using NSolomon::TResolveClusterRequest;
    using NSolomon::TResolveClusterResponse;

    TClusterResolveResponse ConvertClusterFromProto(NSolomon::TResolveClusterResponse&& resp) {
        TClusterResolveResponse native;

        for (auto& result: resp.GetResults()) {
            auto& groupInfo = native.Groups.emplace_back();
            groupInfo.Name = result.GetGroupName();

            if (result.HasHosts()) {
                TVector<THostAndLabels> nativeHosts;
                auto&& hosts = result.GetHosts();
                for (auto&& hostInfo: hosts.GetHosts()) {
                    auto& nativeHost = nativeHosts.emplace_back(hostInfo.GetFqdn());

                    if (auto port = hostInfo.GetPort(); port != 0) {
                        nativeHost.Port = port;
                    }

                    if (auto labels = hostInfo.GetLabels(); !labels.Empty()) {
                        const auto ok = TryLoadLabelsFromString(labels, nativeHost.Labels);
                        Y_VERIFY_DEBUG(ok);
                        // TODO: log
                    }
                }

                groupInfo.Result = groupInfo.Result.FromValue(std::move(nativeHosts));
            } else if (result.HasErrorMessage()) {
                groupInfo.Result = groupInfo.Result.FromError(result.GetErrorMessage());
            } else {
                groupInfo.Result = groupInfo.Result.FromError("Server didn't set neither value nor error");
            }
        }

        return native;
    }

    TGroupConf::EGroupType ToProtoType(TClusterResolveRequest::EGroupType type) {
        switch (type) {
            case TClusterResolveRequest::EGroupType::ConductorTag:
                return TGroupConf::CONDUCTOR_TAG;
            case TClusterResolveRequest::EGroupType::ConductorGroup:
                return TGroupConf::CONDUCTOR_GROUP;
            case TClusterResolveRequest::EGroupType::Nanny:
                return TGroupConf::NANNY;
            case TClusterResolveRequest::EGroupType::Yp:
                return TGroupConf::YP;
            case TClusterResolveRequest::EGroupType::Qloud:
                return TGroupConf::QLOUD;
            case TClusterResolveRequest::EGroupType::Network:
                return TGroupConf::NETWORK;
            case TClusterResolveRequest::EGroupType::InstanceGroup:
                return TGroupConf::INSTANCE_GROUP;
            case TClusterResolveRequest::EGroupType::Host:
                return TGroupConf::HOSTS;
            case TClusterResolveRequest::EGroupType::HostUrl:
                return TGroupConf::HOST_URL;
        };
    }

    class TGrpcFetcherClient final: public IFetcherClient {
    public:
        explicit TGrpcFetcherClient(std::unique_ptr<TGrpcServiceConnection<FetcherService>> connection)
            : Connection_{std::move(connection)}
        {
        }

        TAsyncResolveClusterResponse ResolveCluster(TClusterResolveRequest req) override {
            TResolveClusterRequest grpcReq;

            for (auto& group: req.Groups) {
                auto* g = grpcReq.AddGroups();
                g->SetType(ToProtoType(group.Type));
                g->SetJsonConfig(std::move(group.JsonConfig));
            }

            auto promise = NewPromise<TErrorOr<TClusterResolveResponse, TApiCallError>>();

            auto cb = [promise] (TGrpcStatus&& status, TResolveClusterResponse&& result) mutable {
                if (!status.Ok()) {
                    promise.SetValue(FromGrpcError<TClusterResolveResponse>(std::move(status)));
                } else {
                    promise.SetValue(ConvertClusterFromProto(std::move(result)));
                }
            };

            Connection_->Request<TResolveClusterRequest, TResolveClusterResponse>(
                grpcReq,
                std::move(cb),
                &FetcherService::Stub::AsyncResolveCluster);

            return promise.GetFuture();
        }

    private:
        std::unique_ptr<TGrpcServiceConnection<FetcherService>> Connection_;
    };
} // namespace

    IFetcherClientPtr CreateGrpcClient(const TGrpcClientConfig& conf, NMonitoring::IMetricRegistry& registry, TString clientId) {
        auto threadPool = CreateGrpcThreadPool(conf);
        auto sc = CreateGrpcServiceConnection<FetcherService>(
            conf,
            false,
            registry,
            std::move(threadPool),
            std::move(clientId));
        return MakeHolder<TGrpcFetcherClient>(std::move(sc));
    }

} // namespace NSolomon::NFetcher
