#include "api.h"

#include <solomon/services/fetcher/api/fetcher_api.pb.h>
#include <solomon/services/fetcher/api/fetcher_service.grpc.pb.h>
#include <solomon/services/fetcher/lib/app_data.h>
#include <solomon/services/fetcher/lib/config_updater/config_updater.h>
#include <solomon/services/fetcher/lib/racktables/dc_matcher.h>
#include <solomon/services/fetcher/lib/shard_manager/shard_resolver.h>
#include <solomon/services/fetcher/lib/shard_manager/shard_stat.h>

#include <solomon/libs/cpp/conf_db/model/cluster_config.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/grpc/stats/req_ctx_wrapper.h>
#include <solomon/libs/cpp/grpc/stats/stats_filter.h>
#include <solomon/libs/cpp/grpc/stats/stats_page.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/grpc/server/grpc_request.h>
#include <library/cpp/grpc/server/grpc_server.h>
#include <library/cpp/monlib/metrics/histogram_collector.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/ptr.h>
#include <util/string/cast.h>

#include <google/protobuf/arena.h>

using TGrpcResolveResult = NSolomon::TResolveResult;
using namespace NMonitoring;
using namespace NActors;
using namespace NGrpc;
using yandex::solomon::common::UrlStatusType;


namespace NSolomon::NFetcher {
namespace {
    IFetcherClusterPtr CreateClusterConf(const NSolomon::TResolveClusterRequest& req) {
        NDb::NModel::TClusterConfig cluster;

        for (auto&& group: req.GetGroups()) {
            switch (group.GetType()) {
                case TGroupConf::CONDUCTOR_TAG:
                    cluster.ConductorTags = group.GetJsonConfig();
                    break;
                case TGroupConf::CONDUCTOR_GROUP:
                    cluster.ConductorGroups = group.GetJsonConfig();
                    break;
                case TGroupConf::NANNY:
                    cluster.NannyGroups = group.GetJsonConfig();
                    break;
                case TGroupConf::YP:
                    cluster.YpClusters = group.GetJsonConfig();
                    break;
                case TGroupConf::QLOUD:
                    cluster.QloudGroups = group.GetJsonConfig();
                    break;
                case TGroupConf::NETWORK:
                    cluster.Networks = group.GetJsonConfig();
                    break;
                case TGroupConf::INSTANCE_GROUP:
                    cluster.InstanceGroups = group.GetJsonConfig();
                    break;
                case TGroupConf::HOSTS:
                    cluster.Hosts = group.GetJsonConfig();
                    break;
                case TGroupConf::HOST_URL:
                    cluster.HostUrls = group.GetJsonConfig();
                    break;
                case TGroupConf::CLOUD_DNS:
                    cluster.CloudDns = group.GetJsonConfig();
                    break;
                case TGroupConf::UNKNOWN:
                case TGroupConf_EGroupType_TGroupConf_EGroupType_INT_MAX_SENTINEL_DO_NOT_USE_:
                case TGroupConf_EGroupType_TGroupConf_EGroupType_INT_MIN_SENTINEL_DO_NOT_USE_:
                    // ignore
                    // XXX write to log? return error?
                    break;
            }
        }

        return CreateCluster(cluster);
    }

    struct TFetcherGrpcEvents: private TPrivateEvents {
        enum {
            Start = SpaceBegin,
            EvTargetsStatusRequest,
            EvShardsStatsRequest,
            EvShardsHealthRequest,
            EvReloadShardRequest,
            EvTimeout,
            EvResolveCluster,
            End,
        };
        static_assert(End < SpaceEnd, "too many event types");

        struct TEvTimeout: TEventLocal<TEvTimeout, TFetcherGrpcEvents::EvTimeout> { };

        struct TEvTargetsStatusRequest: TEventLocal<TEvTargetsStatusRequest, EvTargetsStatusRequest> {
            explicit TEvTargetsStatusRequest(IRequestContextBase* ctx)
                : Ctx{ctx}
                , Request{static_cast<const TargetsStatusRequest*>(Ctx->GetRequest())}
            { }

            IRequestContextBase* Ctx{nullptr};
            const TargetsStatusRequest* Request{nullptr};
        };

        struct TEvShardsStatsRequest: TEventLocal<TEvShardsStatsRequest, EvShardsStatsRequest> {
            explicit TEvShardsStatsRequest(IRequestContextBase* ctx)
                : Ctx{ctx}
            { }

            IRequestContextBase* Ctx{nullptr};
        };

        struct TEvShardsHealthRequest: TEventLocal<TEvShardsHealthRequest, EvShardsHealthRequest> {
            explicit TEvShardsHealthRequest(IRequestContextBase* ctx)
                : Ctx{ctx}
                , Request{static_cast<const ShardsHealthRequest*>(Ctx->GetRequest())}
            { }

            IRequestContextBase* Ctx{nullptr};
            const ShardsHealthRequest* Request{nullptr};
        };

        struct TEvReloadShardRequest: TEventLocal<TEvReloadShardRequest, EvReloadShardRequest> {
            explicit TEvReloadShardRequest(IRequestContextBase* ctx)
                : Ctx{ctx}
                , Request{static_cast<const TReloadShardRequest*>(Ctx->GetRequest())}
            { }

            IRequestContextBase* Ctx{nullptr};
            const TReloadShardRequest* Request{nullptr};
        };

        struct TEvResolveCluster: TEventLocal<TEvResolveCluster, EvResolveCluster> {
            TEvResolveCluster() = default;
            explicit TEvResolveCluster(IRequestContextBase* ctx)
                : Ctx{ctx}
                , Request{static_cast<const TResolveClusterRequest*>(Ctx->GetRequest())}
            { }

            IRequestContextBase* Ctx{nullptr};
            const TResolveClusterRequest* Request{nullptr};
        };

        static_assert(End < EventSpaceEnd(TEvents::ES_PRIVATE));
    };

    struct TRequestCounters {
        TRequestCounters(TMetricRegistry& registry)
            : RequestActors{registry.IntGauge({{"sensor", "requestActors"}, {"component", "grpc"}})}
            , ProcessingTime{registry.HistogramRate(
                {{"sensor", "processingTimeUs"}, {"component", "grpcProxy"}},
                ExponentialHistogram(13, 2, 16))}
            , TimedOut{registry.Counter({{"sensor", "requestsTimedOut"}, {"component", "grpc"}})}
            , NotFound{registry.Counter({{"sensor", "shardNotFound"}, {"component", "grpc"}})}
        {
        }

        TIntGauge* RequestActors{nullptr};
        THistogram* ProcessingTime{nullptr};
        TCounter* TimedOut{nullptr};
        TCounter* NotFound{nullptr};
    };

    template <typename TDerived, typename TRequest, typename TResponse>
    class TApiRequest: public TActorBootstrapped<TApiRequest<TDerived, TRequest, TResponse>> {
    public:
        TApiRequest(TMetricRegistry& registry, TActorId shardManagerId, THolder<TRequest> req)
            : ShardManagerId_{shardManagerId}
            , Counters_{registry}
            , Req_{std::move(req)}
            , Ctx_{Req_->Ctx}
        {
            Counters_.RequestActors->Inc();
        }

        ~TApiRequest() {
            Counters_.RequestActors->Dec();
        }

        void Bootstrap(const TActorContext&) {
            Self()->Become(&TDerived::StateWork);
            Request();
        }

        virtual void Request() {
            Timer_.Reset();
            Self()->Send(ShardManagerId_, Self()->FromInternalEvent(*Req_));
            Self()->Schedule(TDuration::Seconds(5), new TFetcherGrpcEvents::TEvTimeout);
        }

        STFUNC(StateWork) {
            switch (ev->GetTypeRewrite()) {
                HFunc(TFetcherGrpcEvents::TEvTimeout, OnTimeout);
                HFunc(TResponse, OnResponse);
                HFunc(TEvNotFound, OnNotFound);
                cFunc(TEvents::TSystem::PoisonPill, TDerived::PassAway);
            }
        };

    private:
        TDerived* Self() {
            return static_cast<TDerived*>(this);
        }

        void OnTimeout(const TFetcherGrpcEvents::TEvTimeout::TPtr&, const TActorContext&) {
            Counters_.TimedOut->Inc();
            Ctx_->ReplyError(grpc::StatusCode::UNAVAILABLE, "Unable to execute request in time");
            TDerived::PassAway();
        }

        void OnNotFound(const TEvNotFound::TPtr&, const TActorContext&) {
            Counters_.NotFound->Inc();
            Ctx_->ReplyError(grpc::StatusCode::NOT_FOUND, "Not found");
            TDerived::PassAway();
        }

        void OnResponse(const typename TResponse::TPtr& ev, const TActorContext&) {
            Counters_.ProcessingTime->Record(TDuration::FromValue(Timer_.Passed()).MicroSeconds());
            NProtoBuf::Arena arena;
            auto* resp = Self()->CreateReply(*ev->Get(), &arena);

            Ctx_->Reply(resp);
            TDerived::PassAway();
        }

    protected:
        THPTimer Timer_;
        TActorId ShardManagerId_;
        TRequestCounters Counters_;
        THolder<TRequest> Req_;
        IRequestContextBase* Ctx_{nullptr};
    };

    class TClusterResolveRequest: public TApiRequest<TClusterResolveRequest, TFetcherGrpcEvents::TEvResolveCluster, TEvClusterResolved> {
        using TBase = TApiRequest<
            TClusterResolveRequest,
            TFetcherGrpcEvents::TEvResolveCluster,
            TEvClusterResolved>;

    public:
        TClusterResolveRequest(TMetricRegistry& registry, THolder<TFetcherGrpcEvents::TEvResolveCluster> req)
            : TBase{registry, TActorId{}, std::move(req)}
        {
        }

        void Request() override {
            Timer_.Reset();

            auto cluster = CreateClusterConf(*Req_->Request);
            auto values = cluster->CreateResolvers(*GetAppData().ResolverFactory());

            TVector<IHostGroupResolverPtr> resolvers;
            for (auto& v: values) {
                if (!v.Success()) {
                    auto& err = ParseErrors_.emplace_back();
                    err.SetErrorMessage(v.Error().MessageString());
                    // we don't have the name for this group yet
                    continue;
                }

                resolvers.push_back(v.Extract());
            }

            Register(ResolveCluster(resolvers, SelfId()));
            Schedule(TDuration::Seconds(25), new TFetcherGrpcEvents::TEvTimeout);
        }

        TFetcherGrpcEvents::TEvResolveCluster* FromInternalEvent(const TFetcherGrpcEvents::TEvResolveCluster&) const {
            Y_VERIFY_DEBUG(false, "this method shouldn't be called at all");
            return new TFetcherGrpcEvents::TEvResolveCluster;
        }

        TResolveClusterResponse* CreateReply(const TEvClusterResolved& ev, NProtoBuf::Arena* arena) const {
            auto* resp = NProtoBuf::Arena::CreateMessage<TResolveClusterResponse>(arena);

            for (auto&& [resolver, result]: *ev.Result) {
                auto* item = resp->AddResults();
                item->SetGroupName(resolver->Name());

                if (!result.Success()) {
                    item->SetErrorMessage(ToString(result.Error().Message()));
                    continue;
                }

                auto* info = NProtoBuf::Arena::CreateMessage<TGrpcResolveResult::TGroupInfo>(arena);
                for (auto& host: result.Value()) {
                    auto* hostInfo = info->AddHosts();
                    hostInfo->SetFqdn(std::move(host.Host));
                    hostInfo->SetPort(std::move(host.Port));
                    hostInfo->SetLabels(ToString(host.Labels));
                }

                item->set_allocated_hosts(info);

            }

            for (auto& err: ParseErrors_) {
                resp->AddResults()->CopyFrom(err);
            }

            return resp;
        }

    private:
        TVector<TGrpcResolveResult> ParseErrors_;
    };

    class TShardHealthRequest: public TApiRequest<TShardHealthRequest, TFetcherGrpcEvents::TEvShardsHealthRequest, TEvShardsHealthResponse> {
        using TBase = TApiRequest<
            TShardHealthRequest,
            TFetcherGrpcEvents::TEvShardsHealthRequest,
            TEvShardsHealthResponse>;

    public:
        TShardHealthRequest(TMetricRegistry& registry, TActorId shardManagerId, THolder<TFetcherGrpcEvents::TEvShardsHealthRequest> req)
            : TBase{registry, shardManagerId, std::move(req)}
        {
        }

        TEvShardsHealthRequest* FromInternalEvent(const TFetcherGrpcEvents::TEvShardsHealthRequest&) const {
            return new TEvShardsHealthRequest;
        }

        ShardsHealthResponse* CreateReply(const TEvShardsHealthResponse& ev, NProtoBuf::Arena* arena) const {
            auto* resp = NProtoBuf::Arena::CreateMessage<ShardsHealthResponse>(arena);

            for (auto&& shard: ev.Shards) {
                auto* item = resp->AddShards();
                item->SetNumId(shard.NumId);
                item->SetUrlsOk(shard.UrlsOk);
                item->SetUrlsFail(shard.UrlsFail);
            }

            return resp;
        }
    };

    class TShardReloadRequest: public TApiRequest<TShardReloadRequest, TFetcherGrpcEvents::TEvReloadShardRequest, TEvReloadShardResponse> {
        using TBase = TApiRequest<
            TShardReloadRequest,
            TFetcherGrpcEvents::TEvReloadShardRequest,
            TEvReloadShardResponse>;

    public:
        TShardReloadRequest(TMetricRegistry& registry, TActorId shardManagerId, THolder<TFetcherGrpcEvents::TEvReloadShardRequest> req)
            : TBase{registry, shardManagerId, std::move(req)}
        {
        }

        TEvReloadShardRequest* FromInternalEvent(const TFetcherGrpcEvents::TEvReloadShardRequest& ev) const {
            return new TEvReloadShardRequest{
                ev.Request->GetProjectId(),
               {ev.Request->GetShardId(), ev.Request->GetNumId()},
            };
        }

        TReloadShardResponse* CreateReply(const TEvReloadShardResponse&, NProtoBuf::Arena* arena) const {
            return NProtoBuf::Arena::CreateMessage<TReloadShardResponse>(arena);
        }
    };

    class TShardStatsRequest: public TApiRequest<TShardStatsRequest, TFetcherGrpcEvents::TEvShardsStatsRequest, TEvShardStatsResponse> {
        using TBase = TApiRequest<
            TShardStatsRequest,
            TFetcherGrpcEvents::TEvShardsStatsRequest,
            TEvShardStatsResponse>;

    public:
        TShardStatsRequest(TMetricRegistry& registry, TActorId shardManagerId, THolder<TFetcherGrpcEvents::TEvShardsStatsRequest> req)
            : TBase{registry, shardManagerId, std::move(req)}
        {
        }

        TEvShardStatsRequest* FromInternalEvent(const TFetcherGrpcEvents::TEvShardsStatsRequest&) const {
            return new TEvShardStatsRequest;
        }

        ShardsStatsResponse* CreateReply(const TEvShardStatsResponse& ev, NProtoBuf::Arena* arena) const {
            ShardsStatsResponse* resp = NProtoBuf::Arena::CreateMessage<ShardsStatsResponse>(arena);

            for (auto&& it: ev.ShardStats) {
                auto* shard = resp->AddShards();
                shard->SetnumId(it.NumId);
                shard->SeturlCountLoad(it.UrlCount);
            }

            return resp;
        }
    };

    class TTargetsStatusRequest: public TApiRequest<TTargetsStatusRequest, TFetcherGrpcEvents::TEvTargetsStatusRequest, TEvTargetsStatusResponse> {
        using TBase = TApiRequest<
            TTargetsStatusRequest,
            TFetcherGrpcEvents::TEvTargetsStatusRequest,
            TEvTargetsStatusResponse>;

    public:
        TTargetsStatusRequest(TMetricRegistry& registry, TActorId shardManagerId, THolder<TFetcherGrpcEvents::TEvTargetsStatusRequest> req)
            : TBase{registry, shardManagerId, std::move(req)}
        {
        }

        TEvTargetsStatusRequest* FromInternalEvent(const TFetcherGrpcEvents::TEvTargetsStatusRequest& internal) const {
            auto ev = MakeHolder<TEvTargetsStatusRequest>();
            auto&& request = *internal.Request;

            ev->NumId = request.GetNumId();
            ev->ServiceProviderId = request.GetServiceProviderId();
            ev->HostGlob = request.GethostGlob();
            ev->Status = request.Getstatus();
            ev->Offset = request.Getoffset();
            ev->Limit = request.Getlimit();
            ev->NotOk = request.GetnotOkStatus();
            ev->Dc = request.Getdc().empty()
                ? EDc::UNKNOWN
                : ::FromString(request.Getdc());

            return ev.Release();
        }

        TargetsStatusResponse* CreateReply(const TEvTargetsStatusResponse& ev, NProtoBuf::Arena* arena) const {
            auto* resp = NProtoBuf::Arena::CreateMessage<TargetsStatusResponse>(arena);

            resp->SettotalCount(ev.Total);

            for (auto&& it: ev.Statuses) {
                auto* status = resp->Addtargets();
                status->Sethost(std::move(it.Url.Host));
                status->Setdc(ToString(it.Url.Dc));
                status->SetlastFetchTime(it.Url.FetchTime.MilliSeconds());
                status->SetlastFetchDurationMillis(it.Url.FetchDuration.MilliSeconds());
                status->SetlastResponseBytes(it.Url.ResponseBytes);
                status->SetlastUrlStatus(it.Url.UrlStatus);
                status->SetlastUrlError(it.Url.UrlError);
                status->Seturl(it.Url.Url);
                status->SetlastContentType(it.Url.ContentType);

                for (auto&& [shardId, shardStatus]: it.ShardIdToShardStatus) {
                    auto* ss = status->add_shard_statuses();
                    ss->SetshardId(shardId);
                    ss->SetlastStatus(shardStatus.Status);
                    ss->SetlastMetricsParsed(shardStatus.MetricsParsed);
                    ss->SetlastMetricsOverflow(shardStatus.LastOverflow.Seconds());
                    ss->SetlastError(shardStatus.Error);
                }

                // old API compatibility. TODO(SOLOMON-8837): remove after a successful release
                bool isFetchingError = it.Url.UrlStatus != UrlStatusType::UNKNOWN_STATUS_TYPE &&
                                       it.Url.UrlStatus != UrlStatusType::OK;
                if (isFetchingError || it.ShardIdToShardStatus.empty()) {
                    // we got an error while fetching or parsing before even processing shard data, so let's just copy
                    // the status
                    status->SetlastStatus(it.Url.UrlStatus);
                    status->SetlastError(it.Url.UrlError);
                } else {
                    // for service providers (size > 1) the status value was the last shard status,
                    // so it had no order guarantee and thus we can just select the first status
                    // without actually breaking anything
                    const auto& shard = it.ShardIdToShardStatus.begin()->second;
                    status->SetlastStatus(shard.Status);
                    status->SetlastError(shard.Error);
                    status->SetlastMetricsParsed(shard.MetricsParsed);
                    // not actually used by anyone
                    // status->SetlastMetricsOverflow();
                }
            }

            return resp;
        }
    };

    class TApiProxy: public TActorBootstrapped<TApiProxy> {
    public:
        TApiProxy(TMetricRegistry& registry, TActorId shardManagerId, ui64 executorPool)
            : Registry_{registry}
            , ShardManagerId_{shardManagerId}
            , ConfigUpdaterId_{MakeConfigUpdaterId()}
            , ExecutorPool_{executorPool}
        {
        }

        void Bootstrap(const TActorContext&) {
            Become(&TThis::StateWork);
        }

        STFUNC(StateWork) {
            Y_UNUSED(ctx);

            switch (ev->GetTypeRewrite()) {
                hFunc(TFetcherGrpcEvents::TEvShardsStatsRequest, OnShardStats);
                hFunc(TFetcherGrpcEvents::TEvShardsHealthRequest, OnShardHealth);
                hFunc(TFetcherGrpcEvents::TEvTargetsStatusRequest, OnTargetsStatusRequest);
                hFunc(TFetcherGrpcEvents::TEvReloadShardRequest, OnShardReload);
                hFunc(TFetcherGrpcEvents::TEvResolveCluster, OnResolveCluster);
            }
        }

        void OnShardStats(const TFetcherGrpcEvents::TEvShardsStatsRequest::TPtr& ev) {
            DoRequest<TShardStatsRequest>(ShardManagerId_, THolder(ev->Release().Release()));
        }

        void OnShardHealth(const TFetcherGrpcEvents::TEvShardsHealthRequest::TPtr& ev) {
            DoRequest<TShardHealthRequest>(ShardManagerId_, THolder(ev->Release().Release()));
        }

        void OnTargetsStatusRequest(const TFetcherGrpcEvents::TEvTargetsStatusRequest::TPtr& ev) {
            DoRequest<TTargetsStatusRequest>(ShardManagerId_, THolder(ev->Release().Release()));
        }

        void OnShardReload(const TFetcherGrpcEvents::TEvReloadShardRequest::TPtr& ev) {
            DoRequest<TShardReloadRequest>(ConfigUpdaterId_, THolder(ev->Release().Release()));
        }

        void OnResolveCluster(const TFetcherGrpcEvents::TEvResolveCluster::TPtr& ev) {
            DoRequest<TClusterResolveRequest>(THolder(ev->Release().Release()));
        }

        template <typename TRequest, typename... TArgs>
        void DoRequest(TArgs&&... args) {
            Register(
                new TRequest{Registry_, std::forward<TArgs>(args)...},
                TMailboxType::Simple,
                ExecutorPool_);
        }

    private:
        TMetricRegistry& Registry_;
        TActorId ShardManagerId_;
        TActorId ConfigUpdaterId_;
        ui64 ExecutorPool_;
    };


    class TFetcherService: public TGrpcServiceBase<yandex::monitoring::fetcher::FetcherService> {
    public:
        TFetcherService(TActorId proxyId, TActorSystem& actorSystem, TMetricRegistry& registry)
            : ActorSystem_{actorSystem}
            , ProxyId_{proxyId}
            , StatsFilter_{CreateStatsFilter(NGrpc::TStatsFilterConf{100, {}, TDuration::Seconds(10)},
                    NGrpc::TStatsTableConf{registry, NGrpc::TStatsTableSettings{TDuration::Days(1), 50}})}
        {
            NSelfMon::RegisterPage(ActorSystem_, "/grpc_server", "gRPC Server", NSolomon::NGrpc::StatsPage(StatsFilter_->GetStatsTable(), StatsFilter_));
        }

        bool IncRequest() {
            return Limiter_->Inc();
        }

        void DecRequest() {
            Limiter_->Dec();
        }

        void InitService(grpc::ServerCompletionQueue* cq, TLoggerPtr logger) override {
            using yandex::monitoring::fetcher::FetcherService;

            MakeIntrusive<TGRpcRequest<TargetsStatusRequest, TargetsStatusResponse, TFetcherService>>(
                    this, &Service_, cq, [this, StatsFilterPtr = StatsFilter_](auto* reqCtx) {
                        ActorSystem_.Send(ProxyId_, new TFetcherGrpcEvents::TEvTargetsStatusRequest{StatsFilterPtr->FilterRequest(reqCtx, ActorSystem_.Timestamp())});
                    }, &FetcherService::AsyncService::RequestTargetStatus,
                    "TargetsStatus", logger, FakeCounterBlock())->Run();

            MakeIntrusive<TGRpcRequest<ShardsHealthRequest, ShardsHealthResponse, TFetcherService>>(
                    this, &Service_, cq, [this, StatsFilterPtr = StatsFilter_](auto* reqCtx) {
                        ActorSystem_.Send(ProxyId_, new TFetcherGrpcEvents::TEvShardsHealthRequest{StatsFilterPtr->FilterRequest(reqCtx, ActorSystem_.Timestamp())});
                    }, &FetcherService::AsyncService::RequestShardsHealth,
                    "ShardsHealth", logger, FakeCounterBlock())->Run();

            MakeIntrusive<TGRpcRequest<TReloadShardRequest, TReloadShardResponse, TFetcherService>>(
                    this, &Service_, cq, [this, StatsFilterPtr = StatsFilter_](auto* reqCtx) {
                        ActorSystem_.Send(ProxyId_, new TFetcherGrpcEvents::TEvReloadShardRequest{StatsFilterPtr->FilterRequest(reqCtx, ActorSystem_.Timestamp())});
                    }, &FetcherService::AsyncService::RequestReloadShard,
                    "ReloadShard", logger, FakeCounterBlock())->Run();

            MakeIntrusive<TGRpcRequest<TResolveClusterRequest, TResolveClusterResponse, TFetcherService>>(
                    this, &Service_, cq, [this, StatsFilterPtr = StatsFilter_](auto* reqCtx) {
                        ActorSystem_.Send(ProxyId_, new TFetcherGrpcEvents::TEvResolveCluster{StatsFilterPtr->FilterRequest(reqCtx, ActorSystem_.Timestamp())});
                    }, &FetcherService::AsyncService::RequestResolveCluster,
                    "ResolveCluster", logger, FakeCounterBlock())->Run();
        }

        void SetGlobalLimiterHandle(TGlobalLimiter* limiter) override {
            Limiter_ = limiter;
        }

    private:
        TActorSystem& ActorSystem_;
        TActorId ProxyId_;
        TGlobalLimiter* Limiter_{nullptr};
        std::shared_ptr<NGrpc::IStatsFilter> StatsFilter_;
    };

} // namespace

NActors::IActor* CreateFetcherApiProxy(TMetricRegistry& registry, TActorId shardManagerId, ui64 executorPool) {
    return new TApiProxy{registry, shardManagerId, executorPool};
}

TIntrusivePtr<IGRpcService> CreateFetcherService(TActorId proxyId, TActorSystem& actorSystem, TMetricRegistry& registry) {
    return MakeIntrusive<TFetcherService>(proxyId, actorSystem, registry);
}

} // namespace NSolomon::NFetcher
