#include <solomon/services/fetcher/lib/api/api.h>

#include <solomon/services/fetcher/client/fetcher_client.h>
#include <solomon/services/fetcher/lib/shard_manager/shard_resolver.h>
#include <solomon/services/fetcher/testlib/actor_system.h>

#include <solomon/protos/configs/rpc/rpc_config.pb.h>

#include <library/cpp/grpc/server/grpc_server.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/common/network.h>

#include <library/cpp/actors/core/actor.h>

using namespace testing;
using namespace NActors;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;

using yandex::solomon::config::rpc::TGrpcClientConfig;
using yandex::solomon::config::rpc::TGrpcServerConfig;

namespace {
    THolder<NGrpc::TGRpcServer> CreateGrpcServer(const TGrpcServerConfig& config) {
        NGrpc::TServerOptions opts;

        opts.SetHost("localhost")
            .SetPort(config.GetPort(0))
            .SetWorkerThreads(1);

        return MakeHolder<NGrpc::TGRpcServer>(opts);
    }

class TApiTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorSystem_ = MakeActorSystem(true);
        ActorSystem_->Initialize();

        auto* proxy = CreateFetcherApiProxy(Registry_, TActorId{}, 0);
        const auto proxyId = ActorSystem_->Register(proxy);
        ActorSystem_->WaitForBootstrap();

        Service_ = CreateFetcherService(proxyId, *ActorSystem_->SingleSys(), Registry_);
        PortHolder_ = NTesting::GetFreePort();

        TGrpcServerConfig serverConf;
        serverConf.AddPort(PortHolder_);
        Server_ = CreateGrpcServer(serverConf);
        Server_->AddService(Service_);
        Server_->Start();

        TGrpcClientConfig clientConf;
        clientConf.AddAddresses("localhost:" + ToString<ui16>(PortHolder_));
        Client_ = CreateGrpcClient(clientConf, Registry_);
    }

    void TearDown() override {
        Server_->Stop();
    }

protected:
    NMonitoring::TMetricRegistry Registry_;
    THolder<TFetcherActorRuntime> ActorSystem_;
    TIntrusivePtr<NGrpc::IGRpcService> Service_;
    NTesting::TPortHolder PortHolder_;
    THolder<NGrpc::TGRpcServer> Server_;
    IFetcherClientPtr Client_;
};

} // namespace

class TClusterResolveTest: public TApiTest {
};

TEST_F(TClusterResolveTest, EmptyCluster) {
    auto f = Client_->ResolveCluster({});
    ActorSystem_->DispatchEvents(TDispatchOptions{}, TDuration::Zero());
    auto v = f.ExtractValueSync();
    ASSERT_TRUE(v.Success());
    ASSERT_TRUE(v.Value().Groups.empty());
}

TEST_F(TClusterResolveTest, SingleConfig) {
    TClusterResolveRequest req;
    auto& group = req.Groups.emplace_back();
    group.Type = TClusterResolveRequest::EGroupType::Host;
    group.JsonConfig = R"([
        {"urlPattern":"distbuild00d.search.yandex.net","ranges":"","dc":"","labels":[]},
        {"urlPattern":"distbuild01d.search.yandex.net","ranges":"","dc":"","labels":[]}
    ])";
    auto f = Client_->ResolveCluster(std::move(req));
    auto v = f.ExtractValueSync();
    ASSERT_TRUE(v.Success());

    const TVector<THostAndLabels> expected{
        THostAndLabels{"distbuild00d.search.yandex.net"},
        THostAndLabels{"distbuild01d.search.yandex.net"},
    };

    ASSERT_THAT(v.Value().Groups.size(), Eq(2u));

    TVector<THostAndLabels> actual;
    for (auto& g: v.Value().Groups) {
        ASSERT_TRUE(g.Result.Success());

        for (auto& host: g.Result.Value()) {
            actual.push_back(std::move(host));
        }
    }

    ASSERT_THAT(actual, UnorderedElementsAreArray(expected));
}

TEST_F(TClusterResolveTest, PartiallyResolved) {
    TClusterResolveRequest req;
    {
        auto& group = req.Groups.emplace_back();
        group.Type = TClusterResolveRequest::EGroupType::Host;
        group.JsonConfig = R"([{"urlPattern":"distbuild00d.search.yandex.net","ranges":"","dc":"","labels":[]}])";
    }

    {
        auto& group = req.Groups.emplace_back();
        group.Type = TClusterResolveRequest::EGroupType::HostUrl;
        group.JsonConfig = R"([{}])";
    }

    auto f = Client_->ResolveCluster(std::move(req));
    auto v = f.ExtractValueSync();
    ASSERT_TRUE(v.Success());
    ASSERT_THAT(v.Value().Groups.size(), Eq(2u));

    TVector<THostAndLabels> actual;
    ui32 successCount{0};
    ui32 failCount{0};

    for (auto& g: v.Value().Groups) {
        if (g.Result.Success()) {
            successCount++;
        } else {
            failCount++;
            continue;
        }

        for (auto& host: g.Result.Value()) {
            actual.push_back(std::move(host));
        }
    }

    const TVector<THostAndLabels> expected{
        THostAndLabels{"distbuild00d.search.yandex.net"},
    };

    ASSERT_THAT(actual, UnorderedElementsAreArray(expected));
    ASSERT_THAT(successCount, Eq(1u));
    ASSERT_THAT(failCount, Eq(1u));
}
