#pragma once

#include "metric_verbosity.h"

#include <solomon/services/fetcher/config/fetcher_config.pb.h>
#include <solomon/services/fetcher/config/resolver_config.pb.h>
#include <solomon/services/fetcher/lib/host_groups/factory.h>

#include <solomon/libs/cpp/auth/tvm/tvm.h>
#include <solomon/libs/cpp/cloud/access_service/access_service.h>
#include <solomon/libs/cpp/cloud/iam/client.h>
#include <solomon/libs/cpp/cloud/instance_group/client.h>
#include <solomon/libs/cpp/dns/dns.h>
#include <solomon/libs/cpp/http/client/curl/client.h>

#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/executor_thread.h>
#include <library/cpp/logger/log.h>
#include <library/cpp/monlib/dynamic_counters/counters.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NFetcher {

struct TFetchConfig {
    TDuration ConnectTimeout;
    size_t DownloadPoolId{0};
    size_t ParsePoolId{0};
    bool DoSkipAuth{false};
    bool SkipProviderDataWithWrongService{false};
};

struct TAppData: TNonCopyable {
    void Deinit() {
        // must be destroyed before all clients
        ResolverFactory_.Reset();

        DnsClient.Reset();
        DataHttpClient_.Reset();
        if (ResolverHttpClient_) {
            ResolverHttpClient_.Reset();
        }

        if (!IamExecutors.empty()) {
            for (auto& ex: IamExecutors) {
                ex->Stop();
            }
        }

        TicketProvider.Reset();
    }

    std::shared_ptr<NMonitoring::TMetricRegistry> Metrics;
    /**
     * Registry that keeps metrics related to specific shards
     */
    std::shared_ptr<NMonitoring::TMetricRegistry> ShardMetrics;
    IDnsClientPtr DnsClient;
    std::optional<TString> YpToken;
    THttpClientConfig HttpClientConfig;
    TVector<THolder<NCloud::IExecutor>> IamExecutors;
    NCloud::IIamClientPtr IamClient;
    IAccessServiceClientPtr AccessServiceClient;
    NCloud::IInstanceGroupClientPtr InstanceGroupClient;
    NAuth::NTvm::ITvmClientPtr TvmClient;
    NAuth::NTvm::ITicketProviderPtr TicketProvider;
    NActors::TActorId RackTablesActorId;
    NActors::TActorId AuthenticationActorId;

    TAppData()
        : Metrics{std::make_shared<NMonitoring::TMetricRegistry>()}
        , ShardMetrics{std::make_shared<NMonitoring::TMetricRegistry>()}
    {
    }

    IHttpClientPtr DataHttpClient() const {
        return DataHttpClient_;
    }

    void SetDataHttpClient(IHttpClientPtr httpClient) {
        DataHttpClient_ = std::move(httpClient);
    }

    IHttpClientPtr ResolverHttpClient() const {
        return ResolverHttpClient_;
    }

    void SetResolverHttpClient(IHttpClientPtr httpClient) {
        ResolverHttpClient_ = std::move(httpClient);
    }

    IHostResolverFactoryPtr ResolverFactory() const {
        return ResolverFactory_;
    }

    void SetResolverFactory(IHostResolverFactoryPtr factory) {
        ResolverFactory_ = std::move(factory);
    }

    void SetFetchConfig(TFetchConfig fetchConf) {
        FetchConf_ = fetchConf;
    }

    const TFetchConfig& FetchConfig() const {
        return FetchConf_;
    }

    void SetAccessServiceConfig(TAccessServiceConfig conf) {
        AccessServiceConf_ = std::move(conf);
    }

    const std::optional<TAccessServiceConfig>& AccessServiceConfig() const {
        return AccessServiceConf_;
    }

    void SetClusterInfo(TClusterInfo val) {
        ClusterInfo_ = std::move(val);
    }

    const TClusterInfo& ClusterInfo() const {
        return ClusterInfo_;
    }

    EMetricVerbosity MetricVerbosity() const {
        return MetricVerbosity_;
    }

    void SetMetricVerbosity(EMetricVerbosity verbosity) {
        MetricVerbosity_ = verbosity;
    }

    void SetMetricVerbosity(TFetcherConfig::EMetricVerbosity verbosity) {
        switch (verbosity) {
            case TFetcherConfig::ALL_METRICS:
                MetricVerbosity_ = EMetricVerbosity::All;
                break;
            case TFetcherConfig::ONLY_TOTALS:
                MetricVerbosity_ = EMetricVerbosity::OnlyTotal;
                break;
            default:
                Cerr << "Unknown metric verbosity: " << TFetcherConfig::EMetricVerbosity_Name(verbosity) << Endl;
        }
    }

    void AddIamTokenProvider(const TString& serviceAccountId, NCloud::ITokenProviderPtr tokenProvider) {
        Y_VERIFY(
                !IamTokenProviders_.contains(serviceAccountId),
                "IAM service account \"%s\" already has token provider",
                serviceAccountId.c_str());
        IamTokenProviders_[serviceAccountId] = std::move(tokenProvider);
    }

    NCloud::ITokenProviderPtr GetIamTokenProvider(const TString& serviceAccountId) const {
        auto it = IamTokenProviders_.find(serviceAccountId);
        if (it == IamTokenProviders_.end()) {
            return {};
        }
        return it->second;
    }

private:
    IHttpClientPtr DataHttpClient_;
    IHttpClientPtr ResolverHttpClient_;
    TFetchConfig FetchConf_;
    std::optional<TAccessServiceConfig> AccessServiceConf_;
    TClusterInfo ClusterInfo_;
    IHostResolverFactoryPtr ResolverFactory_;
    EMetricVerbosity MetricVerbosity_{EMetricVerbosity::All};
    // key - service account id
    THashMap<TString, NCloud::ITokenProviderPtr> IamTokenProviders_;
};

inline const TAppData& GetAppData() {
    return *NActors::TActorContext::ActorSystem()->AppData<TAppData>();
}

} // NSolomon::NFetcher
