#include "access_service.h"

#include <solomon/libs/cpp/actors/scheduler/scheduler.h>
#include <solomon/libs/cpp/backoff/backoff.h>
#include <solomon/libs/cpp/backoff/jitter.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

#include <util/digest/multi.h>

#include <set>

namespace NSolomon::NFetcher {
namespace {

using namespace NActors;
using namespace yandex::cloud::priv::servicecontrol::v1;

const TString INTERNAL_ERR{"internal error on auth"};
const TString TIMEOUT_MSG{"the request has timed out"};
const TString RESOURCE_TYPE_FOLDER{"resource-manager.folder"};
const TString WRITE_PERMISSION{"monitoring.data.write"};

struct TRequestCtx {
    TActorId ReplyTo;
    ui64 Cookie{};
    TInstant ReceivedAt;
    TExpBackoff<THalfJitter> Backoff;
    ui8 CurrentTryCnt{1};
    /**
     * id of an event constructed by the Scheduler actor
     */
    ui64 RetryId{0};
    ui64 TimeoutId{0};
    bool IsInflight{false};

    TRequestCtx(TActorId replyTo, ui64 cookie, TInstant now, TDuration minDelay, TDuration maxDelay)
        : ReplyTo{replyTo}
        , Cookie{cookie}
        , ReceivedAt{now}
        , Backoff{minDelay, maxDelay}
    {
    }
};

struct TAuthenticationCtx: public TRequestCtx {
    TString Token;

    TAuthenticationCtx(TActorId replyTo, ui64 cookie, TInstant now, TDuration minDelay, TDuration maxDelay, TString token)
        : TRequestCtx{replyTo, cookie, now, minDelay, maxDelay}
        , Token{std::move(token)}
    {
    }
};

struct TAuthorizationCtx: public TRequestCtx {
    AuthorizeRequest Req;

    TAuthorizationCtx(TActorId replyTo, ui64 cookie, TInstant now, TDuration minDelay, TDuration maxDelay, AuthorizeRequest req)
        : TRequestCtx{replyTo, cookie, now, minDelay, maxDelay}
        , Req{std::move(req)}
    {
    }
};

template <typename T>
struct TPtrCompare {
    bool operator()(const std::unique_ptr<T>& left, const std::unique_ptr<T>& right) const {
        return left.get() == right.get();
    }
};

struct TPostponedReqCompare {
    bool operator()(const TRequestCtx* left, const TRequestCtx* right) const {
        return left->ReceivedAt < right->ReceivedAt;
    }
};

class TMetrics {
public:
    explicit TMetrics(std::shared_ptr<NMonitoring::TMetricRegistry> registry)
        : Registry_{std::move(registry)}
        , Inflight{Registry_->IntGauge({ {"sensor", "auth.inflight"} })}
        , Postponed{Registry_->IntGauge({ {"sensor", "auth.postponed"} })}
    {
    }

private:
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry_;

public:
    NMonitoring::IIntGauge* Inflight;
    NMonitoring::IIntGauge* Postponed;
};

class TAccessServiceActor: public NActors::TActor<TAccessServiceActor> {
    struct TPrivateEvents: private NSolomon::TPrivateEvents {
        enum {
            AuthenticationFailure = SpaceBegin,
            AuthorizationFailure,
            AuthenticationTimeout,
            AuthorizationTimeout,
            FreeAuthenticationCtx,
            FreeAuthorizationCtx,
            AuthenticationRetry,
            AuthorizationRetry,
            End
        };

        static_assert(End < SpaceEnd, "too many event types");

        struct TAuthenticationFailure: NActors::TEventLocal<TAuthenticationFailure, AuthenticationFailure> {
            EAccessServiceAuthErrorType Type;
            TString Message;
            TAuthenticationCtx* Ctx;

            TAuthenticationFailure(decltype(Type) type, TString msg, TAuthenticationCtx* ctx)
                : Type{type}
                , Message{std::move(msg)}
                , Ctx{ctx}
            {}
        };

        struct TAuthorizationFailure: NActors::TEventLocal<TAuthorizationFailure, AuthorizationFailure> {
            EFailureType Type;
            TString Message;
            TAuthorizationCtx* Ctx;
            bool ShouldRetry{true};

            TAuthorizationFailure(decltype(Type) type, TString msg, TAuthorizationCtx* ctx, bool shouldRetry)
                : Type{type}
                , Message{std::move(msg)}
                , Ctx{ctx}
                , ShouldRetry{shouldRetry}
            {}
        };

        struct TFreeAuthenticationCtx: NActors::TEventLocal<TFreeAuthenticationCtx, FreeAuthenticationCtx> {
            TAuthenticationCtx* Ctx;

            explicit TFreeAuthenticationCtx(TAuthenticationCtx* ctx)
                : Ctx{ctx}
            {}
        };

        struct TFreeAuthorizationCtx: NActors::TEventLocal<TFreeAuthorizationCtx, FreeAuthorizationCtx> {
            TAuthorizationCtx* Ctx;

            explicit TFreeAuthorizationCtx(TAuthorizationCtx* ctx)
                : Ctx{ctx}
            {}
        };

        struct TAuthenticationRetry: NActors::TEventLocal<TAuthenticationRetry, AuthenticationRetry> {
            ui64 RetryId;
            TAuthenticationCtx* Ctx;
        };

        struct TAuthorizationRetry: NActors::TEventLocal<TAuthorizationRetry, AuthorizationRetry> {
            ui64 RetryId;
            TAuthorizationCtx* Ctx;
        };

        struct TAuthenticationTimeout: NActors::TEventLocal<TAuthenticationTimeout, AuthenticationTimeout> {
            ui64 TimeoutId;
            TAuthenticationCtx* Ctx;
        };

        struct TAuthorizationTimeout: NActors::TEventLocal<TAuthorizationTimeout, AuthorizationTimeout> {
            ui64 TimeoutId;
            TAuthorizationCtx* Ctx;
        };
    };

public:
    TAccessServiceActor(
            IAccessServiceClientPtr accessServiceClient,
            TActorId scheduler,
            const TAccessServiceConfig& config,
            std::shared_ptr<NMonitoring::TMetricRegistry> registry)
        : NActors::TActor<TAccessServiceActor>(&TThis::StateWork)
        , AccessServiceClient_{std::move(accessServiceClient)}
        , Scheduler_{scheduler}
        , MaxTries_{config.GetRetriesCnt() + 1}
        , MaxInflight_{config.GetMaxInflight()}
        , MinRetryDelay_{FromProtoTime(config.GetMinRetryDelay())}
        , MaxRetryDelay_{FromProtoTime(config.GetMaxRetryDelay())}
        , Timeout_{FromProtoTime(config.GetTimeout())}
        , Metrics_{std::move(registry)}
    {
    }

    STATEFN(StateWork) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TAccessServiceEvents::TAuthenticate, OnAuthenticate);
            hFunc(TAccessServiceEvents::TAuthorize, OnAuthorize);
            hFunc(TEvents::TEvPoison, OnPoison);

            // internal
            hFunc(TPrivateEvents::TAuthenticationFailure, OnAuthenticationFail);
            hFunc(TPrivateEvents::TAuthorizationFailure, OnAuthorizationFail);
            hFunc(TPrivateEvents::TAuthenticationRetry, OnAuthenticationRetry);
            hFunc(TPrivateEvents::TAuthorizationRetry, OnAuthorizationRetry);
            hFunc(TPrivateEvents::TFreeAuthenticationCtx, OnFreeAuthenticationCtx);
            hFunc(TPrivateEvents::TFreeAuthorizationCtx, OnFreeAuthorizationCtx);
            hFunc(TPrivateEvents::TAuthenticationTimeout, OnAuthenticationTimeout);
            hFunc(TPrivateEvents::TAuthorizationTimeout, OnAuthorizationTimeout);
        }
    }

private:
    void IncInflight() {
        ++Inflight_;
        Metrics_.Inflight->Inc();
    }

    void DecInflight() {
        Y_VERIFY(Inflight_ > 0);
        --Inflight_;
        Metrics_.Inflight->Dec();

        if (IsPassingAway_ && Inflight_ == 0) {
            Authentications_.clear();
            Authorizations_.clear();

            Send(ReplyToBeforePassignAway_, new TEvents::TEvPoisonTaken);
            PassAway();
        }
    }

    void OnAuthenticate(const TAccessServiceEvents::TAuthenticate::TPtr& evPtr) {
        auto authCtx = std::make_unique<TAuthenticationCtx>(
            evPtr->Sender,
            evPtr->Cookie,
            TActivationContext::Now(),
            MinRetryDelay_,
            MaxRetryDelay_,
            evPtr->Get()->IamToken);

        auto* ctxAddr = authCtx.get();
        Authentications_.emplace(ctxAddr, std::move(authCtx));

        if (Inflight_ >= MaxInflight_) {
            PostponedAuthentications_.emplace(ctxAddr);
            Metrics_.Postponed->Inc();
            return;
        }

        auto evId = ++ScheduledEventCnt_;
        ScheduledEvents_.emplace(evId);
        ctxAddr->TimeoutId = evId;

        auto replyEv = std::make_unique<TPrivateEvents::TAuthenticationTimeout>();
        replyEv->TimeoutId = evId;
        replyEv->Ctx = ctxAddr;

        Send(Scheduler_, new TSchedulerEvents::TScheduleAfter{evId, Timeout_, std::move(replyEv)});
        SendAuthenticateRequest(ctxAddr);
    }

    void OnAuthenticationTimeout(const TPrivateEvents::TAuthenticationTimeout::TPtr& evPtr) {
        auto timeoutId = evPtr->Get()->TimeoutId;
        auto* ctx = evPtr->Get()->Ctx;

        if (!ScheduledEvents_.contains(timeoutId)) {
            return;
        }

        if (auto retryId = ctx->RetryId) {
            Send(Scheduler_, new TSchedulerEvents::TCancel{retryId});
        }

        Send(
            ctx->ReplyTo,
            new TAccessServiceEvents::TAuthenticationFailure{EFailureType::Internal, TIMEOUT_MSG},
            0,
            ctx->Cookie);

        if (ctx->IsInflight) {
            // we cannot delete it right now, let's just mark it -- it will be deleted later
            ctx->ReplyTo = {};
            return;
        }

        FreeAuthenticationCtx(ctx);
    }

    void OnFreeAuthenticationCtx(const TPrivateEvents::TFreeAuthenticationCtx::TPtr& evPtr) {
        DecInflight();
        FreeAuthenticationCtx(evPtr->Get()->Ctx);
    }

    void FreeAuthenticationCtx(TAuthenticationCtx* ctx) {
        if (ctx->RetryId) {
            ScheduledEvents_.erase(ctx->RetryId);
        }

        if (ctx->TimeoutId) {
            ScheduledEvents_.erase(ctx->TimeoutId);
        }

        auto postponedIt = PostponedAuthentications_.find(ctx);
        while (postponedIt != PostponedAuthentications_.end() && (*postponedIt)->ReceivedAt == ctx->ReceivedAt) {
            if (ctx == *postponedIt) {
                PostponedAuthentications_.erase(postponedIt);
                Metrics_.Postponed->Dec();
                break;
            }

            ++postponedIt;
        }

        Authentications_.erase(ctx);
        ProcessPostponedRequests();
    }

    void ProcessPostponedRequests() {
        TAuthenticationCtx* authentication = PostponedAuthentications_.empty() ? nullptr : *PostponedAuthentications_.begin();
        TAuthorizationCtx* authorization = PostponedAuthorizations_.empty() ? nullptr : *PostponedAuthorizations_.begin();

        if (!authentication && !authorization) {
            return;
        }

        if (authentication && authorization) {
            if (authorization->ReceivedAt < authentication->ReceivedAt) {
                authentication = nullptr;
            } else {
                authorization = nullptr;
            }
        }

        if (authorization) {
            PostponedAuthorizations_.erase(PostponedAuthorizations_.begin());
            Metrics_.Postponed->Dec();
            SendAuthorizeRequest(authorization);
            return;
        }

        if (authentication) {
            PostponedAuthentications_.erase(PostponedAuthentications_.begin());
            Metrics_.Postponed->Dec();
            SendAuthenticateRequest(authentication);
        }
    }

    void SendAuthenticateRequest(TAuthenticationCtx* ctx) {
        IncInflight();
        auto* actorSystem = NActors::TActorContext::ActorSystem();
        ctx->IsInflight = true;

        AccessServiceClient_->Authenticate(ctx->Token)
            .Subscribe(
                [actorSystem{actorSystem}, self{SelfId()}, ctx{ctx}]
                (const TAsyncAuthenticateResponse& future) mutable {
                    try {
                        auto resOrErr = TAsyncAuthenticateResponse{future}.ExtractValue();

                        if (resOrErr.Fail()) {
                            TAuthError err = resOrErr.ExtractError();
                            auto ev = std::make_unique<TPrivateEvents::TAuthenticationFailure>(
                                    err.Type,
                                    std::move(err.Message),
                                    ctx);

                            actorSystem->Send(self, ev.release());
                        } else {
                            auto ev = std::make_unique<TAccessServiceEvents::TAuthenticationSuccess>(
                                    ctx->Token,
                                    resOrErr.Extract());
                            actorSystem->Send(new IEventHandle{ctx->ReplyTo, self, ev.release(), 0, ctx->Cookie});

                            actorSystem->Send(self, new TPrivateEvents::TFreeAuthenticationCtx{ctx});
                        }
                    } catch (...) {
                        auto ev = std::make_unique<TPrivateEvents::TAuthenticationFailure>(
                                EAccessServiceAuthErrorType::InternalNonRetriable,
                                CurrentExceptionMessage(),
                                ctx);

                        actorSystem->Send(self, ev.release());
                    }
                });
    }

    void OnAuthenticationFail(const TPrivateEvents::TAuthenticationFailure::TPtr& evPtr) {
        const auto& ev = *(evPtr->Get());
        TAuthenticationCtx* ctx = ev.Ctx;

        DecInflight();
        ctx->IsInflight = false;

        if (!ctx->ReplyTo) {
            FreeAuthenticationCtx(ctx);
            return;
        }

        switch (ev.Type) {
            case EAccessServiceAuthErrorType::InternalNonRetriable:
                MON_ERROR(AccessService, ev.Message);
                Send(
                    ctx->ReplyTo,
                    new TAccessServiceEvents::TAuthenticationFailure{EFailureType::Internal, INTERNAL_ERR},
                    0,
                    ctx->Cookie);

                FreeAuthenticationCtx(ctx);
                break;
            case EAccessServiceAuthErrorType::FailedAuth:
                MON_INFO(AccessService, ev.Message);
                Send(
                    ctx->ReplyTo,
                    new TAccessServiceEvents::TAuthenticationFailure{EFailureType::FailedAuth, ev.Message},
                    0,
                    ctx->Cookie);

                FreeAuthenticationCtx(ctx);
                break;
            case EAccessServiceAuthErrorType::InternalRetriable:
                MON_ERROR(AccessService, ev.Message);

                if (ctx->CurrentTryCnt < MaxTries_) {
                    auto retryId = ++ScheduledEventCnt_;
                    ScheduledEvents_.emplace(retryId);
                    ctx->RetryId = retryId;

                    auto retryEv = std::make_unique<TPrivateEvents::TAuthenticationRetry>();
                    retryEv->RetryId = retryId;
                    retryEv->Ctx = ctx;

                    Send(Scheduler_, new TSchedulerEvents::TScheduleAfter{retryId, ctx->Backoff(), std::move(retryEv)});
                    return;
                }

                Send(
                    ctx->ReplyTo,
                    new TAccessServiceEvents::TAuthenticationFailure{EFailureType::Internal, INTERNAL_ERR},
                    0,
                    ctx->Cookie);

                FreeAuthenticationCtx(ctx);
                break;
        }
    }

    void OnAuthenticationRetry(const TPrivateEvents::TAuthenticationRetry::TPtr& evPtr) {
        auto retryId = evPtr->Get()->RetryId;
        auto* ctx = evPtr->Get()->Ctx;

        if (!ScheduledEvents_.contains(retryId)) {
            // the request was completed or canceled, nothing to process
            return;
        }

        ctx->RetryId = 0;

        ++ctx->CurrentTryCnt;
        SendAuthenticateRequest(ctx);
    }

    void OnAuthorize(const TAccessServiceEvents::TAuthorize::TPtr& evPtr) {
        auto& ev = (*evPtr->Get());

        AuthorizeRequest req;
        req.mutable_subject()->mutable_service_account()->set_id(ev.ServiceAccountId);
        auto* resource = req.add_resource_path();
        resource->set_type(RESOURCE_TYPE_FOLDER);
        resource->set_id(ev.FolderId);
        req.set_permission(WRITE_PERMISSION);

        auto authCtx = std::make_unique<TAuthorizationCtx>(
            evPtr->Sender,
            evPtr->Cookie,
            TActivationContext::Now(),
            MinRetryDelay_,
            MaxRetryDelay_,
            std::move(req));

        auto* ctxAddr = authCtx.get();
        Authorizations_.emplace(ctxAddr, std::move(authCtx));

        if (Inflight_ >= MaxInflight_) {
            PostponedAuthorizations_.emplace(ctxAddr);
            Metrics_.Postponed->Inc();
            return;
        }

        auto evId = ++ScheduledEventCnt_;
        ScheduledEvents_.emplace(evId);
        ctxAddr->TimeoutId = evId;

        auto replyEv = std::make_unique<TPrivateEvents::TAuthorizationTimeout>();
        replyEv->TimeoutId = evId;
        replyEv->Ctx = ctxAddr;

        Send(Scheduler_, new TSchedulerEvents::TScheduleAfter{evId, Timeout_, std::move(replyEv)});
        SendAuthorizeRequest(ctxAddr);
    }

    void OnAuthorizationTimeout(const TPrivateEvents::TAuthorizationTimeout::TPtr& evPtr) {
        auto timeoutId = evPtr->Get()->TimeoutId;
        auto* ctx = evPtr->Get()->Ctx;

        if (!ScheduledEvents_.contains(timeoutId)) {
            return;
        }

        if (auto retryId = ctx->RetryId) {
            Send(Scheduler_, new TSchedulerEvents::TCancel{retryId});
        }

        Send(
            ctx->ReplyTo,
            new TAccessServiceEvents::TAuthorizationFailure{EFailureType::Internal, TIMEOUT_MSG},
            0,
            ctx->Cookie);

        if (ctx->IsInflight) {
            // we cannot delete it right now, let's just mark it -- it will be deleted later
            ctx->ReplyTo = {};
            return;
        }

        FreeAuthorizationCtx(ctx);
    }

    static bool IsAuthorizationRetryable(grpc::StatusCode code) {
        switch (code) {
            case grpc::StatusCode::UNAUTHENTICATED:
            case grpc::StatusCode::PERMISSION_DENIED:
            case grpc::StatusCode::INVALID_ARGUMENT:
                return false;
            default:
                return true;
        }
    }

    void OnFreeAuthorizationCtx(const TPrivateEvents::TFreeAuthorizationCtx::TPtr& evPtr) {
        DecInflight();
        FreeAuthorizationCtx(evPtr->Get()->Ctx);
    }

    void FreeAuthorizationCtx(TAuthorizationCtx* ctx) {
        if (ctx->RetryId) {
            ScheduledEvents_.erase(ctx->RetryId);
        }

        if (ctx->TimeoutId) {
            ScheduledEvents_.erase(ctx->TimeoutId);
        }

        auto postponedIt = PostponedAuthorizations_.find(ctx);
        while (postponedIt != PostponedAuthorizations_.end() && (*postponedIt)->ReceivedAt == ctx->ReceivedAt) {
            if (ctx == *postponedIt) {
                PostponedAuthorizations_.erase(postponedIt);
                Metrics_.Postponed->Dec();
                break;
            }

            ++postponedIt;
        }

        Authorizations_.erase(ctx);
        ProcessPostponedRequests();
    }

    void SendAuthorizeRequest(TAuthorizationCtx* ctx) {
        IncInflight();
        ctx->IsInflight = true;

        auto* actorSystem = NActors::TActorContext::ActorSystem();

        AccessServiceClient_->Authorize(ctx->Req)
            .Subscribe(
                [this, actorSystem{actorSystem}, self{SelfId()}, ctx]
                (TAsyncAuthorizeResponse future) mutable {
                    try {
                        auto resOrErr = future.ExtractValue();

                        if (resOrErr.Fail()) {
                            auto err = resOrErr.ExtractError();
                            auto grpcCode = static_cast<grpc::StatusCode>(err.GRpcStatusCode);

                            if (err.InternalError || !IsAuthorizationRetryable(grpcCode) || ctx->CurrentTryCnt >= MaxTries_) {
                                auto ev = std::make_unique<TPrivateEvents::TAuthorizationFailure>(
                                    err.InternalError ? EFailureType::Internal : EFailureType::FailedAuth,
                                    std::move(err.Msg),
                                    ctx,
                                    false);

                                actorSystem->Send(self, ev.release());
                                return;
                            }

                            auto ev = std::make_unique<TPrivateEvents::TAuthorizationFailure>(
                                err.InternalError ? EFailureType::Internal : EFailureType::FailedAuth,
                                std::move(err.Msg),
                                ctx,
                                true);

                            actorSystem->Send(self, ev.release());
                        } else {
                            auto ev = std::make_unique<TAccessServiceEvents::TAuthorizationSuccess>();
                            auto eh = std::make_unique<IEventHandle>(ctx->ReplyTo, self, ev.release(), 0, ctx->Cookie);
                            actorSystem->Send(eh.release());

                            actorSystem->Send(self, new TPrivateEvents::TFreeAuthorizationCtx{ctx});
                        }
                    } catch (...) {
                        auto ev = std::make_unique<TPrivateEvents::TAuthorizationFailure>(
                            EFailureType::Internal,
                            CurrentExceptionMessage(),
                            ctx,
                            false);

                        actorSystem->Send(self, ev.release());
                    }
                });
    }

    void LogAuthorizationError(const TString& errMsg, const AuthorizeRequest& req, EFailureType failureType) {
        auto logMsg = TStringBuilder()
            << "error while authorizing (sa_id=" << req.subject().service_account().id()
            << ", folder=" << req.resource_path().Get(0).id() << "): "
            << errMsg;

        if (failureType == EFailureType::Internal) {
            MON_ERROR(AccessService, logMsg);
        } else {
            MON_INFO(AccessService, logMsg);
        }
    }

    void OnAuthorizationFail(const TPrivateEvents::TAuthorizationFailure::TPtr& evPtr) {
        const auto& ev = *(evPtr->Get());
        TAuthorizationCtx* ctx = ev.Ctx;

        DecInflight();
        ctx->IsInflight = false;

        if (!ctx->ReplyTo) {
            FreeAuthorizationCtx(ctx);
            return;
        }

        if (!ev.ShouldRetry || ctx->CurrentTryCnt >= MaxTries_) {
            LogAuthorizationError(ev.Message, ctx->Req, ev.Type);

            TString msg = ev.Type == EFailureType::Internal ? INTERNAL_ERR : ev.Message;

            Send(
                ctx->ReplyTo,
                new TAccessServiceEvents::TAuthorizationFailure{ev.Type, std::move(msg)},
                0,
                ctx->Cookie);

            FreeAuthorizationCtx(ctx);
            return;
        }

        auto retryId = ++ScheduledEventCnt_;
        ScheduledEvents_.emplace(retryId);
        ctx->RetryId = retryId;

        auto retryEv = std::make_unique<TPrivateEvents::TAuthorizationRetry>();
        retryEv->RetryId = retryId;
        retryEv->Ctx = ctx;

        Send(Scheduler_, new TSchedulerEvents::TScheduleAfter{retryId, ctx->Backoff(), std::move(retryEv)});
    }

    void OnAuthorizationRetry(const TPrivateEvents::TAuthorizationRetry::TPtr& evPtr) {
        auto retryId = evPtr->Get()->RetryId;
        auto* ctx = evPtr->Get()->Ctx;

        if (!ScheduledEvents_.contains(retryId)) {
            // the request was completed or canceled, nothing to process
            return;
        }

        ctx->RetryId = 0;

        ++ctx->CurrentTryCnt;
        SendAuthorizeRequest(ctx);
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& evPtr) {
        if (Inflight_ == 0) {
            Send(evPtr->Sender, new TEvents::TEvPoisonTaken);
            PassAway();
            return;
        }

        IsPassingAway_ = true;
        ReplyToBeforePassignAway_ = evPtr->Sender;
    }

private:
    IAccessServiceClientPtr AccessServiceClient_;
    TActorId Scheduler_;
    size_t MaxTries_{1};
    size_t MaxInflight_{100};
    TDuration MinRetryDelay_{};
    TDuration MaxRetryDelay_{};
    TDuration Timeout_;
    // TODO(ivanzhukov): apply ewma, 'cause otherwise UI shows only 0
    TMetrics Metrics_;

    size_t Inflight_{0};
    // TODO(ivanzhukov): make code more understandable + allocate one contiguous vector instead of a map<raw_ptr, smart_ptr>
    absl::flat_hash_map<TAuthenticationCtx*, std::unique_ptr<TAuthenticationCtx>> Authentications_;
    absl::flat_hash_map<TAuthorizationCtx*, std::unique_ptr<TAuthorizationCtx>> Authorizations_;
    std::multiset<TAuthenticationCtx*, TPostponedReqCompare> PostponedAuthentications_;
    std::multiset<TAuthorizationCtx*, TPostponedReqCompare> PostponedAuthorizations_;
    absl::flat_hash_set<ui64> ScheduledEvents_; // retries or timeouts
    size_t ScheduledEventCnt_{0};
    bool IsPassingAway_{false};
    TActorId ReplyToBeforePassignAway_;
};

} // namespace

/**
 * This actor is used only from AuthGatekeeper actor. Thus, cache and subscriptions are implemented inside AuthGatekeeper
 */
std::unique_ptr<NActors::IActor> CreateAccessServiceActor(
        IAccessServiceClientPtr accessServiceClient,
        TActorId scheduler,
        TAccessServiceConfig config,  // NOLINT(performance-unnecessary-value-param): false positive
        std::shared_ptr<NMonitoring::TMetricRegistry> registry)
{
    return std::make_unique<TAccessServiceActor>(
        std::move(accessServiceClient),
        scheduler,
        std::move(config),
        std::move(registry));
}

} // namespace NSolomon::NFetcher
