#include "conductor.h"

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/http/http.h>
#include <library/cpp/actors/http/http_proxy.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <util/generic/hash.h>
#include <util/string/split.h>

using namespace NActors;
using namespace NHttp;


namespace NSolomon::NFetcher {
namespace {
    TVector<TString> ParseGroupHosts(TStringBuf raw) {
        TVector<TString> result;
        StringSplitter(raw)
            .Split('\n')
            .SkipEmpty()
            .Collect(&result);

        return result;
    }

    constexpr TStringBuf GROUP2HOSTS = "/api/groups2hosts/";
    constexpr TStringBuf TAG2HOSTS = "/api/tag2hosts/";

    struct TRequestContext {
        TRequestContext(TActorId recv, ui8 retries, TDuration backoff)
            : Receiver{recv}
            , Retries{retries}
            , RetryBackoff{backoff}
        {
        }

        TActorId Receiver;
        ui8 Retries;
        TDuration RetryBackoff;
    };

    struct TConductorClient: TActor<TConductorClient> {
        using TRequestMap = THashMap<THttpOutgoingRequestPtr, TRequestContext>;

    public:
        TConductorClient(TConductorClientConf conf)
            : TActor<TConductorClient>{&TThis::StateWork}
            , Address_{std::move(conf.Address)}
            , HttpProxy_{conf.HttpProxy}
            , Timeout_{conf.Timeout}
            , Retries_{conf.Retries}
            , RetryBackoff_{conf.RetryBackoff}
        {
        }

        STFUNC(StateWork) {
            Y_UNUSED(ctx);

            switch (ev->GetTypeRewrite()) {
                hFunc(TConductorEvents::TEvResolveTag, OnResolveTag);
                hFunc(TConductorEvents::TEvResolveGroup, OnResolveGroup);
                hFunc(TEvHttpProxy::TEvHttpIncomingResponse, OnResponse);
                cFunc(TEvents::TSystem::PoisonPill, PassAway);
            }
        }

        void OnResolveTag(const TConductorEvents::TEvResolveTag::TPtr& ev) {
            const TString path = TStringBuilder() << TAG2HOSTS << ev->Get()->Tag;
            MakeRequest(path, ev->Sender);
        }

        void OnResolveGroup(const TConductorEvents::TEvResolveGroup::TPtr& ev) {
            const TString path = TStringBuilder() << GROUP2HOSTS << ev->Get()->Group;
            MakeRequest(path, ev->Sender);
        }

        void MakeRequest(TStringBuf path, TActorId sender) {
            auto req = THttpOutgoingRequest::CreateRequestGet(Address_, path);
            auto [_, isInserted] = Requests_.try_emplace(req, sender, Retries_, RetryBackoff_);
            Y_VERIFY_DEBUG(isInserted);
            Send(HttpProxy_, new TEvHttpProxy::TEvHttpOutgoingRequest{req, Timeout_});
        }

        void OnResponse(const TEvHttpProxy::TEvHttpIncomingResponse::TPtr& ev) {
            auto it = Requests_.find(ev->Get()->Request.Get());
            if (it == Requests_.end()) {
                return;
            }

            if (ev->Get()->Error) {
                Reply(it, TConductorEvents::TEvResponse::TResult::FromError(ev->Get()->Error));
                return;
            }

            auto& reqCtx = it->second;

            auto&& resp = *ev->Get()->Response;
            auto status = FromString<int>(resp.Status);
            const auto isServerError = IsServerError(status);

            if (status == HTTP_OK) {
                auto hosts = ParseGroupHosts(resp.Body);
                Reply(it, TConductorEvents::TEvResponse::TResult::FromValue(std::move(hosts)));
            } else if (IsUserError(status)) {
                TString errorMessage = TStringBuilder() << resp.Status << ": " << resp.Body;
                Reply(it, TConductorEvents::TEvResponse::TResult::FromError(std::move(errorMessage)));
            } else if (isServerError && reqCtx.Retries > 0) {
                --reqCtx.Retries;
                TActivationContext::Schedule(reqCtx.RetryBackoff,
                    new IEventHandle(HttpProxy_, SelfId(), new TEvHttpProxy::TEvHttpOutgoingRequest{it->first, Timeout_}));
            } else {
                Reply(it, TConductorEvents::TEvResponse::TResult::FromError(ev->Get()->GetError()));
            }
        }

    private:
        void Reply(TRequestMap::iterator it, TConductorEvents::TEvResponse::TResult&& result) {
            Send(it->second.Receiver, new TConductorEvents::TEvResponse{
                std::move(result)
            });
            Requests_.erase(it);
        }

    private:
        const TString Address_;
        const TActorId HttpProxy_;
        const TDuration Timeout_;
        const ui8 Retries_;
        const TDuration RetryBackoff_;

        TRequestMap Requests_;
    };

} // namespace

    NActors::IActor* CreateConductorClient(TConductorClientConf conf) {
        return new TConductorClient{std::move(conf)};
    }

} // namespace NSolomon::NFetcher
