#include <solomon/services/fetcher/lib/clients/access_service.h>

#include <solomon/libs/cpp/actors/scheduler/scheduler.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace yandex::cloud::priv::servicecontrol::v1;

class TASGrpcClient: public IAccessServiceClient {
public:
    TAsyncAuthenticateResponse Authenticate(TString iamToken) noexcept override {
        if (iamToken == "errToken") {
            TAuthError err{NSolomon::EAccessServiceAuthErrorType::FailedAuth, ""};

            return NThreading::MakeFuture<TAsyncAuthenticateResponse::value_type>(
                TAsyncAuthenticateResponse::value_type::FromError(std::move(err)));
        }

        TIamAccount acc;

        return NThreading::MakeFuture<TAsyncAuthenticateResponse::value_type>(
            TAsyncAuthenticateResponse::value_type::FromValue(std::move(acc)));
    }

    TAsyncAuthorizeResponse Authorize(const yandex::cloud::priv::servicecontrol::v1::AuthorizeRequest& req) noexcept override {
        const auto& rp = req.resource_path(0);

        if (rp.id() == "errFolder") {
            NGrpc::TGrpcStatus errStatus{"err", 1, false};
            return NThreading::MakeFuture<TAsyncAuthorizeResponse::value_type>(errStatus);
        }

        yandex::cloud::priv::servicecontrol::v1::AuthorizeResponse resp;
        return NThreading::MakeFuture<TAsyncAuthorizeResponse::value_type>(std::move(resp));
    }

    void Stop(bool) override {
    }
};

class TASGrpcClientForRetries: public IAccessServiceClient {
public:
    explicit TASGrpcClientForRetries(size_t successOnCnt, EAccessServiceAuthErrorType errorType, TTestActorRuntime& actorSystem)
        : SuccessOnCnt{successOnCnt}
        , ErrorType{errorType}
        , ActorSystem{actorSystem}
    {}

private:
    TAsyncAuthenticateResponse Authenticate(TString) noexcept override {
        AuthenticationCallTimestamps.emplace_back(ActorSystem.GetCurrentTime());

        if (AuthenticationCallTimestamps.size() >= SuccessOnCnt) {
            TIamAccount acc;

            return NThreading::MakeFuture<TAsyncAuthenticateResponse::value_type>(
                TAsyncAuthenticateResponse::value_type::FromValue(std::move(acc)));
        }

        TAuthError err{ErrorType, ""};

        return NThreading::MakeFuture<TAsyncAuthenticateResponse::value_type>(
            TAsyncAuthenticateResponse::value_type::FromError(std::move(err)));
    }

    TAsyncAuthorizeResponse Authorize(const yandex::cloud::priv::servicecontrol::v1::AuthorizeRequest&) noexcept override {
        AuthorizationCallTimestamps.emplace_back(ActorSystem.GetCurrentTime());

        if (AuthorizationCallTimestamps.size() >= SuccessOnCnt) {
            yandex::cloud::priv::servicecontrol::v1::AuthorizeResponse resp;
            return NThreading::MakeFuture<TAsyncAuthorizeResponse::value_type>(std::move(resp));
        }

        NGrpc::TGrpcStatus errStatus{"err", 1, false};
        return NThreading::MakeFuture<TAsyncAuthorizeResponse::value_type>(errStatus);
    }

    void Stop(bool) override {
    }

public:
    size_t SuccessOnCnt;
    EAccessServiceAuthErrorType ErrorType;
    TTestActorRuntime& ActorSystem;

    TASGrpcClient Impl;
    TVector<TInstant> AuthenticationCallTimestamps;
    TVector<TInstant> AuthorizationCallTimestamps;
};

class TASGrpcClientWithPromises: public IAccessServiceClient {
private:
    TAsyncAuthenticateResponse Authenticate(TString) noexcept override {
        auto promise = NThreading::NewPromise<TAsyncAuthenticateResponse::value_type>();
        AuthenticationPromises.emplace_back(promise);

        return promise.GetFuture();
    }

    TAsyncAuthorizeResponse Authorize(const yandex::cloud::priv::servicecontrol::v1::AuthorizeRequest&) noexcept override {
        auto promise = NThreading::NewPromise<TAsyncAuthorizeResponse::value_type>();
        AuthorizationPromises.emplace_back(promise);

        return promise.GetFuture();
    }

    void Stop(bool) override {
    }

public:
    TVector<NThreading::TPromise<TAsyncAuthenticateResponse::value_type>> AuthenticationPromises;
    TVector<NThreading::TPromise<TAsyncAuthorizeResponse::value_type>> AuthorizationPromises;
};

class TAccessServiceTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime = TTestActorRuntime::CreateInited(1, false, true);
        EdgeId = ActorRuntime->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime.Reset();
    }

    TActorId RegisterASActor(IAccessServiceClientPtr accessServiceClient, size_t maxRetries = 0, size_t maxInflight = 190) {
        auto scheduler = ActorRuntime->Register(
            CreateScheduler(ActorRuntime->GetCurrentTime(), TDuration::Seconds(1)).release());

        TAccessServiceConfig config;
        config.SetRetriesCnt(maxRetries);
        config.SetMaxInflight(maxInflight);

        auto* d = config.MutableMinRetryDelay();
        d->SetValue(MinRetryDelay.Seconds());
        d->SetUnit(yandex::solomon::config::SECONDS);

        d = config.MutableMaxRetryDelay();
        d->SetValue(MaxRetryDelay.Seconds());
        d->SetUnit(yandex::solomon::config::SECONDS);

        d = config.MutableTimeout();
        d->SetValue(Timeout.Seconds());
        d->SetUnit(yandex::solomon::config::SECONDS);

        auto asId = ActorRuntime->Register(CreateAccessServiceActor(
            std::move(accessServiceClient),
            scheduler,
            std::move(config),
            std::make_shared<NMonitoring::TMetricRegistry>()).release());
        ActorRuntime->WaitForBootstrap();

        return asId;
    }

public:
    THolder<NSolomon::TTestActorRuntime> ActorRuntime;
    TActorId EdgeId;
    TDuration MinRetryDelay{TDuration::Seconds(3)};
    TDuration MaxRetryDelay{TDuration::Seconds(10)};
    TDuration Timeout{TDuration::Seconds(40)};
};

TEST_F(TAccessServiceTest, Robustness) { // SOLOMON-7360
    auto accessServiceClient = MakeIntrusive<TASGrpcClient>();
    auto asId = RegisterASActor(std::move(accessServiceClient));

    {
        auto auth1 = std::make_unique<TAccessServiceEvents::TAuthenticate>();
        auth1->IamToken = "errToken";
        auto auth2 = std::make_unique<TAccessServiceEvents::TAuthenticate>();
        auth2->IamToken = "token2";

        ActorRuntime->Send(asId, EdgeId, THolder(auth1.release()));
        ActorRuntime->Send(asId, EdgeId, THolder(auth2.release()));

        auto failEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthenticationFailure>(EdgeId);
        auto succEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthenticationSuccess>(EdgeId);
        ASSERT_TRUE(failEv);
        ASSERT_TRUE(succEv);
    }

    {
        auto auth1 = std::make_unique<TAccessServiceEvents::TAuthorize>();
        auth1->ServiceAccountId = "sa";
        auth1->FolderId = "errFolder";
        auto auth2 = std::make_unique<TAccessServiceEvents::TAuthorize>();
        auth2->ServiceAccountId = "sa";
        auth2->FolderId = "folder2";

        ActorRuntime->Send(asId, EdgeId, THolder(auth1.release()));
        ActorRuntime->Send(asId, EdgeId, THolder(auth2.release()));

        auto failEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthorizationFailure>(EdgeId);
        auto succEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthorizationSuccess>(EdgeId);
        ASSERT_TRUE(failEv);
        ASSERT_TRUE(succEv);
    }
}

TEST_F(TAccessServiceTest, Cookie) {
    auto accessServiceClient = MakeIntrusive<TASGrpcClient>();
    auto asId = RegisterASActor(std::move(accessServiceClient));

    struct TStruct {
        int X;
        int Y;
    };

    TStruct s1{ 123, 456 };
    TStruct s2{ 456, 123 };

    auto doNotFilter = [](auto&&) { return true; };

    {
        auto auth1 = std::make_unique<TAccessServiceEvents::TAuthenticate>();
        auth1->IamToken = "errToken";
        auto auth2 = std::make_unique<TAccessServiceEvents::TAuthenticate>();
        auth2->IamToken = "token2";

        ActorRuntime->Send(asId, EdgeId, THolder(auth1.release()), 0, reinterpret_cast<ui64>(&s1));
        ActorRuntime->Send(asId, EdgeId, THolder(auth2.release()), 0, reinterpret_cast<ui64>(&s2));

        auto handle1 = ActorRuntime->GrabEdgeEventIf<TAccessServiceEvents::TAuthenticationFailure>({EdgeId}, doNotFilter);
        auto handle2 = ActorRuntime->GrabEdgeEventIf<TAccessServiceEvents::TAuthenticationSuccess>({EdgeId}, doNotFilter);
        ASSERT_TRUE(handle1);
        ASSERT_TRUE(handle2);

        auto* receivedS1 = reinterpret_cast<TStruct*>(handle1->Cookie);
        auto* receivedS2 = reinterpret_cast<TStruct*>(handle2->Cookie);
        ASSERT_EQ(receivedS1, &s1);
        ASSERT_EQ(receivedS2, &s2);

        ASSERT_EQ(receivedS1->X, s1.X);
        ASSERT_EQ(receivedS1->Y, s1.Y);
        ASSERT_EQ(receivedS2->X, s2.X);
        ASSERT_EQ(receivedS2->Y, s2.Y);
    }

    {
        auto auth1 = std::make_unique<TAccessServiceEvents::TAuthorize>();
        auth1->ServiceAccountId = "sa";
        auth1->FolderId = "errFolder";
        auto auth2 = std::make_unique<TAccessServiceEvents::TAuthorize>();
        auth2->ServiceAccountId = "sa";
        auth2->FolderId = "folder2";

        ActorRuntime->Send(asId, EdgeId, THolder(auth1.release()), 0, reinterpret_cast<ui64>(&s1));
        ActorRuntime->Send(asId, EdgeId, THolder(auth2.release()), 0, reinterpret_cast<ui64>(&s2));

        auto handle1 = ActorRuntime->GrabEdgeEventIf<TAccessServiceEvents::TAuthorizationFailure>({EdgeId}, doNotFilter);
        auto handle2 = ActorRuntime->GrabEdgeEventIf<TAccessServiceEvents::TAuthorizationSuccess>({EdgeId}, doNotFilter);
        ASSERT_TRUE(handle1);
        ASSERT_TRUE(handle2);

        auto* receivedS1 = reinterpret_cast<TStruct*>(handle1->Cookie);
        auto* receivedS2 = reinterpret_cast<TStruct*>(handle2->Cookie);
        ASSERT_EQ(receivedS1, &s1);
        ASSERT_EQ(receivedS2, &s2);

        ASSERT_EQ(receivedS1->X, s1.X);
        ASSERT_EQ(receivedS1->Y, s1.Y);
        ASSERT_EQ(receivedS2->X, s2.X);
        ASSERT_EQ(receivedS2->Y, s2.Y);
    }
}

TEST_F(TAccessServiceTest, RetriesSuccess) {
    auto grpcClient = MakeIntrusive<TASGrpcClientForRetries>(
        4ul,
        EAccessServiceAuthErrorType::InternalRetriable,
        *ActorRuntime);
    size_t maxRetries = 3;
    auto asId = RegisterASActor(grpcClient, maxRetries);

    auto authentication = std::make_unique<TAccessServiceEvents::TAuthenticate>();
    authentication->IamToken = "token";
    ActorRuntime->Send(asId, EdgeId, THolder(authentication.release()));

    auto authorization = std::make_unique<TAccessServiceEvents::TAuthorize>();
    authorization->FolderId = "folder";
    authorization->ServiceAccountId = "sa";
    ActorRuntime->Send(asId, EdgeId, THolder(authorization.release()));

    {
        auto successfulAuthenticationEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthenticationSuccess>(
            EdgeId,
            Timeout + TDuration::Seconds(10));
        ASSERT_TRUE(successfulAuthenticationEv);
        auto& ts = grpcClient->AuthenticationCallTimestamps;
        ASSERT_EQ(ts.size(), 4ul);

        TVector<TDuration> retryDelays;
        for (size_t i = 0; i != ts.size() - 1; ++i) {
            retryDelays.emplace_back(ts[i + 1] - ts[i]);
        }

        ASSERT_GE(retryDelays[0].Seconds(), MinRetryDelay.Seconds() / 2);
        ASSERT_GE(retryDelays[1].Seconds(), MinRetryDelay.Seconds() / 2);
        ASSERT_GE(retryDelays[2].Seconds(), MinRetryDelay.Seconds() / 2);

        ASSERT_GE(retryDelays[1], retryDelays[0]);
        ASSERT_GE(retryDelays[2], retryDelays[1]);
        ASSERT_GT(retryDelays[2], retryDelays[0]);
    }

    {
        auto successfulAuthorizationEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthorizationSuccess>(
            EdgeId,
            Timeout + TDuration::Seconds(10));
        ASSERT_TRUE(successfulAuthorizationEv);
        auto& ts = grpcClient->AuthorizationCallTimestamps;
        ASSERT_EQ(ts.size(), 4ul);

        TVector<TDuration> retryDelays;
        for (size_t i = 0; i != ts.size() - 1; ++i) {
            retryDelays.emplace_back(ts[i + 1] - ts[i]);
        }

        ASSERT_GE(retryDelays[0].Seconds(), MinRetryDelay.Seconds() / 2);
        ASSERT_GE(retryDelays[1].Seconds(), MinRetryDelay.Seconds() / 2);
        ASSERT_GE(retryDelays[2].Seconds(), MinRetryDelay.Seconds() / 2);

        ASSERT_GE(retryDelays[1], retryDelays[0]);
        ASSERT_GE(retryDelays[2], retryDelays[1]);
        ASSERT_GT(retryDelays[2], retryDelays[0]);
    }
}

TEST_F(TAccessServiceTest, RetriesFail) {
    auto grpcClient = MakeIntrusive<TASGrpcClientForRetries>(
        4ul,
        EAccessServiceAuthErrorType::InternalRetriable,
        *ActorRuntime);
    size_t maxRetries = 2;
    auto asId = RegisterASActor(grpcClient, maxRetries);

    {
        auto authentication = std::make_unique<TAccessServiceEvents::TAuthenticate>();
        authentication->IamToken = "token";
        ActorRuntime->Send(asId, EdgeId, THolder(authentication.release()));

        auto authorization = std::make_unique<TAccessServiceEvents::TAuthorize>();
        authorization->FolderId = "folder";
        authorization->ServiceAccountId = "sa";
        ActorRuntime->Send(asId, EdgeId, THolder(authorization.release()));

        auto failedAuthenticationEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthenticationFailure>(
            EdgeId,
            Timeout + TDuration::Seconds(10));
        ASSERT_TRUE(failedAuthenticationEv);
        ASSERT_EQ(grpcClient->AuthenticationCallTimestamps.size(), 3ul);

        auto failedAuthorizationEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthorizationFailure>(
            EdgeId,
            Timeout + TDuration::Seconds(10));
        ASSERT_TRUE(failedAuthorizationEv);
        ASSERT_EQ(grpcClient->AuthorizationCallTimestamps.size(), 3ul);
    }
}

TEST_F(TAccessServiceTest, Inflight) {
    auto grpcClient = MakeIntrusive<TASGrpcClientWithPromises>();
    size_t maxRetries = 0;
    size_t maxInflight = 10;
    auto asId = RegisterASActor(grpcClient, maxRetries, maxInflight);

    size_t reqNum = 25;
    size_t atCnt = 0;

    for (size_t i = 0; i != maxInflight / 2; ++i) {
        auto ev = std::make_unique<TAccessServiceEvents::TAuthenticate>();
        ev->IamToken = "token" + ToString(++atCnt);
        ActorRuntime->Send(asId, EdgeId, THolder(ev.release()));
        ActorRuntime->AdvanceCurrentTime(TDuration::MilliSeconds(1));
    }

    for (size_t i = 0; i != maxInflight / 2; ++i) {
        auto ev = std::make_unique<TAccessServiceEvents::TAuthorize>();
        ActorRuntime->Send(asId, EdgeId, THolder(ev.release()));
        ActorRuntime->AdvanceCurrentTime(TDuration::MilliSeconds(1));
    }

    for (size_t i = 0; i != maxInflight; ++i) {
        auto ev = std::make_unique<TAccessServiceEvents::TAuthenticate>();
        ev->IamToken = "token" + ToString(++atCnt);
        ActorRuntime->Send(asId, EdgeId, THolder(ev.release()));
        ActorRuntime->AdvanceCurrentTime(TDuration::MilliSeconds(1));
    }

    for (size_t i = 0; i != maxInflight / 2; ++i) {
        auto ev = std::make_unique<TAccessServiceEvents::TAuthorize>();
        ActorRuntime->Send(asId, EdgeId, THolder(ev.release()));
        ActorRuntime->AdvanceCurrentTime(TDuration::MilliSeconds(1));
    }

    size_t eventsCnt{0};

    auto& atPromises = grpcClient->AuthenticationPromises;
    auto& azPromises = grpcClient->AuthorizationPromises;
    ASSERT_EQ(atPromises.size(), maxInflight / 2);
    ASSERT_EQ(azPromises.size(), maxInflight / 2);

    auto resolvePromises = [&]<typename TEv, typename TResp, typename T>(
        TVector<NThreading::TPromise<T>>& promises,
        const TResp& resp)
    {
        TVector<NThreading::TPromise<T>> promisesCopy;

        for (auto& prom: promises) {
            promisesCopy.emplace_back(std::move(prom));
        }
        promises.clear();

        for (auto& prom: promisesCopy) {
            prom.SetValue(T::FromValue(resp));
        }

        while (auto ev = ActorRuntime->GrabEdgeEvent<TEv>(EdgeId, TDuration::Seconds(1))) {
            ++eventsCnt;
        }
    };

    auto resolveAT = [&]() {
        resolvePromises.operator()<TAccessServiceEvents::TAuthenticationSuccess>(
            grpcClient->AuthenticationPromises,
            TIamAccount{});
    };
    auto resolveAZ = [&]() {
        resolvePromises.operator()<TAccessServiceEvents::TAuthorizationSuccess>(
            grpcClient->AuthorizationPromises,
            yandex::cloud::priv::servicecontrol::v1::AuthorizeResponse{});
    };

    resolveAT();
    resolveAZ();
    ASSERT_EQ(atPromises.size(), maxInflight);
    ASSERT_EQ(azPromises.size(), 0ul);
    ASSERT_EQ(eventsCnt, maxInflight);

    resolveAT();
    ASSERT_EQ(azPromises.size(), maxInflight / 2);
    ASSERT_EQ(atPromises.size(), 0ul);
    ASSERT_EQ(eventsCnt, maxInflight * 2);

    resolveAZ();
    ASSERT_EQ(atPromises.size(), 0ul);
    ASSERT_EQ(azPromises.size(), 0ul);
    ASSERT_EQ(eventsCnt, reqNum);
}

TEST_F(TAccessServiceTest, Timeout) {
    auto accessServiceClient = MakeIntrusive<TASGrpcClientWithPromises>();
    size_t maxRetries = 1000;
    size_t maxInflight = 2;
    auto asId = RegisterASActor(std::move(accessServiceClient), maxRetries, maxInflight);

    struct TStruct {
        int X;
        int Y;
    };

    TStruct s1{ 123, 456 };

    {
        ActorRuntime->Send(
            asId,
            EdgeId,
            MakeHolder<TAccessServiceEvents::TAuthenticate>(),
            0,
            reinterpret_cast<ui64>(&s1));

        // the promise is never resolved, hence a timeout event is sent
        ActorRuntime->AdvanceCurrentTime(TDuration::Seconds(50));
        ActorRuntime->DispatchEvents({}, TDuration::Seconds(5));

        auto failEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthenticationFailure>(EdgeId);
        auto* gotS1 = reinterpret_cast<TStruct*>(failEv->Cookie);

        ASSERT_TRUE(failEv);
        EXPECT_EQ(&s1, gotS1);
        EXPECT_EQ(s1.X, gotS1->X);
        EXPECT_EQ(s1.Y, gotS1->Y);
    }

    {
        auto auth = std::make_unique<TAccessServiceEvents::TAuthorize>();
        ActorRuntime->Send(
            asId,
            EdgeId,
            MakeHolder<TAccessServiceEvents::TAuthorize>(),
            0,
            reinterpret_cast<ui64>(&s1));

        // the promise is never resolved, hence a timeout event is sent
        ActorRuntime->AdvanceCurrentTime(TDuration::Seconds(50));
        ActorRuntime->DispatchEvents({}, TDuration::Seconds(5));

        auto failEv = ActorRuntime->GrabEdgeEvent<TAccessServiceEvents::TAuthorizationFailure>(EdgeId);
        auto* gotS1 = reinterpret_cast<TStruct*>(failEv->Cookie);

        ASSERT_TRUE(failEv);
        EXPECT_EQ(&s1, gotS1);
        EXPECT_EQ(s1.X, gotS1->X);
        EXPECT_EQ(s1.Y, gotS1->Y);
    }
}
