#include <solomon/services/fetcher/lib/clients/conductor.h>
#include <solomon/services/fetcher/testlib/http_server.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/http/http_proxy.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/split.h>

using namespace NActors;
using namespace testing;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NMonitoring;

static std::shared_ptr<TMetricRegistry> Registry = std::make_shared<TMetricRegistry>();

const TString RESPONSE_CONTENT{
R"(solomon-pre-front-sas-00.search.yandex.net
solomon-pre-front-sas-01.search.yandex.net
solomon-pre-front-sas-02.search.yandex.net
solomon-pre-alert-00.search.yandex.net
solomon-pre-alert-01.search.yandex.net
solomon-pre-alert-02.search.yandex.net)"
};

const auto EXPECTED_HOSTS = [] {
    return StringSplitter(RESPONSE_CONTENT)
        .Split('\n')
        .SkipEmpty()
        .ToList<TString>();
}();

THttpResponse OkResponse() {
    THttpResponse resp;
    resp.SetContent(RESPONSE_CONTENT);
    return resp;
}

class TConductorClientTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorSystem_ = NSolomon::TTestActorRuntime::CreateInited(1, false, false);
        Server_.Reset(new TTestServer);

        auto httpProxy = ActorSystem_->Register(NHttp::CreateHttpProxy(Registry));

        TConductorClientConf conf{
            .HttpProxy = httpProxy,
            .Retries = 1,
            .RetryBackoff = TDuration::MilliSeconds(50),
            .Address = "localhost:" + ToString(Server_->Port()),
        };

        ClientActor_ = ActorSystem_->Register(CreateConductorClient(conf));
        Listener_ = ActorSystem_->AllocateEdgeActor();
    }

protected:
    template <typename TEv>
    TConductorEvents::TEvResponse::TResult MakeRequest(TString name = "foo") {
        auto ev = MakeHolder<TEv>(name);
        ActorSystem_->Send(new IEventHandle{
            ClientActor_, Listener_, ev.Release()
        });

        auto responseEv = ActorSystem_->GrabEdgeEvent<TConductorEvents::TEvResponse>();
        return std::move(responseEv.Get()->Hosts);
    }

    TConductorEvents::TEvResponse::TResult MakeRequestTag() {
        return MakeRequest<TConductorEvents::TEvResolveTag>();
    }

    TConductorEvents::TEvResponse::TResult MakeRequestGroup(TString name = "foo") {
        return MakeRequest<TConductorEvents::TEvResolveGroup>(std::move(name));
    }

    TTestServer* Server() {
        return Server_.Get();
    }

private:
    THolder<NSolomon::TTestActorRuntime> ActorSystem_;
    THolder<TTestServer> Server_;
    TActorId ClientActor_;
    TActorId Listener_;
};

TEST_F(TConductorClientTest, GroupResponseOk) {
    Server()->AddHandler("/api/groups2hosts/foo", OkResponse);

    auto result = MakeRequestGroup();
    ASSERT_TRUE(result.Success());
    ASSERT_THAT(result.Value(), ElementsAreArray(EXPECTED_HOSTS));
}

TEST_F(TConductorClientTest, TagResponseOk) {
    Server()->AddHandler("/api/tag2hosts/foo", OkResponse);

    auto result = MakeRequestTag();
    ASSERT_TRUE(result.Success());
    ASSERT_THAT(result.Value(), ElementsAreArray(EXPECTED_HOSTS));
}

TEST_F(TConductorClientTest, NonExisting) {
    auto result = MakeRequestGroup("bar");
    ASSERT_TRUE(result.Fail());
}
