#include "codec.h"

#include <library/cpp/blockcodecs/core/codecs.h>
#include <library/cpp/streams/brotli/brotli.h>

#include <util/generic/hash.h>
#include <util/stream/zlib.h>
#include <util/stream/mem.h>

namespace NSolomon {
namespace {

    class TCodecWrapper final: public ICodec {
    public:
        TCodecWrapper(const NBlockCodecs::ICodec* codec)
            : Impl_{codec}
        {
        }

        ICodec::TDecodeResult Decompress(TStringBuf in) noexcept override {
            try {
                return ICodec::TDecodeResult::FromValue(Impl_->Decode(in));
            } catch (...) {
                return ICodec::TDecodeResult::FromError(CurrentExceptionMessage());
            }
        }

        ui64 DecompressedSize(TStringBuf in) const noexcept override {
            try {
                return Impl_->DecompressedLength(in);
            } catch (...) {
                return 0;
            }
        }

    private:
        const NBlockCodecs::ICodec* Impl_;
    };

    template <typename TDecoderStream>
    class TStreamWrapper final: public ICodec {
    public:
        ICodec::TDecodeResult Decompress(TStringBuf in) noexcept override {
            TMemoryInput inStream{in};
            TDecoderStream decoder{&inStream};

            try {
                return ICodec::TDecodeResult::FromValue(decoder.ReadAll());
            } catch (...) {
                return ICodec::TDecodeResult::FromError(CurrentExceptionMessage());
            }
        }

        ui64 DecompressedSize(TStringBuf) const noexcept override {
            return 0;
        }
    };

    using TZlibCodec = TStreamWrapper<TZLibDecompress>;
    using TBrotliCodec = TStreamWrapper<TBrotliDecompress>;

    class TCodecFactory final: public ICodecFactory {
    public:
        TCodecFactory()
            : Zlib_{new TZlibCodec}
            , Brotli_{new TBrotliCodec}
        {
            auto codecs = NBlockCodecs::ListAllCodecs();
            Wrappers_.reserve(codecs.size());

            for (auto name: codecs) {
                if (name.StartsWith("zstd06") || name.StartsWith("zstd08")) {
                    continue;
                }

                auto& wrapper = Wrappers_.emplace_back(NBlockCodecs::Codec(name));
                auto [_, ok] = Codecs_.emplace(name, &wrapper);
                Y_VERIFY_DEBUG(ok);
            }

            for (TStringBuf name: {"gzip", "deflate"}) {
                auto [_, ok] = Codecs_.emplace(name, Zlib_.Get());
                Y_VERIFY_DEBUG(ok);
            }

            auto [_, ok] = Codecs_.emplace("br", Brotli_.Get());
            Y_VERIFY_DEBUG(ok);
        }

    ICodec* Codec(TStringBuf format) const override {
        if (auto it = Codecs_.find(format); it != Codecs_.end()) {
            return it->second;
        }

        return nullptr;
    }

    private:
        TVector<TCodecWrapper> Wrappers_;
        THolder<TZlibCodec> Zlib_;
        THolder<TBrotliCodec> Brotli_;
        THashMap<TString, ICodec*> Codecs_;
    };

} // namespace

    ICodecFactory* CodecFactory() {
        return Singleton<TCodecFactory>();
    }

} // namespace NSolomon
