#include "config_holder.h"
#include "diff.h"

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/generic/algorithm.h>
#include <util/digest/numeric.h>

using namespace NSolomon::NDb::NModel;

namespace NSolomon::NFetcher {
namespace {
    struct TAgentsInfo {
        TDuration PullInterval;
        TVector<TAgentConfig> Agents;
    };

    TVector<TFetcherShard> MakeAgentShards(
            const THashSet<TAgentConfig>& agents,
            const THashMap<TProviderId, TProviderConfigPtr>& providers)
    {
        absl::flat_hash_map<TStringBuf, TAgentsInfo> providerIdToAgentsInfo;
        providerIdToAgentsInfo.reserve(providers.size());

        for (auto&& agent: agents) {
            auto providerIt = providers.find(agent.Provider);

            if (providerIt == providers.end()) {
                continue;
            }

            auto& info = providerIdToAgentsInfo[agent.Provider];
            if (info.Agents.empty()) {
                info.PullInterval = providerIt->second->ShardSettings.Interval;
            }
            info.Agents.emplace_back(agent);
        }

        TVector<TFetcherShard> result;
        result.reserve(providerIdToAgentsInfo.size());
        for (auto&& [providerId, info]: providerIdToAgentsInfo) {
            result.push_back(
                CreateAgentShard(providerId, info.PullInterval, std::move(info.Agents))
            );
        }

        return result;
    }

    bool HasClusterChanged(const TClusterConfig& lhs, const TClusterConfig& rhs) {
        if (AnyHostGroupChanged(lhs, rhs)) {
            return true;
        }

        return lhs.ShardSettings.PullOrPush != rhs.ShardSettings.PullOrPush;
    }

    bool HasProviderChanged(const TProviderConfig& lhs, const TProviderConfig& rhs) {
        return !(lhs.IamServiceAccountIds == rhs.IamServiceAccountIds
                && lhs.TvmServiceIds == rhs.TvmServiceIds
                && lhs.TvmDestId == rhs.TvmDestId
                && lhs.ShardSettings.Interval == rhs.ShardSettings.Interval
        );
    }

    bool HasServiceChanged(const TServiceConfig& lhs, const TServiceConfig& rhs) {
        return !(lhs.Interval == rhs.Interval
            && lhs.GridSec == rhs.GridSec
            && lhs.ShardSettings == rhs.ShardSettings);
    }

    bool HasShardChanged(const TShardConfig& lhs, const TShardConfig& rhs) {
        return !(lhs.MaxResponseSizeBytes == rhs.MaxResponseSizeBytes
            && lhs.MaxMetricsPerUrl == rhs.MaxMetricsPerUrl
            && lhs.ServiceId == rhs.ServiceId
            && lhs.ServiceName == rhs.ServiceName
            && lhs.ClusterId == rhs.ClusterId
            && lhs.ClusterName == rhs.ClusterName
            && lhs.State == rhs.State);
    }

    TShardId MakeIdForConf(const TShardConfig& shardConf) {
        return TShardId{shardConf.Id, shardConf.NumId};
    }

    TServiceId MakeIdForConf(const TServiceConfig& serviceConf) {
        return TServiceId{serviceConf.Id, serviceConf.ProjectId};
    }

    TClusterId MakeIdForConf(const TClusterConfig& clusterConf) {
        return TClusterId{clusterConf.Id, clusterConf.ProjectId};
    }

    TProviderId MakeIdForConf(const TProviderConfig& providerConf) {
        return providerConf.Id;
    }

    template <typename TModel>
    using TPredicate = bool (*) (const TModel&, const TModel&);

    template <typename TModel, typename TId>
    auto DoCalculateDiff(
        const TVector<TModel>& newConfs,
        const THashMap<TId, TAtomicSharedPtr<const TModel>>& oldConfs,
        TPredicate<TModel> predicate)
    {
        THashMap<TId, const TModel*> newConfSet;

        for (auto& conf: newConfs) {
            newConfSet[MakeIdForConf(conf)] = &conf;
        }

        return TDiffBuilder<TId, TModel>{}
            .SetChangePredicate(predicate)
            .Build(oldConfs, newConfSet);
    }

    struct TDiffInput {
        TConfigs& NewConfigs;

        THashMap<TShardId, TShardConfigPtr>& CurrentShards;
        THashMap<TClusterId, TClusterConfigPtr>& CurrentClusters;
        THashMap<TServiceId, TServiceConfigPtr>& CurrentServices;
        THashMap<TProviderId, TProviderConfigPtr>& CurrentProviders;
    };

    TConfigDiff BuildFullDiff(TDiffInput input) {
        TConfigDiff result;
        result.ShardDiff = DoCalculateDiff(input.NewConfigs.Shards, input.CurrentShards, HasShardChanged);
        result.ServiceDiff = DoCalculateDiff(input.NewConfigs.Services, input.CurrentServices, HasServiceChanged);
        result.ClusterDiff = DoCalculateDiff(input.NewConfigs.Clusters, input.CurrentClusters, HasClusterChanged);
        result.ProviderDiff = DoCalculateDiff(input.NewConfigs.Providers, input.CurrentProviders, HasProviderChanged);

        return result;
    }

} // namespace

    TString TConfigDiff::ToString() const {
        TStringBuilder sb;
        if (!ServiceDiff.IsEmpty()) {
            sb << "service diff: " << ServiceDiff.ToString() << '\n';
        }

        if (!ClusterDiff.IsEmpty()) {
            sb << "cluster diff: " << ClusterDiff.ToString() << '\n';
        }

        if (!ShardDiff.IsEmpty()) {
            sb << "shard diff: " << ShardDiff.ToString() << '\n';
        }

        if (!ProviderDiff.IsEmpty()) {
            sb << "provider diff: " << ProviderDiff.ToString() << '\n';
        }

        // TODO(ivanzhukov): add a diff for agents

        return sb;
    }

    bool TConfigDiff::IsEmpty() const {
        return ShardDiff.IsEmpty()
            && ClusterDiff.IsEmpty()
            && ServiceDiff.IsEmpty()
            && AgentDiff.IsEmpty
            && ProviderDiff.IsEmpty();
    }

    TConfigHolder::TConfigHolder(IShardLocator& shardLocator, TShardFilter filter, bool enableYasm)
        : ShardLocator_{shardLocator}
        , Filter_{std::move(filter)}
    {
        if (enableYasm) {
            YasmShard_ = CreateYasmAgentShard();
        }
    }

    std::optional<TFetcherShard> TConfigHolder::GetShardImpl(const TShardConfigPtr& shard) const {
        auto* servicePtr = Services_.FindPtr(TServiceId{shard->ServiceId, shard->ProjectId});
        if (!servicePtr) {
            return {};
        }

        auto* clusterPtr = Clusters_.FindPtr(TClusterId{shard->ClusterId, shard->ProjectId});
        if (!clusterPtr) {
            return {};
        }

        auto loc = ShardLocator_.Location(shard->NumId);

        return CreateSimpleShard(
            shard,
            *clusterPtr,
            *servicePtr,
            (loc ? *loc : TClusterNode{})
        );
    }

    std::optional<TFetcherShard> TConfigHolder::GetShard(TShardId::TNumId numId) const {
        if (auto* shardPtr = Shards_.FindPtr(TShardId{numId})) {
            return GetShardImpl(*shardPtr);
        }

        return {};
    }

    TVector<TFetcherShard> TConfigHolder::AllShards() const {
        TVector<TFetcherShard> result;

        for (auto&& [_, shard]: Shards_) {
            auto fetcherShard = GetShardImpl(shard);
            if (!fetcherShard) {
                continue;
            }

            result.push_back(std::move(*fetcherShard));
        }

        auto agentShards = MakeAgentShards(Agents_, Providers_);
        Copy(
            std::make_move_iterator(agentShards.begin()),
            std::make_move_iterator(agentShards.end()),
            std::back_inserter(result)
        );

        if (YasmShard_) {
            result.push_back(*YasmShard_);
        }

        return result;
    }

    TVector<TProviderConfigPtr> TConfigHolder::AllProviders() const {
        TVector<TProviderConfigPtr> providers(::Reserve(Providers_.size()));
        for (const auto& [_, conf]: Providers_) {
            providers.emplace_back(conf);
        }

        return providers;
    }

    TConfigDiff TConfigHolder::UpdateConfig(TConfigs& configs) {
        EraseIf(configs.Shards, [&] (auto&& shard) {
            return !Filter_(shard);
        });

        TConfigDiff diff = BuildFullDiff(TDiffInput{
            .NewConfigs = configs,

            .CurrentShards = Shards_,
            .CurrentClusters = Clusters_,
            .CurrentServices = Services_,
            .CurrentProviders = Providers_,
            // Agents are handled below
        });

        auto update = [] (auto& configs, auto& dest) {
            std::decay_t<decltype(dest)> result;
            for (auto& conf: configs) {
                auto id = MakeIdForConf(conf);
                result[id] = MakeAtomicShared<std::decay_t<decltype(conf)>>(conf);
            }

            dest = std::move(result);
        };

        if (!diff.ShardDiff.IsEmpty()) {
            update(configs.Shards, Shards_);
        }

        if (!diff.ClusterDiff.IsEmpty()) {
            update(configs.Clusters, Clusters_);
        }

        if (!diff.ServiceDiff.IsEmpty()) {
            update(configs.Services, Services_);
        }

        if (!diff.ProviderDiff.IsEmpty()) {
            update(configs.Providers, Providers_);
        }

        THashSet<TAgentConfig> newAgentConfs;
        CopyIf(
            std::make_move_iterator(configs.Agents.begin()),
            std::make_move_iterator(configs.Agents.end()),
            std::inserter(newAgentConfs, newAgentConfs.begin()),
            [this](const TAgentConfig& agentConfig) {
                return !agentConfig.Provider.empty() &&
                        Providers_.find(agentConfig.Provider) != Providers_.end();
            }
        );

        const bool isAgentsChanged = Agents_ != newAgentConfs;
        diff.AgentDiff.IsEmpty = !isAgentsChanged;

        if (isAgentsChanged) {
            Agents_ = std::move(newAgentConfs);
        }

        return diff;
    }
} // namespace NSolomon::NFetcher
