#pragma once

#include "config_holder.h"

#include <solomon/services/fetcher/lib/events.h>
#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/shard_assignments/assignments.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/clients/coremon/coremon_client.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/clients/ingestor/ingestor_client.h>

#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/monlib/metrics/fwd.h>

#include <util/datetime/base.h>

namespace NSolomon::NFetcher {
    class IProcessingClusterClient;
    using IProcessingClusterClientPtr = TIntrusivePtr<IProcessingClusterClient>;

    struct TUpdaterEvents: private TEventSlot<EEventSpace::Fetcher, FS_UPDATER> {
        enum {
            EvConfigChanged = SpaceBegin,
            EvClustersChanged,
            EvProvidersChanged,

            EvReloadShardRequest,
            EvReloadShardResponse,

            EvLoadShardMap,
            EvLoadShardMapResponse,

            End,
        };
        static_assert(End < SpaceEnd, "too many event types");
    };

    struct TEvReloadShardRequest: NActors::TEventLocal<TEvReloadShardRequest, TUpdaterEvents::EvReloadShardRequest> {
        TEvReloadShardRequest(TString projectId, TShardId shardId)
            : ProjectId{std::move(projectId)}
            , ShardId{std::move(shardId)}
        {
        }

        TString ProjectId;
        TShardId ShardId;
    };

    struct TEvReloadShardResponse: NActors::TEventLocal<TEvReloadShardResponse, TUpdaterEvents::EvReloadShardResponse> { };

    struct TInfoOfAShardToRemove {
        TShardId Id;
        EFetcherShardType Type;

        TInfoOfAShardToRemove(TShardId id, EFetcherShardType type)
            : Id{std::move(id)}
            , Type{type}
        {
        }

        bool operator==(const TInfoOfAShardToRemove& other) const = default;
    };

    struct TEvConfigChanged: public NActors::TEventLocal<TEvConfigChanged, TUpdaterEvents::EvConfigChanged> {
        TEvConfigChanged(
                TVector<TFetcherShard> added,
                TVector<TFetcherShard> changed,
                TVector<TInfoOfAShardToRemove> removed
        )
            : Added{std::move(added)}
            , Changed{std::move(changed)}
            , Removed{std::move(removed)}
        {
        }

        TVector<TFetcherShard> Added;
        TVector<TFetcherShard> Changed;
        TVector<TInfoOfAShardToRemove> Removed;
    };

    struct TEvProvidersChanged: public NActors::TEventLocal<TEvProvidersChanged, TUpdaterEvents::EvProvidersChanged> {
        TEvProvidersChanged(
                TVector<TProviderConfigPtr> added,
                TVector<TProviderConfigPtr> changed,
                TVector<TProviderId> removed)
            : Added{std::move(added)}
            , Changed{std::move(changed)}
            , Removed{std::move(removed)}
        {}

        TVector<TProviderConfigPtr> Added;
        TVector<TProviderConfigPtr> Changed;
        TVector<TProviderId> Removed;
    };

    struct TEvClustersChanged: public NActors::TEventLocal<TEvClustersChanged, TUpdaterEvents::EvClustersChanged> {
        explicit TEvClustersChanged(TVector<IFetcherClusterPtr> clusters)
            : Clusters{std::move(clusters)}
        {
        }

        TVector<IFetcherClusterPtr> Clusters;
    };

    enum class EUpdaterMode: ui8 {
        OnlyLocal = 0,
        All,
        // take a comma-separated list of shards from the SOLOMON_FETCHER_SHARDS environment variable
        Env,
        OnlyYasm,
        // TODO(ivanzhukov@): do not update agents as well
        None,
    };

    struct TConfigUpdaterConfig {
        NDb::IServiceConfigDaoPtr ServiceDao;
        NDb::IShardConfigDaoPtr ShardDao;
        NDb::IClusterConfigDaoPtr ClusterDao;
        NDb::IAgentConfigDaoPtr AgentDao;
        NDb::IProviderConfigDaoPtr ProviderDao;
        NActors::TActorId ShardMap;
        TDuration Interval = TDuration::Minutes(1);
        NMonitoring::TMetricRegistry& MetricRegistry;
        EUpdaterMode Mode = EUpdaterMode::All;
        bool EnableYasmPulling = false;
    };

    NActors::TActorId MakeConfigUpdaterId();
    NActors::IActor* CreateConfigUpdaterActor(const TConfigUpdaterConfig& config);

    /// Shard map events
    struct TEvLoadShardMap: NActors::TEventLocal<TEvLoadShardMap, TUpdaterEvents::EvLoadShardMap> {
        struct TCachePolicy {
            TCachePolicy(TDuration maxAge)
                : MaxAge{maxAge}
            {
            }

            TDuration MaxAge;
        };

        TEvLoadShardMap() = default;
        explicit TEvLoadShardMap(TDuration cacheMaxAge)
            : CachePolicy{cacheMaxAge}
        {
        }

        TCachePolicy CachePolicy = TDuration::Zero();
    };

    using TLoadShardMapResult = TErrorOr<TVector<TShardAssignment>, TGenericError>;
    struct TEvLoadShardMapResponse: NActors::TEventLocal<TEvLoadShardMapResponse, TUpdaterEvents::EvLoadShardMapResponse> {
        TEvLoadShardMapResponse(TLoadShardMapResult&& result)
            : Result{std::move(result)}
        {
        }

        TLoadShardMapResult Result;
    };

    NActors::TActorId MakeShardMapBuilderId();
    NActors::IActor* CreateShardMapBuilderRequestingLeader(
            IProcessingClusterClientPtr clients,
            IClusterMapPtr cluster);
    NActors::IActor* CreateShardMapBuilderRequestingCluster(
            NSolomon::NIngestor::IIngestorClusterClientPtr clients,
            IClusterMapPtr cluster);
} // namespace NSolomon::NFetcher
