#pragma once

#include <util/generic/hash.h>
#include <util/generic/map.h>
#include <util/generic/ptr.h>
#include <util/string/builder.h>

namespace NSolomon::NFetcher {
    template <typename TKey, typename TValue>
    struct TDiffBuilder {
        using TChangePredicate = std::function<bool(const TValue&, const TValue&)>;

        TDiffBuilder()
            : ChangePredicate_{TDiffBuilder<TKey, TValue>::Compare}
        {
        }

        TDiffBuilder& SetChangePredicate(TChangePredicate pred) {
            ChangePredicate_ = pred;
            return *this;
        }

        struct TDiff {
            using TItem = std::pair<TKey, TValue>;

            TVector<TItem> Added;
            TVector<TItem> Removed;
            TVector<TItem> Changed;

            bool IsEmpty() const {
                return Added.empty() && Removed.empty() && Changed.empty();
            }

            TString ToString() const {
                TStringBuilder str;

                auto print = [&str] (char sign, auto&& items) {
                    if (items.size() > 0) {
                        str << sign << " [ ";
                        for (auto& [key, _]: items) {
                            str << key << " ";
                        }

                        str << "] ";
                    }
                };

                print('+', Added);
                print('-', Removed);
                print('~', Changed);

                return str;
            }
        };

        // making 'current' non-const here allows somewhat more efficient implementation
        template <typename T>
        TDiff Build(const THashMap<TKey, TAtomicSharedPtr<const TValue>>& old, T& current) {
            TDiff result;
            for (auto&& [key, value]: old) {
                auto it = current.find(key);
                if (it == current.end()) {
                    result.Removed.emplace_back(key, *value);
                    continue;
                } else if (ChangePredicate_(*it->second, *value)) {
                    result.Changed.emplace_back(it->first, *it->second);
                }

                current.erase(it);
            }

            result.Added.reserve(current.size());
            for (auto&& [key, value]: current) {
                result.Added.emplace_back(key, *value);
            }

            return result;
        }

    private:
        static bool Compare(const TValue& old, const TValue& current) {
            return current.Version > old.Version;
        }

    private:
        TChangePredicate ChangePredicate_;
    };

    template <typename T, typename TContainer, typename TComp = std::equal_to<T>>
    auto MakeDiff(TContainer& old, const TVector<T>& current, TComp compare = {}) {
        struct {
            TVector<T> Added;
            TVector<T> Changed;
            TVector<T> Removed;
        } result;

        for (auto&& item: current) {
            auto it = old.find(item);
            const auto contains = it != old.end();

            if (!contains) {
                result.Added.push_back(item);
                continue;
            }

            if (!compare(*it, item)) {
                result.Changed.push_back(item);
            }

            old.erase(it);
        }

        result.Removed.reserve(old.size());
        Copy(old.begin(), old.end(), std::back_inserter(result.Removed));

        return result;
    }
} // namespace NSolomon::NFetcher
