#include "config_updater.h"

#include <solomon/services/fetcher/lib/data_sink/processing_client.h>
#include <solomon/services/fetcher/lib/ingestor/assignments_requester.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

#include <util/generic/hash.h>
#include <util/generic/algorithm.h>

#include <functional>
#include <memory>

using namespace NActors;
using namespace NSolomon::NCoremon;
using namespace NSolomon::NIngestor;
using namespace NSolomon;
using namespace std::string_view_literals;

namespace NSolomon::NFetcher {
namespace {
    class TCache {
    public:
        void Write(TVector<TShardAssignment> assn, TInstant ts) {
            Value_ = std::move(assn);
            Timestamp_ = ts;
        }

        std::optional<TVector<TShardAssignment>> Get(TDuration maxAge = TDuration::Max()) {
            if (Timestamp_ >= (TInstant::Now() - maxAge)) {
                return Value_;
            }

            return {};
        }

    private:
        TVector<TShardAssignment> Value_;
        TInstant Timestamp_;
    };

    struct TRequestAssignmentsFromServiceLeader:
            public TActorBootstrapped<TRequestAssignmentsFromServiceLeader>,
            private TPrivateEvents
    {
        enum {
            EvResponse = SpaceBegin,
            EvLeader,
            End,
        };
        static_assert(End < SpaceEnd, "too many event types");

        struct TEvResponse: TEventLocal<TEvResponse, EvResponse> {
            TErrorOr<NSolomon::NFetcher::TShardAssignments, TApiCallError> Assignments;

            explicit TEvResponse(TErrorOr<NSolomon::NFetcher::TShardAssignments, TApiCallError> assn)
                : Assignments{std::move(assn)}
            {}
        };

        struct TEvLeader: TEventLocal<TEvLeader, EvLeader> {
            TEvLeader(std::optional<TClusterNode> leader)
                : Leader{std::move(leader)}
            {
            }

            std::optional<TClusterNode> Leader;
        };

        TRequestAssignmentsFromServiceLeader(IProcessingClient* client, const IClusterMap& cluster, TActorId receiver)
            : Client_{std::move(client)}
            , Receiver_{receiver}
            , Cluster_{cluster}
        {}

        void Bootstrap(const TActorContext& ctx) {
            Become(&TThis::StateWait);
            const auto me = SelfId();
            auto* as = ctx.ExecutorThread.ActorSystem;

            Client_->GetShardAssignments().Subscribe([=] (auto f) {
                try {
                    auto v = f.ExtractValue();
                    as->Send(me, new TEvResponse{
                        std::move(v),
                    });
                } catch (...) {
                    as->Send(me, new TEvResponse{
                        TErrorOr<NSolomon::NFetcher::TShardAssignments, TApiCallError>::FromError(CurrentExceptionMessage())
                    });
                }
            });
        }

        STATEFN(StateWait) {
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvResponse, OnResponse);
            }
        }

        void OnResponse(const TEvResponse::TPtr& ev) {
            if (!ev->Get()->Assignments.Success()) {
                auto& err = ev->Get()->Assignments.Error();
                MON_ERROR(ShardUpdater, "Unable to get shards: " << err.Message());
                Send(Receiver_, new TEvLeader{std::nullopt});
                RespondAndDie(TLoadShardMapResult::FromError(err));
                return;
            }

            auto assignments = ev->Get()->Assignments.Extract();

            TVector<TShardAssignment> result;
            result.reserve(assignments.Assignments.size());

            Transform(
                assignments.Assignments.begin(),
                assignments.Assignments.end(),
                std::back_inserter(result),
                [&] (auto&& hostShards) {
                    // remap node ids, since we might be not is sync with coremon in this regard
                    const auto loc = Cluster_.NodeByFqdn(hostShards.first.first);
                    return TShardAssignment{
                        loc ? *loc : TClusterNode{hostShards.first.first, TClusterNode::UNKNOWN, 0},
                        hostShards.second,
                    };
                }
            );

            auto leaderLoc = Cluster_.NodeByFqdn(assignments.Leader);
            if (!leaderLoc) {
                MON_ERROR(ShardUpdater, "Leader is on an unknown node: " << assignments.Leader);
            }

            Send(Receiver_, new TEvLeader{std::move(leaderLoc)});
            RespondAndDie(TLoadShardMapResult::FromValue(std::move(result)));
        }

        template <typename... TArgs>
        void RespondAndDie(TArgs&&... args) {
            Send(Receiver_, new TEvLoadShardMapResponse{std::forward<TArgs>(args)...});
            PassAway();
        }

    private:
        IProcessingClient* Client_;
        const TActorId Receiver_;
        const IClusterMap& Cluster_;
    };

    class TShardMapBuilderIngestor: public TActorBootstrapped<TShardMapBuilderIngestor> {
    public:
        TShardMapBuilderIngestor(IIngestorClusterClientPtr clients, IClusterMapPtr cluster)
            : Clients_{std::move(clients)}
            , ClusterInfo_{std::move(cluster)}
        {}

        void Bootstrap() {
            Become(&TThis::StateWork);

            // TODO(ivanzhukov): take value from a config
            auto updateInterval = TDuration::Seconds(1);
            ClusterRequester_ = Register(CreateIngestorClusterRequester(ClusterInfo_, Clients_, updateInterval));
        }

    private:
        STATEFN(StateWork) {
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvLoadShardMap, OnLoadShardMap);
                hFunc(TIngestorRequesterEvents::TAssignmentsResponse, OnResponse);
            }
        }

        void OnLoadShardMap(const TEvLoadShardMap::TPtr& ev) {
            const auto maxAge = ev->Get()->CachePolicy.MaxAge;

            if (maxAge != TDuration::Zero()) {
                if (auto entry = Cache_.Get(maxAge)) {
                    Send(ev->Sender, new TEvLoadShardMapResponse{
                            TLoadShardMapResult::FromValue(std::move(*entry)),
                    });

                    return;
                }
            }

            ResponseReceivers_.emplace_back(ev->Sender);

            if (!IsWaitingForResponse_) {
                Send(ClusterRequester_, new TIngestorRequesterEvents::TRequestAssignments{});
                IsWaitingForResponse_ = true;
            }
        }

        void OnResponse(const TIngestorRequesterEvents::TAssignmentsResponse::TPtr& evPtr) {
            IsWaitingForResponse_ = false;
            auto assignments = std::move(evPtr->Get()->Assignments);

            std::function<std::unique_ptr<TEvLoadShardMapResponse>()> eventConstructor;
            if (assignments.empty()) {
                eventConstructor = []() {
                    return std::make_unique<TEvLoadShardMapResponse>(
                            TLoadShardMapResult::FromError("empty assignments"));
                };
            } else {
                eventConstructor = [&assignments]() {
                    return std::make_unique<TEvLoadShardMapResponse>(
                            TLoadShardMapResult::FromValue(assignments));
                };

                Cache_.Write(assignments, TInstant::Now());
            }

            for (auto receiver: ResponseReceivers_) {
                Send(receiver, eventConstructor().release());
            }

            ResponseReceivers_.clear();
            ResponseReceivers_.shrink_to_fit();
        }

    private:
        IIngestorClusterClientPtr Clients_;
        IClusterMapPtr ClusterInfo_;

        TActorId ClusterRequester_;
        TCache Cache_;
        TVector<TActorId> ResponseReceivers_;
        bool IsWaitingForResponse_;
    };

    class TShardMapBuilderRequestingLeader: public TActor<TShardMapBuilderRequestingLeader> {
    public:
        TShardMapBuilderRequestingLeader(IProcessingClusterClientPtr clients, IClusterMapPtr cluster)
            : TActor<TShardMapBuilderRequestingLeader>{&TThis::StateWork}
            , Clients_{std::move(clients)}
            , ClusterInfo_{std::move(cluster)}
            , CurrentLeader_{ClusterInfo_->Local()}
        {}

        STATEFN(StateWork) {
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvLoadShardMap, OnLoadShardMap);
                hFunc(TEvLoadShardMapResponse, OnResponse);
                hFunc(TRequestAssignmentsFromServiceLeader::TEvLeader, OnLeaderInfo);
                hFunc(TEvents::TEvPoison, OnPoison);
            }
        }

    private:
        void OnLoadShardMap(const TEvLoadShardMap::TPtr& ev) {
            const auto maxAge = ev->Get()->CachePolicy.MaxAge;

            if (maxAge != TDuration::Zero()) {
                if (auto entry = Cache_.Get(maxAge)) {
                    Send(ev->Sender, new TEvLoadShardMapResponse{
                        TLoadShardMapResult::FromValue(std::move(*entry))
                    });

                    return;
                }
            }

            auto* client = GetProcessingClient();
            const auto reqId = Register(new TRequestAssignmentsFromServiceLeader{client, *ClusterInfo_, SelfId()});

            ResponseReceivers_.emplace(reqId, ev->Sender);
        }

        IProcessingClient* GetProcessingClient() const {
            if (auto* client = Clients_->GetClient(CurrentLeader_).Get()) {
                return client;
            }

            if (auto* client = GetLocalClient()) {
                MON_WARN(ShardUpdater, "Metric proccessor leader is on an unknown node, falling back to local");
                return client;
            }

            MON_WARN(ShardUpdater, "Local node has no associated metric processor client");
            return Clients_->GetAnyClient().Get();
        }

        IProcessingClient* GetLocalClient() const {
            auto loc = ClusterInfo_->Local();
            return Clients_->GetClient(loc).Get();
        }

        void OnResponse(const TEvLoadShardMapResponse::TPtr& ev) {
            auto& result = ev->Get()->Result;
            if (result.Success()) {
                Cache_.Write(result.Value(), TInstant::Now());
            }

            const auto reqId = ev->Sender;
            auto it = ResponseReceivers_.find(reqId);

            if (it != ResponseReceivers_.end()) {
                // Forward doesn't play well with GrabEdgeEvent
                Send(it->second, new TEvLoadShardMapResponse{
                    std::move(ev->Get()->Result)
                });

                ResponseReceivers_.erase(it);
            } else {
                Y_VERIFY_DEBUG(false);
            }

            if (IsDying_ && ResponseReceivers_.empty()) {
                Send(ReplyTo_, new TEvents::TEvActorDied);
                PassAway();
            }
        }

        void OnLeaderInfo(const TRequestAssignmentsFromServiceLeader::TEvLeader::TPtr& ev) {
            const auto& leader = ev->Get()->Leader;
            if (!leader) {
                auto local = ClusterInfo_->Local();
                // first time fallback to a local node. If we get an error once again, try a random node
                CurrentLeader_ = (CurrentLeader_ == local) ? ClusterInfo_->Any() : local;
                MON_ERROR(ShardUpdater, "Metric processor leader is on an unknown node, fallback to " << CurrentLeader_);
            } else if (*leader != CurrentLeader_ && Clients_->GetClient(*leader)) {
                MON_WARN(ShardUpdater, "Metric processor leader changed from " << CurrentLeader_ << " to " << *leader);
                CurrentLeader_ = *leader;
            }
        }

        void OnPoison(const TEvents::TEvPoison::TPtr& evPtr) {
            if (ResponseReceivers_.empty()) {
                Send(evPtr->Sender, new TEvents::TEvActorDied);
                PassAway();
                return;
            }

            ReplyTo_ = evPtr->Sender;
            IsDying_ = true;
        }

    private:
        IProcessingClusterClientPtr Clients_;
        THashMap<TActorId, TActorId> ResponseReceivers_;
        TCache Cache_;
        IClusterMapPtr ClusterInfo_;
        TClusterNode CurrentLeader_;
        TActorId ReplyTo_;
        bool IsDying_{false};
    };

} // namespace

    // using a new api
    NActors::IActor* CreateShardMapBuilderRequestingCluster(IIngestorClusterClientPtr ingestorClients, IClusterMapPtr cluster) {
        return new TShardMapBuilderIngestor{std::move(ingestorClients), std::move(cluster)};
    }

    // using an old api
    NActors::IActor* CreateShardMapBuilderRequestingLeader(IProcessingClusterClientPtr clusterClient, IClusterMapPtr cluster) {
        return new TShardMapBuilderRequestingLeader{std::move(clusterClient), std::move(cluster)};
    }

    NActors::TActorId MakeShardMapBuilderId() {
        static constexpr TStringBuf ID = "ShardMapSrv\0"sv;
        return TActorId(0, ID);
    }

} // namespace NSolomon::NFetcher
