#include <solomon/services/fetcher/lib/config_updater/config_holder.h>

#include <solomon/services/fetcher/testlib/db.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NDb::NModel;
using namespace NSolomon::NTesting;
using namespace testing;

namespace NSolomon::NFetcher {
    std::ostream& operator<<(std::ostream& os, const TFetcherShard& shard) {
        TStringBuilder sb;
        sb << shard.ToString();

        os << sb;
        return os;
    }

    bool operator==(const TFetcherShard& lhs, const TFetcherShard& rhs) {
        return lhs.IsEqual(rhs);
    }
}

template <typename TConfig, typename TId>
TVector<TConfig> FromDiff(const TVector<std::pair<TId, TConfig>>& diffItems) {
    TVector<TConfig> result;
    for (auto& [_, val]: diffItems) {
        result.emplace_back(val);
    }

    return result;
}

const auto DEFAULT_LOCATION = TClusterNode{"localhost", 0, 0};

struct TFakeLocator: IShardLocator {
    TMaybeLocation Location(ui32) const override {
        return DEFAULT_LOCATION;
    }
};

TFakeLocator ShardLocator;

class TConfigHolderTest: public ::testing::Test {
public:
    void SetUp() override {
        auto locator = MakeHolder<TFakeLocator>();
        Configs_.Reset(new TConfigHolder{ShardLocator});
    }

protected:
    TConfigs MakeInitialConfigs() {
        auto service = ConfigProvider_.NewService();
        auto cluster = ConfigProvider_.NewCluster();
        auto shard = ConfigProvider_.NewShard(service, cluster);

        return {
            .Services = {service},
            .Clusters = {cluster},
            .Shards = {shard},
        };
    }

    TVector<TFetcherShard> MakeFetcherShards(const TConfigs& configs) {
        TVector<TFetcherShard> result;

        for (auto&& shard: configs.Shards) {
            auto serviceIt = FindIf(configs.Services, [&shard] (auto&& service) {
                return service.Id == shard.ServiceId;
            });

            auto clusterIt = FindIf(configs.Clusters, [&shard] (auto&& cluster) {
                return cluster.Id == shard.ClusterId;
            });

            Y_VERIFY(serviceIt != configs.Services.end());
            Y_VERIFY(clusterIt != configs.Clusters.end());

            result.push_back(CreateSimpleShard(
                MakeAtomicShared<TShardConfig>(shard),
                MakeAtomicShared<TClusterConfig>(*clusterIt),
                MakeAtomicShared<TServiceConfig>(*serviceIt),
                DEFAULT_LOCATION
            ));
        }

        return result;
    }

protected:
    TConfigProvider ConfigProvider_;
    THolder<TConfigHolder> Configs_;
};

TEST_F(TConfigHolderTest, EmptyShardEnumeration) {
    auto shards = Configs_->AllShards();
    ASSERT_THAT(shards, IsEmpty());
}

TEST_F(TConfigHolderTest, AllShardsAfterAdd) {
    auto configs = MakeInitialConfigs();
    Configs_->UpdateConfig(configs);

    auto fetcherShards = Configs_->AllShards();
    ASSERT_THAT(fetcherShards, UnorderedElementsAreArray(MakeFetcherShards(configs)));
}

TEST_F(TConfigHolderTest, AllShardsAfterRemove) {
    auto configs = MakeInitialConfigs();
    Configs_->UpdateConfig(configs);

    TConfigs emptyConfigs;
    Configs_->UpdateConfig(emptyConfigs);
    auto fetcherShards = Configs_->AllShards();

    ASSERT_THAT(fetcherShards, IsEmpty());
}

TEST_F(TConfigHolderTest, AllShardsAfterServiceChange) {
    auto configs = MakeInitialConfigs();
    configs.Services[0].ShardSettings.PullOrPush = TPullSettings{.Port = 100};
    Configs_->UpdateConfig(configs);

    configs.Services[0].ShardSettings.PullOrPush = TPullSettings{.Port = 200};
    Configs_->UpdateConfig(configs);

    auto fetcherShards = Configs_->AllShards();
    ASSERT_THAT(fetcherShards, UnorderedElementsAreArray(MakeFetcherShards(configs)));
}

TEST_F(TConfigHolderTest, AllShardsAfterShardChange) {
    auto configs = MakeInitialConfigs();
    configs.Shards[0].State = EShardState::Active;
    Configs_->UpdateConfig(configs);

    configs.Shards[0].State = EShardState::Inactive;
    Configs_->UpdateConfig(configs);

    auto fetcherShards = Configs_->AllShards();
    ASSERT_THAT(fetcherShards, UnorderedElementsAreArray(MakeFetcherShards(configs)));
}

TEST_F(TConfigHolderTest, EmptyShardUpdate) {
    TVector<TServiceConfig> expected{MakeServiceConfig()};

    TConfigs configs{
            .Services = expected,
            .Clusters = {},
            .Shards = {},
    };

    auto diff = Configs_->UpdateConfig(configs);

    ASSERT_THAT(FromDiff(diff.ServiceDiff.Added), UnorderedElementsAreArray(expected));
    ASSERT_THAT(diff.ServiceDiff.Changed, IsEmpty());
    ASSERT_THAT(diff.ServiceDiff.Removed, IsEmpty());

    ASSERT_TRUE(diff.ShardDiff.IsEmpty());
    ASSERT_TRUE(diff.ClusterDiff.IsEmpty());

    ASSERT_THAT(Configs_->AllShards(), IsEmpty());
}

TEST_F(TConfigHolderTest, ShardUpdate) {
    auto configs = MakeInitialConfigs();
    auto diff = Configs_->UpdateConfig(configs);

    ASSERT_THAT(FromDiff(diff.ServiceDiff.Added), UnorderedElementsAreArray(configs.Services));
    ASSERT_THAT(diff.ServiceDiff.Changed, IsEmpty());
    ASSERT_THAT(diff.ServiceDiff.Removed, IsEmpty());

    ASSERT_THAT(FromDiff(diff.ShardDiff.Added), UnorderedElementsAreArray(configs.Shards));
    ASSERT_THAT(diff.ShardDiff.Changed, IsEmpty());
    ASSERT_THAT(diff.ShardDiff.Removed, IsEmpty());

    ASSERT_THAT(FromDiff(diff.ClusterDiff.Added), UnorderedElementsAreArray(configs.Clusters));
    ASSERT_THAT(diff.ClusterDiff.Changed, IsEmpty());
    ASSERT_THAT(diff.ClusterDiff.Removed, IsEmpty());
}

TEST_F(TConfigHolderTest, ShardsAreFiltered) {
    auto locator = MakeHolder<TFakeLocator>();
    Configs_.Reset(new TConfigHolder{ShardLocator, [] (auto&& shard) {
        return IsWriteable(shard);
    }});

    {
        auto configs = MakeInitialConfigs();
        configs.Shards[0].State = EShardState::Inactive;
        auto diff = Configs_->UpdateConfig(configs);
        ASSERT_TRUE(diff.ShardDiff.IsEmpty());
    }

    {
        auto configs = MakeInitialConfigs();
        configs.Shards[0].State = EShardState::Active;
        auto diff = Configs_->UpdateConfig(configs);
        ASSERT_THAT(diff.ShardDiff.Added, SizeIs(1));
    }
}

TEST_F(TConfigHolderTest, ShardsAreCreatedFromAgentConfs) {
    auto locator = MakeHolder<TFakeLocator>();
    auto defaultAgent = MakeAgentConfig();
    auto defaultProvider = MakeProviderConfig();

    auto a1 = defaultAgent;
    auto a2 = defaultAgent;
    auto a3 = defaultAgent;
    auto a4 = defaultAgent;
    a1.Provider = "provider1";
    a2.Provider = "provider2";
    a3.Provider = "provider2";
    a4.Provider = "provider3";

    auto p1 = defaultProvider;
    auto p2 = defaultProvider;
    p1.Id = "provider1";
    p2.Id = "provider2";

    auto configs = TConfigs{
            .Agents = {
                    a1, a2, a3, a4
            },
            .Providers = {
                    p1, p2
            }
    };

    auto diff = Configs_->UpdateConfig(configs);
    Y_UNUSED(diff);
    auto shards = Configs_->AllShards();

    ASSERT_THAT(shards, SizeIs(2));
    ASSERT_THAT(shards, Each(Property(&TFetcherShard::IsAddTsArgs, Eq(false))));
    ASSERT_THAT(shards, Each(Property(&TFetcherShard::IsUseFqdn, Eq(true))));
    ASSERT_THAT(shards, Each(Property(&TFetcherShard::IsPull, Eq(true))));
    ASSERT_THAT(shards, Each(Property(&TFetcherShard::Type, Eq(EFetcherShardType::Agent))));
    ASSERT_THAT(shards, Each(Property(&TFetcherShard::UrlPath, Eq("/storage/readAll"))));
}

TEST_F(TConfigHolderTest, AgentShardsAreUpdated) {
    auto locator = MakeHolder<TFakeLocator>();
    auto defaultAgent = MakeAgentConfig();
    auto defaultProvider = MakeProviderConfig();

    {
        auto first = defaultAgent;
        first.Provider = "provider1";
        auto second = defaultAgent;
        second.Provider = "provider2";
        auto third = defaultAgent;
        third.Provider = "provider2";

        auto p1 = defaultProvider;
        p1.Id = "provider1";
        auto p2 = defaultProvider;
        p2.Id = "provider2";
        auto p3 = defaultProvider;
        p3.Id = "provider3";

        auto configs = TConfigs{
                .Agents = {
                        first, second, third
                },
                .Providers = {
                        p1, p2, p3
                }
        };

        auto diff = Configs_->UpdateConfig(configs);
        Y_UNUSED(diff);
        ASSERT_THAT(Configs_->AllShards(), SizeIs(2));
    }
    {
        auto first = defaultAgent;
        first.Provider = "provider1";
        auto p1 = defaultProvider;
        p1.Id = "provider1";

        auto configs = TConfigs{
                .Agents = {
                        first
                },
                .Providers = {
                        p1
                }
        };

        auto diff = Configs_->UpdateConfig(configs);
        Y_UNUSED(diff);
        ASSERT_THAT(Configs_->AllShards(), SizeIs(1));
    }
}

TEST_F(TConfigHolderTest, AgentShardIsRemovedAfterProviderDeletion) {
    auto defaultAgent = MakeAgentConfig();
    auto defaultProvider = MakeProviderConfig();

    auto agent1 = defaultAgent;
    agent1.Hostname = "hostname1";
    agent1.Provider = "provider1";

    auto agent2 = defaultAgent;
    agent2.Hostname = "hostname2";
    agent2.Provider = "provider2";

    auto provider1 = defaultProvider;
    provider1.Id = "provider1";

    auto provider2 = defaultProvider;
    provider2.Id = "provider2";

    auto configs = TConfigs{
            .Agents = {
                    agent1, agent2
            },
            .Providers = {
                    provider1, provider2
            }
    };

    auto diff = Configs_->UpdateConfig(configs);
    Y_UNUSED(diff);
    ASSERT_THAT(Configs_->AllShards(), SizeIs(2));

    auto newConfigs = TConfigs{
            .Agents = {
                    agent1, agent2
            },
            .Providers = {
                    provider1
            }
    };

    diff = Configs_->UpdateConfig(newConfigs);
    Y_UNUSED(diff);
    ASSERT_THAT(Configs_->AllShards(), SizeIs(1));
    ASSERT_THAT(Configs_->AllShards()[0].ProjectId(), Eq(provider1.Id));
}

TEST_F(TConfigHolderTest, AgentShardIsChangedAfterAddition) {
    auto defaultAgent = MakeAgentConfig();
    auto defaultProvider = MakeProviderConfig();
    {
        auto a1 = defaultAgent;
        auto a2 = defaultAgent;
        a1.Hostname = "hostname1";
        a1.Provider = "provider1";
        a2.Hostname = "hostname2";
        a2.Provider = "provider2";

        auto p1 = defaultProvider;
        auto p2 = defaultProvider;
        p1.Id = "provider1";
        p2.Id = "provider2";

        auto configs = TConfigs{
                .Agents = {
                        a1
                },
                .Providers = {
                        p1, p2
                }
        };

        auto diff = Configs_->UpdateConfig(configs);
        ASSERT_THAT(diff.IsEmpty(), false);

        configs = TConfigs{
                .Agents = {
                        a1, a2
                },
                .Providers = {
                        p1, p2
                }
        };

        diff = Configs_->UpdateConfig(configs);
        ASSERT_THAT(diff.IsEmpty(), false);
    }
}

TEST_F(TConfigHolderTest, GetShardByNumId) {
    auto configs = MakeInitialConfigs();
    Configs_->UpdateConfig(configs);
    auto shards = MakeFetcherShards(configs);
    auto expected = shards[0];
    ASSERT_THAT(*Configs_->GetShard(configs.Shards[0].NumId), Eq(expected));
}

TEST_F(TConfigHolderTest, ShardServiceChange) {
    auto configs = MakeInitialConfigs();
    auto diff = Configs_->UpdateConfig(configs);
    ASSERT_THAT(FromDiff(diff.ShardDiff.Added), SizeIs(1));

    auto&& newService = ConfigProvider_.NewService();
    configs.Services.push_back(newService);
    configs.Shards[0].ServiceId = newService.Id;
    configs.Shards[0].ServiceName = newService.Name;

    diff = Configs_->UpdateConfig(configs);
    ASSERT_THAT(FromDiff(diff.ShardDiff.Changed), SizeIs(1));
    auto shards = Configs_->AllShards();
    ASSERT_THAT(shards, SizeIs(1));
    auto shard = shards[0];
    ASSERT_THAT(shard.ServiceName(), StrEq(newService.Name));
}

TEST_F(TConfigHolderTest, ShardClusterChange) {
    auto configs = MakeInitialConfigs();
    auto diff = Configs_->UpdateConfig(configs);
    ASSERT_THAT(FromDiff(diff.ShardDiff.Added), SizeIs(1));

    auto&& newCluster = ConfigProvider_.NewCluster();
    configs.Clusters = {newCluster};
    configs.Shards[0].ClusterId = newCluster.Id;
    configs.Shards[0].ClusterName = newCluster.Name;

    diff = Configs_->UpdateConfig(configs);
    ASSERT_THAT(FromDiff(diff.ShardDiff.Changed), SizeIs(1));
    auto shards = Configs_->AllShards();
    ASSERT_THAT(shards, SizeIs(1));
    auto shard = shards[0];
    ASSERT_THAT(shard.Cluster()->Name(), StrEq(newCluster.Name));
}
